<?php

declare(strict_types=1);

namespace Internetgalerie\Flipbook\ViewHelpers;

use TYPO3\CMS\Core\Resource\File;
use TYPO3\CMS\Core\Resource\FileReference;
use TYPO3\CMS\Core\Utility\PathUtility;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper;
use TYPO3\CMS\Core\Exception\SiteNotFoundException;
use TYPO3\CMS\Core\Page\AssetCollector;

final class PdfToLightboxViewHelper extends AbstractTagBasedViewHelper
{
    protected $tagName = 'a';
    protected $cssClass = 'ig-lightbox-book-container';
    protected $libs = [
        'EXT:flipbook/Resources/Public/JavaScripts/pdf_js/lib/pdf.min.mjs',
        'EXT:flipbook/Resources/Public/JavaScripts/pdf_js/lib/pdf.worker.min.mjs',
        'EXT:flipbook/Resources/Public/JavaScripts/pdf_js/PdfToLightBoxConverter.js'
    ];

    public function __construct(
        private readonly AssetCollector $assetCollector,
    ) {
        parent::__construct();
    }

    public function initializeArguments(): void
    {
        parent::initializeArguments();

        $this->registerArgument(
            'file',
            'mixed',
            'The FileReference object, File object, or a string path to the file.',
            true
        );
    }
    public function render(): string
    {
        $fileInput = $this->arguments['file'];
        $fileUrl = $this->resolveFileUrl($fileInput);

        $this->assetCollector->addJavaScript('PdfToLightBox', 'EXT:flipbook/Resources/Public/JavaScripts/pdf_js/PdfToLightBox.js');
        $this->assetCollector->addStyleSheet('PdfToLightBox', 'EXT:flipbook/Resources/Public/Css/PdfToLightBox.css');

        $this->tag->addAttribute('href', $fileUrl);
        $this->tag->addAttribute('target', '_blank');
        $this->tag->addAttribute('data-libs', $this->getResolvedFilePaths($this->libs));
        $this->tag->addAttribute('class', $this->cssClass);
        $this->tag->setContent($this->renderChildren());

        return $this->tag->render();
    }

    protected function resolveFileUrl(mixed $fileInput): string
    {
        if ($fileInput instanceof FileReference || $fileInput instanceof File) {
            return $fileInput->getPublicUrl();
        }
        if (is_string($fileInput)) {
            try {
                return PathUtility::getAbsoluteWebPath($fileInput);
            } catch (SiteNotFoundException $e) {
                return $fileInput;
            }
        }
        throw new \Exception(
            'The "file" argument must be a FileReference, File object, or a string path.',
            1678893922
        );
    }

    private function getResolvedFilePaths(array $pathArray): string
    {
        $resolvedModulePaths = [];
        foreach ($pathArray as $lib) {
            $resolvedModulePaths[] = PathUtility::getPublicResourceWebPath($lib);
        }
        return implode(',', $resolvedModulePaths);
    }
}
