class PdfToLightBoxConverter {
    constructor({ container, source }) {
        this.triggerElement = container;
        this.pdfSourceUrl = source;
        this.pdfDocument = null;
        this.totalPages = 0;
        this.uniqueId = '0';
        this.isReady = false;
        this.lightboxInstance = null;
        this.renderFormat = 'image/jpeg';
        this.renderQuality = 1;
        this.renderScale = 3;
        this.useCanvas = true;
        this.initializationPromise = this.initializeProcess();
        this.lightboxContainer = null;
        this.slides = null;
        this.thumbnails = null;
        this.pagesCache = [];
        this.loadingOffset = Math.min(20, Math.floor((window.innerWidth / 2) / 90));
    }
    async initializeProcess() {
        try {
            await this.loadPdfDocument();
            // Setup the lightbox component
            this.prepareLightbox();
            // Render first page
            await this.renderPages(1, 1);
            // open lightbox
            this.open();
            this.renderPages();
        } catch (error) {
            this.handleInitializationError(error);
        }
    }

    /**
     * Fetches the PDF data and initializes the PDF document object.
     */
    async loadPdfDocument() {
        const fetchResponse = await fetch(this.pdfSourceUrl);
        if (!fetchResponse.ok) {
            throw new Error(`Failed to fetch PDF: ${fetchResponse.statusText}`);
        }

        const arrayBuffer = await fetchResponse.arrayBuffer();
        const fileArray = new Uint8Array(arrayBuffer);

        // Use a clearer check for pdfjsLib availability
        if (typeof pdfjsLib === 'undefined' || !pdfjsLib.getDocument) {
            throw new Error('PDF viewer library (pdfjsLib) is not available after loading modules.');
        }

        this.pdfDocument = await pdfjsLib.getDocument({
            data: fileArray,
            useSystemFonts: true,
        }).promise;
        this.uniqueId = this.createTimeBasedId();
        this.totalPages = this.pdfDocument.numPages;

        console.log(`PDF Loaded: ${this.totalPages} pages.`);
    }

    /**
     * Creates placeholder elements and initializes the lightbox instance.
     */
    prepareLightbox() {
        if (!IgLightboxConfig) {
            throw new Error('Global configuration "IgLightboxConfig" is missing.');
        }

        this.pagePlaceholders = Array.from({ length: this.totalPages }, () => {
            const link = document.createElement('a');
            link.setAttribute('href', 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7');
            return link;
        });
        this.pagePlaceholders.forEach((page) => {
            this.pagesCache.push({
                element: page,
                loaded: false
            });
        });
        const bookConfig = {
            ...IgLightboxConfig,
            ...{
                id: this.uniqueId,
                images: this.pagePlaceholders,
                downloadTarget: this.pdfSourceUrl,
                download: true,
                book: true
            }
        };
        this.lightboxInstance = new IgLightbox(bookConfig);
        this.lightboxInstance.open();
        this.lightboxContainer = document.querySelector(`.ig-lightbox-container[data-id="${this.uniqueId}"]`);
        this.slides = this.lightboxContainer.querySelector('.ig-lightbox-slides');
        this.thumbnails = this.lightboxContainer.querySelector('.ig-lightbox-thumbnails');
        this.lightboxContainer.addEventListener('lightbox:goTo', (event) => {
            let currentSlideIndex = event.detail.current.index + 1;
            this.renderPages(currentSlideIndex, this.loadingOffset);
        });
    }

    /**
     * Renders all PDF pages concurrently.
     */
    async renderPages(centerPage = 1, offset = this.loadingOffset) {
        let page = centerPage - offset;
        if (page < 1) {
            page += this.totalPages;
        }
        const renderPromises = [];
        for (let i = 0; i <= offset * 2; i++) {
            if (this.pagesCache[page - 1] && !this.pagesCache[page - 1].loaded) {
                renderPromises.push(this.renderPage(page));
            }
            page++;
            if (page > this.totalPages) {
                page -= this.totalPages;
            }
        }
        await Promise.allSettled(renderPromises.map((promise, index) =>
            promise.catch(error => console.error(`Failed to render page ${index + 1}:`, error))
        ));
    }

    /**
     * Renders a single PDF page into a Canvas or Data URL.
     * @param {number} pageNumber - The 1-based index of the page.
     * @returns {Promise<HTMLElement | string>} The Canvas element or Data URL.
     */
    async renderPage(pageNumber) {
        const page = await this.pdfDocument.getPage(pageNumber);
        this.pagesCache[pageNumber - 1].loaded = true;
        const viewport = page.getViewport({ scale: this.renderScale });
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');

        canvas.width = viewport.width;
        canvas.height = viewport.height;

        const renderTask = page.render({ canvasContext: ctx, viewport: viewport });
        await renderTask.promise;

        this.updateLightboxDom(pageNumber, canvas);
        page.cleanup();
        return this.useCanvas ? canvas : canvas.toDataURL(this.renderFormat, this.renderQuality);
    }

    /**
     * Finds the corresponding slide/thumbnail elements and replaces the placeholder.
     * @param {number} pageNumber - The 1-based index of the page.
     * @param {HTMLCanvasElement} canvas - The rendered canvas element.
     */
    updateLightboxDom(pageNumber, canvas) {
        const slideIndex = pageNumber - 1;

        if (!this.lightboxContainer) {
            console.warn(`Lightbox container with ID ${this.uniqueId} not found for page ${pageNumber}.`);
            return;
        }
        const slideContainer = this.slides?.querySelector(`[data-index="${slideIndex}"]`);
        const thumbnailContainer = this.thumbnails?.querySelector(`[data-index="${slideIndex}"]`);

        if (slideContainer) {
            const slideImg = slideContainer.firstElementChild;
            if (this.useCanvas) {
                // Replace placeholder link/img with the Canvas element
                slideContainer.replaceChild(canvas, slideImg);
                if (pageNumber > 1) {
                    const shadow = document.createElement('span');
                    shadow.style.aspectRatio = canvas.width / canvas.height;
                    slideContainer.appendChild(shadow);
                }
            } else {
                // Update src of the placeholder img (assuming it's an image element)
                slideImg.src = canvas.toDataURL(this.renderFormat, this.renderQuality);
            }
        }

        if (thumbnailContainer) {
            const thumbnailImg = thumbnailContainer.firstElementChild;
            // Generate a lower quality Data URL for the thumbnail regardless of useCanvas setting
            const thumbnailDataUrl = canvas.toDataURL(this.renderFormat, 0.2);
            thumbnailImg.src = thumbnailDataUrl;
        }
    }

    /**
     * Opens the initialized lightbox.
     */
    open() {
        this.lightboxInstance?.open();
    }

    /**
     * Handles and displays errors during the initialization process.
     * @param {Error} error - The error object.
     */
    handleInitializationError(error) {
        const errorMessage = `Error loading PDF: ${error.message}`;
        this.triggerElement.innerHTML = `<p class="text-red-600">${errorMessage}</p>`;
        console.error('Initialization Failed:', error);
    }

    /**
     * Creates a unique ID based on current time.
     * @returns {string} The unique ID.
     */
    createTimeBasedId() {
        const timestamp = Date.now().toString(36);
        const randomPart = Math.random().toString(36).substring(2, 7);
        return `${timestamp}-${randomPart}`;
    }
}
window.PdfToLightBoxConverter = PdfToLightBoxConverter;
