<?php

declare(strict_types=1);

namespace Internetgalerie\IgAcl\Utility;

use Internetgalerie\IgAcl\Domain\Model\EntityWithAcl;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;

class AclUtility implements SingletonInterface
{
    public const ROLE_ADMIN = 'admin';
    protected array $settings = [];
    protected ConfigurationManager $configurationManager;
    protected array $aclContexts = [];
    private static $frontendUserGroupIds = null;

    public function __construct(
        private readonly Context $context,
    ) {
    }
    
    public function injectConfigurationManager(ConfigurationManager $configurationManager): void
    {
        $this->configurationManager = $configurationManager;
        $this->settings = $this->configurationManager->getConfiguration(
            ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        )['plugin.']['tx_igacl.']['settings.'];
        $aclContexts = $this->settings['context.'] ?? [];
        $this->aclContexts = $this->convertTypoScriptToAclContext($aclContexts);
    }

    
    public function getFrontendUserId(): ?int
    {
        return $this->context->getPropertyFromAspect('frontend.user', 'id');
    }

    public function getFrontendUserGroupIds()
    {
        if (static::$frontendUserGroupIds === null) {
            static::$frontendUserGroupIds = [];
            foreach ($this->context->getPropertyFromAspect('frontend.user', 'groupIds') as $groupId) {
                if ($groupId > 0) {
                    static::$frontendUserGroupIds[] = $groupId;
                }
            }
        }
        return static::$frontendUserGroupIds;
    }

    /**
     * has the current frontend user in given $context the given role $role (checked with group uids)
     */
    public function hasRole(string $role, string $context = ''): bool
    {
        $frontendUserGroupIds = $this->getFrontendUserGroupIds();
        $contextPath = explode('.', $context);
        $currentContext = $this->aclContexts;
        //var_dump($currentContext, $contextPath , $role);exit(0);

        // check global context for role
        if ($this->inRole($role, $currentContext, $frontendUserGroupIds)) {
            return true;
        }

        // loop over context path
        foreach ($contextPath as $ctxPathName) {
            if (isset($currentContext[$ctxPathName])) {
                $currentContext = $currentContext[$ctxPathName];
            } else {
                return false;
            }
            if ($this->inRole($role, $currentContext, $frontendUserGroupIds)) {
                return true;
            }
        }
        return false;
    }
    /**
     * has the current frontend user admin role
     */
    public function hasRoleAdmin(string $context = '')
    {
        return $this->hasRole(self::ROLE_ADMIN, $context);
    }

    /**
     * is current frontend user owner of the object
     */
    //public function isOwner(EntityWithAcl|array $object = null)
    public function isOwner($object = null): bool
    {
        // grant access to new objects
        if ($object === null) {
            return true;
        }
        if ($object instanceof EntityWithAcl) {
            if ($object->getAclOwner() == $this->getFrontendUserId()) {
                return true;
            }
        } elseif (is_array($object)) {
            if (isset($object['aclOwner']) && $object['aclOwner'] == $this->getFrontendUserId()) {
                return true;
            }
        }
        return false;
    }

    // hasPermissionWrite
    public function hasPermissionWrite($object = null, string $context = ''): bool
    {
        if ($this->hasRoleAdmin($context)) {
            return true;
        }
        if ($this->isOwner($object)) {
            return true;
        }
        
        /*
          $frontendUserGroupIds = $this->getFrontendUserGroupIds();
          foreach ($object->getAclWriteGroups() as $group) {
          if (in_array($group->getUid(), $frontendUserGroupIds)) {
          return true;
          }
          }
        */
        return false;
    }
    /**
     * has the current frontend user group ids in given role
     */
    protected function inRole(string $role, array $aclContext, array $frontendUserGroupIds): bool
    {
        $roleGroups = $aclContext['role'][$role] ?? [];
        if (!empty($roleGroups)) {
            $commonGroups = array_intersect($frontendUserGroupIds, $roleGroups);
            return !empty($commonGroups);
        }
        return false;
    }
    private function convertRoles(array $typoScriptArray): array
    {
        $aclRoles = [];
        foreach ($typoScriptArray as $role => $uids) {
            $aclRoles[$role] = GeneralUtility::intExplode(',', $uids, true);
        }
        return $aclRoles;
    }

    private function convertTypoScriptToAclContext(array $typoScriptArray): array
    {
        $aclContext = [];
        foreach ($typoScriptArray as $key => $value) {
            if (is_array($value)) {
                $key = rtrim($key, '.');
                if ($key === 'role') {
                    $aclContext['role'] = [];
                    $aclContext['role'] = $this->convertRoles($value);
                } else {
                    $aclContext[$key] = $this->convertTypoScriptToAclContext($value);
                }
            } else {
                die('s');
            }
        }
        return $aclContext;
    }
}
