<?php

declare(strict_types=1);

namespace Internetgalerie\IgAcl\Utility;

use Internetgalerie\IgAcl\Domain\Model\AclOwnerInterface;
use Internetgalerie\IgAcl\Domain\Model\AclGroupInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use Internetgalerie\IgsCrm\Domain\Repository\FrontendUserRepository;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Persistence\ObjectStorage;

class AclUtility implements SingletonInterface
{
    public const ROLE_ADMIN = 'admin';
    protected array $settings = [];
    protected ConfigurationManager $configurationManager;
    protected array $aclContexts = [];
    protected ?FrontendUser $frontendUser = null;
    protected static $frontendUserGroupIds = null;

    public function __construct(
        protected readonly Context $context,
        protected readonly FrontendUserRepository $frontendUserRepository,
    ) {
    }
    
    public function injectConfigurationManager(ConfigurationManager $configurationManager): void
    {
        $this->configurationManager = $configurationManager;
        $this->settings = $this->configurationManager->getConfiguration(
            ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        )['plugin.']['tx_igacl.']['settings.'];
        $aclContexts = $this->settings['context.'] ?? [];
        $this->aclContexts = $this->convertTypoScriptToAclContext($aclContexts);
    }

    public function getFrontendUserId(): ?int
    {
        return $this->context->getPropertyFromAspect('frontend.user', 'id');
    }

    public function getFrontendUser()
    {
        if ($this->frontendUser === null) {
            $this->frontendUserRepository->setRespectStoragePage(false);
            $frontendUserId = $this->getFrontendUserId();
            if ($frontendUserId > 0) {
                $this->frontendUser = $this->frontendUserRepository->findOneBy([
                    'uid' => $frontendUserId,
                ]);
            }
        }
        return $this->frontendUser;
    }

    public function getFrontendUserGroupIds()
    {
        if (static::$frontendUserGroupIds === null) {
            static::$frontendUserGroupIds = [];
            foreach ($this->context->getPropertyFromAspect('frontend.user', 'groupIds') as $groupId) {
                if ($groupId > 0) {
                    static::$frontendUserGroupIds[] = $groupId;
                }
            }
        }
        return static::$frontendUserGroupIds;
    }

    public function getFrontendUserGroupNames()
    {
        return $this->context->getPropertyFromAspect('frontend.user', 'groupNames');
    }

    public function hasGroupName(string $groupName): bool
    {
        return in_array($groupName, $this->getFrontendUserGroupNames());
    }

    /**
     * check if user has all given group names
     */
    public function hasAllGroupNames(array $groupNames): bool
    {
        $frontendUserGroupNames = $this->getFrontendUserGroupNames();
        // check if user has all given groupNames
        if (!empty($groupNames)) {
            foreach ($groupNames as $groupName) {
                if (!in_array($groupName, $frontendUserGroupNames)) {
                    return false;
                }
            }
        }
        return true;
    }
    
    /**
     * has the current frontend user in given $roleContext the given role $role (checked with group uids)
     */
    public function hasRole(string $role, string $roleContext = ''): bool
    {
        $frontendUserGroupIds = $this->getFrontendUserGroupIds();
        $roleContextPath = explode('.', $roleContext);
        $currentRoleContext = $this->aclContexts;
        //var_dump($currentRoleContext, $roleContextPath , $role);exit(0);

        // check global context for role
        if ($this->inRole($role, $currentRoleContext, $frontendUserGroupIds)) {
            return true;
        }

        // loop over context path
        foreach ($roleContextPath as $ctxPathName) {
            if (isset($currentRoleContext[$ctxPathName])) {
                $currentRoleContext = $currentRoleContext[$ctxPathName];
            } else {
                return false;
            }
            if ($this->inRole($role, $currentRoleContext, $frontendUserGroupIds)) {
                return true;
            }
        }
        return false;
    }

    /**
     * has the current frontend user admin role
     */
    public function hasRoleAdmin(string $roleContext = '')
    {
        return $this->hasRole(self::ROLE_ADMIN, $roleContext);
    }

    /**
     * has the current frontend user group ids in given role
     */
    protected function inRole(string $role, array $aclContext, array $frontendUserGroupIds): bool
    {
        $roleGroups = $aclContext['role'][$role] ?? [];
        if (!empty($roleGroups)) {
            $commonGroups = array_intersect($frontendUserGroupIds, $roleGroups);
            return !empty($commonGroups);
        }
        return false;
    }

    private function convertRoles(array $typoScriptArray): array
    {
        $aclRoles = [];
        foreach ($typoScriptArray as $role => $uids) {
            $aclRoles[$role] = GeneralUtility::intExplode(',', $uids, true);
        }
        return $aclRoles;
    }

    /**
     * is current frontend user owner of the object
     */
    public function isOwner($object = null): bool
    {
        // grant access to new objects
        if ($object === null) {
            return true;
        }
        if ($object instanceof AclOwnerInterface) {
            if ($object->getAclOwner() == $this->getFrontendUserId()) {
                return true;
            }
        } elseif (is_array($object)) {
            if (isset($object['aclOwner']) && $object['aclOwner'] == $this->getFrontendUserId()) {
                return true;
            }
        }
        return false;
    }

    public function objectStorageHasGroup($object = null, ObjectStorage $aclGroups): bool
    {
        $frontendUserGroupIds = $this->getFrontendUserGroupIds();
        foreach ($aclGroups as $group) {
            if (in_array($group->getUid(), $frontendUserGroupIds)) {
                return true;
            }
        }
        return false;
    }

    public function stringListHasGroup($object = null, string $aclGroupsList): bool
    {
        //$aclGroupArray = GeneralUtility::trimExplode(',', $aclGroupsList, true);
        $aclGroupArray = GeneralUtility::intExplode(',', $aclGroupsList, true);

        $frontendUserGroupIds = $this->getFrontendUserGroupIds();
        foreach ($aclGroupArray as $groupUid) {
            if (in_array($groupUid, $frontendUserGroupIds)) {
                return true;
            }
        }
        return false;
    }

    /**
     * is current frontend user a group in the aclReadGroups of the object
     */
    public function hasReadGroup($object = null): bool
    {
        // grant access to new objects
        if ($object === null) {
            return true;
        }
        if ($object instanceof AclGroupInterface) {
            if ($this->objectStorageHasGroup($object->getAclReadGroups())) {
                return true;
            }
        } elseif (is_array($object) && is_string($object['aclReadGroups'] ?? null)) {
            if ($this->stringListHasGroup($object['aclReadGroups'])) {
                return true;
            }
        }
        return false;
    }

    /**
     * is current frontend user a group in the aclWriteGroups of the object
     */
    public function hasWriteGroup($object = null): bool
    {
        // grant access to new objects
        if ($object === null) {
            return true;
        }
        if ($object instanceof AclGroupInterface) {
            if ($this->objectStorageHasGroup($object->getAclWriteGroups())) {
                return true;
            }
        } elseif (is_array($object) && is_string($object['aclWriteGroups'] ?? null)) {
            if ($this->stringListHasGroup($object['aclWriteGroups'])) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * has the current frontend user read permission to the object with the given context and is in all requiredGroups
     */
    public function hasPermissionRead($object = null, string $roleContext = '', array $requiredGroups = []): bool
    {
        // check if user has all requiredGroups
        if (!$this->hasAllGroupNames($requiredGroups)) {
            return false;
        }
        // admin has permission
        if ($this->hasRoleAdmin($roleContext)) {
            return true;
        }
        // owner has permission
        if ($this->isOwner($object)) {
            return true;
        }
        // permission from groups
        if ($this->hasReadGroup($object)) {
            return true;
        }
        // special object permissions
        if ($object instanceof AclObjectInterface) {
            return $object->frontendUserHasPermissionRead($this->getFrontendUserId());
        }
        return false;
    }

    /**
     * has the current frontend user write permission to the object with the given context and is in all requiredGroups
     */
    public function hasPermissionWrite($object = null, string $roleContext = '', array $requiredGroups = []): bool
    {
        // check if user has all requiredGroups
        if (!$this->hasAllGroupNames($requiredGroups)) {
            return false;
        }
        // admin has permission
        if ($this->hasRoleAdmin($roleContext)) {
            return true;
        }
        // owner has permission or if object is new (null)
        if ($this->isOwner($object)) {
            return true;
        }
        // permission from groups
        if ($this->hasWriteGroup($object)) {
            return true;
        }
        // special object permissions
        if ($object instanceof AclObjectInterface) {
            return $object->frontendUserHasPermissionWrite($this->getFrontendUserId());
        }
        return false;
    }

    private function convertTypoScriptToAclContext(array $typoScriptArray): array
    {
        $aclContext = [];
        foreach ($typoScriptArray as $key => $value) {
            if (is_array($value)) {
                $key = rtrim($key, '.');
                if ($key === 'role') {
                    $aclContext['role'] = [];
                    $aclContext['role'] = $this->convertRoles($value);
                } else {
                    $aclContext[$key] = $this->convertTypoScriptToAclContext($value);
                }
            } else {
                die('AclUtility: convertTypoScriptToAclContext value is no array');
            }
        }
        return $aclContext;
    }

}
