<?php

declare(strict_types=1);

namespace Internetgalerie\IgApiLogin\Dispatcher;

use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\RequestHandlerInterface;
use SourceBroker\T3api\Service\SerializerService;
use SourceBroker\T3api\Dispatcher\AbstractDispatcher;
use TYPO3\CMS\Core\Authentication\LoginType;
use TYPO3\CMS\Core\Authentication\Event\AfterUserLoggedInEvent;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Http\Response;
use TYPO3\CMS\Core\RateLimiter\RateLimiterFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;

use TYPO3\CMS\Frontend\Authentication\FrontendUserAuthentication;
use TYPO3\CMS\Frontend\Middleware\FrontendUserAuthenticator;

use SourceBroker\T3api\Configuration\Configuration;
use SourceBroker\T3api\Processor\ProcessorInterface;
use Symfony\Bridge\PsrHttpMessage\Factory\HttpFoundationFactory;
use Symfony\Component\HttpFoundation\Request;

use TYPO3\CMS\Core\Session\UserSessionManager;
use TYPO3\CMS\Core\Http\CookieScopeTrait;
use Internetgalerie\IgApiLogin\Middleware\IgApiRequestResolver;

class Bootstrap extends FrontendUserAuthenticator
{
    use CookieScopeTrait;
    protected ?int $storagePid = null;
    protected HttpFoundationFactory $httpFoundationFactory;
    
    public function __construct(
        protected readonly Context $context,
        protected readonly RateLimiterFactory $rateLimiterFactory,
        protected readonly EventDispatcherInterface $eventDispatcher,
        protected readonly SerializerService $serializerService,
    ) {
        $this->httpFoundationFactory = GeneralUtility::makeInstance(HttpFoundationFactory::class);
    }
     
    /**
     * @throws \Throwable
     */
    public function processLogin(ServerRequestInterface $request, RequestHandlerInterface $handler): ResponseInterface
    {
        $this->response = new Response('php://temp', 200, ['Content-Type' => 'application/ld+json']);
        //$request = $this->httpFoundationFactory->createRequest($inputRequest);
        $parsedBody = $request->getParsedBody();
        $username = $request->getQueryParams()['username'] ?? $parsedBody['username'] ?? null;
        $password = $request->getQueryParams()['password'] ?? $parsedBody['password'] ?? null;
        $action = $action;
        $success = false;

        /*
        $username = 'da';
            $password = 'da';
        */

        // set data for typo3 login
        $parsedBody['user'] = $username;
        $parsedBody['pass'] = $password;
        $parsedBody['logintype'] = LoginType::LOGIN->value ?? 'login';
        $request = $request->withParsedBody($parsedBody);
        $symfonyRequest = $this->httpFoundationFactory->createRequest($request);
        $this->callProcessors($symfonyRequest, $this->response);

        $frontendUser = GeneralUtility::makeInstance(FrontendUserAuthentication::class);
        $GLOBALS['TYPO3_CONF_VARS']['SVCONF']['auth']['setup']['FE' . '_fetchUserIfNoSession'] = true;
        //$frontendUser->checkPid = false;
        $frontendUser->checkPid_value = $this->storagePid;

        // Rate Limiting
        $rateLimiter = $this->ensureLoginRateLimit($frontendUser, $request);


        $frontendUser->start($request);
        $frontendUser->fetchGroupData($request);
        $this->context->setAspect('frontend.user', $frontendUser->createUserAspect());
        $request = $request->withAttribute('frontend.user', $frontendUser);

        if ($this->context->getAspect('frontend.user')->isLoggedIn() && $rateLimiter) {
            $rateLimiter->reset();
            $this->eventDispatcher->dispatch(new AfterUserLoggedInEvent($frontendUser, $request));
        }

       // Store session data for fe_users if it still exists
        if ($frontendUser instanceof FrontendUserAuthentication) {
            $success = ($frontendUser->user !== null);
            $frontendUser->storeSessionData();
            //var_dump($frontendUser->id);
            $this->response = $frontendUser->appendCookieToResponse($this->response, $request->getAttribute('normalizedParams'));
            // Collect garbage in Frontend requests, which aren't fully cacheable (e.g. with cookies)
            if ($this->response->hasHeader('Set-Cookie')) {
                $this->sessionGarbageCollection();
            }
        }
        if (!$success) {
            $this->response = $this->response->withStatus(401, 'Unauthorized');
        }
        $session = $frontendUser->getSession();
        $scope = $this->getCookieScope($request->getAttribute('normalizedParams'));
        $data = [
            'success' => $success,
            'session' => $session->getJwt($scope),
            //'sessionCookie' => $this->response->getHeader('Set-Cookie'),
            'mode' => 'cookie',
            //'input' => $parsedBody,
            'user' => [
                'uid' => $frontendUser->user['uid']  ?? null,
                'username' => $frontendUser->user['username'] ?? null
            ],
        ];
        $output = $this->serializerService->serialize($data);
        $this->response->getBody()->write($output);
        return $this->response;
    }

    /**
     * @throws \Throwable
     */
    public function processLogout(ServerRequestInterface $request, RequestHandlerInterface $handler): ResponseInterface
    {
        $this->response = new Response('php://temp', 200, ['Content-Type' => 'application/ld+json']);
        $data = ['success' => true];
        $parsedBody = $request->getParsedBody();
        $parsedBody['logintype'] = LoginType::LOGOUT->value ?? 'logout';
        $request = $request->withParsedBody($parsedBody);
        $symfonyRequest = $this->httpFoundationFactory->createRequest($request);
        $this->callProcessors($symfonyRequest, $this->response);

        $frontendUser = GeneralUtility::makeInstance(FrontendUserAuthentication::class);
        $frontendUser->start($request);

        $output = $this->serializerService->serialize($data);
        $this->response->getBody()->write($output);
        return $this->response;
    }

    protected function callProcessors(Request $request, ResponseInterface &$response): void
    {
        array_filter(
            Configuration::getProcessors(),
            static function (string $processorClass) use ($request, &$response) {
                if (!is_subclass_of($processorClass, ProcessorInterface::class, true)) {
                    throw new \RuntimeException(
                        sprintf(
                            'Process `%s` needs to be an instance of `%s`',
                            $processorClass,
                            ProcessorInterface::class
                        ),
                        1603705384
                    );
                }

                /** @var ProcessorInterface $processor */
                $processor = GeneralUtility::makeInstance($processorClass);

                $processor->process($request, $response);
            }
        );
    }
    
}