<?php

declare(strict_types=1);

namespace Internetgalerie\IgApiLogin\EventListener;

use Internetgalerie\IgApiLogin\Middleware\IgApiRequestResolver;
use SourceBroker\T3api\Event\AfterProcessPathsEvent;

use GoldSpecDigital\ObjectOrientedOAS\Objects\Header;
use GoldSpecDigital\ObjectOrientedOAS\Objects\Operation;
use GoldSpecDigital\ObjectOrientedOAS\Objects\Parameter;
use GoldSpecDigital\ObjectOrientedOAS\Objects\PathItem;
use GoldSpecDigital\ObjectOrientedOAS\Objects\Response;
use GoldSpecDigital\ObjectOrientedOAS\Objects\Schema;
use GoldSpecDigital\ObjectOrientedOAS\Objects\MediaType;


final class AfterProcessPathsEventListener
{

    protected static function getLoginParameters(): array
    {
        return [
            Parameter::create()
                ->name('username')
                ->in(Parameter::IN_QUERY)
                ->description('username for login')
                ->required(true)
                ->schema(Schema::string()),
            Parameter::create()
                ->name('password')
                ->in(Parameter::IN_QUERY)
                ->description('password for login')
                ->required(true)
                ->schema(Schema::string()),
        ];
    }

    public function __invoke(AfterProcessPathsEvent $event): void
    {
        // frontend user login
        $event->addPath(
            '/' . IgApiRequestResolver::PARAMETER_LOGIN,
            [
                'path' => PathItem::create()->route('/' . IgApiRequestResolver::PARAMETER_LOGIN),
                'operations' => [
                    Operation::create()
                        ->tags('Authentication')
                        ->action(constant(Operation::class . '::ACTION_' . 'POST'))
                        ->summary('frontend user login with username and password')
                        ->parameters(...self::getLoginParameters())
                        ->responses(Response::create()
                                    ->statusCode(200)
                                    ->description('Successful login')
                                    ->headers(
                                        Header::create('Set-Cookie')
                                            ->description('Session cookie for frontend user authentication')
                                            ->schema(
                                                Schema::create()
                                                    ->type(Schema::TYPE_STRING)
                                                    ->example('fe_typo_user=abcde12345; Path=/; HttpOnly; Secure')
                                            )
                                    )
                                    ->content(
                                        MediaType::json()->schema(
                                            Schema::create()
                                            ->type(Schema::TYPE_OBJECT)
                                                ->properties(
                                                    Schema::create('success')->type(Schema::TYPE_BOOLEAN)->example('bool'),
                                                    Schema::create('session')->type(Schema::TYPE_STRING)->example('string'),
                                                    Schema::create('mode')->type(Schema::TYPE_STRING)->example('cookie'),
                                                    Schema::create('user')
                                                        ->type(Schema::TYPE_OBJECT)
                                                        ->properties(
                                                            Schema::create('uid')->type(Schema::TYPE_INTEGER)->example('int'),
                                                            Schema::create('username')->type(Schema::TYPE_STRING)->example('string')
                                                        )
                                                ),
                                        ),
                                    ),
                                    Response::create()
                                    ->statusCode(401)
                                    ->description('Unauthorized - e.g. Invalid credentials')
                        )
                        ->requestBody(null)
                ],
            ]
        );

        // frontend user logout
        $event->addPath(
            '/' . IgApiRequestResolver::PARAMETER_LOGOUT,
            [
                'path' => PathItem::create()->route('/' . IgApiRequestResolver::PARAMETER_LOGOUT),
                'operations' => [
                    Operation::create()
                        ->tags('Authentication')
                        ->action(constant(Operation::class . '::ACTION_' . 'GET'))
                        ->summary('logout with current frontend user')
                        ->parameters()
                        ->responses(Response::create()
                                    ->description('Successful operation')
                        )
                        ->requestBody(null)
                ],
            ]
        );

    }
}
