<?php

declare(strict_types=1);

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace Internetgalerie\IgBackendHelpers\Controller\Category;

use RuntimeException;
use Internetgalerie\IgBackendHelpers\Utility\BackendFrameUtility;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use TYPO3\CMS\Backend\Configuration\BackendUserConfiguration;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use Internetgalerie\IgBackendHelpers\Tree\Repository\CategoryTreeRepository;
use TYPO3\CMS\Backend\Utility\BackendUtility;
use TYPO3\CMS\Core\Authentication\BackendUserAuthentication;
use TYPO3\CMS\Core\Exception\Page\RootLineException;
use TYPO3\CMS\Core\Exception\SiteNotFoundException;
use TYPO3\CMS\Core\Http\JsonResponse;
use TYPO3\CMS\Core\Imaging\Icon;
use TYPO3\CMS\Core\Imaging\IconFactory;
use TYPO3\CMS\Core\Localization\LanguageService;
use TYPO3\CMS\Core\Log\LogManager;
use TYPO3\CMS\Core\Site\SiteFinder;
use TYPO3\CMS\Core\Type\Bitmask\Permission;
use TYPO3\CMS\Core\Utility\ExtensionManagementUtility;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\RootlineUtility;
use TYPO3\CMS\Core\Utility\VersionNumberUtility;
use TYPO3\CMS\Core\Versioning\VersionState;
use TYPO3\CMS\Workspaces\Service\WorkspaceService;


/**
 * Controller providing data to the page tree
 * @internal This class is a specific Backend controller implementation and is not considered part of the Public TYPO3 API.
 */
class TreeController
{
    /**
     * Option to prefix the page ID when outputting the tree items, set via userTS.
     *
     * @var bool
     */
    protected $addIdAsPrefix = false;

    /**
     * Option to add the rootline path above each mount point, set via userTS.
     *
     * @var bool
     */
    protected $showMountPathAboveMounts = false;

    /**
     * An array of background colors for a branch in the tree, set via userTS.
     *
     * @var array
     */
    protected $backgroundColors = [];

    /**
     * A list of pages not to be shown.
     *
     * @var array
     */
    protected $hiddenRecords = [];

    /**
     * Contains the state of all items that are expanded.
     *
     * @var array
     */
    protected $expandedState = [];

    /**
     * Instance of the icon factory, to be used for generating the items.
     *
     * @var IconFactory
     */
    protected $iconFactory;

    /**
     * Number of tree levels which should be returned on the first page tree load
     *
     * @var int
     */
    protected $levelsToFetch = 99;

    /**
     * When set to true all nodes returend by API will be expanded
     * @var bool
     */
    protected $expandAllNodes = false;

    /**
     * Used in the record link picker to limit the page tree only to a specific list
     * of alternative entry points for selecting only from a list of pages
     */
    protected array $alternativeEntryPoints = [];

    protected UriBuilder $uriBuilder;

    /**
     * Mount Pages
     * @var array
     */
    protected $mountPointPages = [];

    /**
     * @var BackendFrameUtility $backendFrameUtility
     */
    protected $backendFrameUtility = null;

    protected $extensionName = '';

    protected $tableName = '';

    protected $titleField = 'name';

    protected $parentField = 'parent';
    
    protected $type = 'category';

    /**
     * Constructor to set up common objects needed in various places.
     */
    public function __construct()
    {
        if($this->extensionName===null ) {
            $className = static::class;
            $classNameParts = explode('\\', $className, 4);
            // Skip vendor and product name for core classes
            if (str_starts_with($className, 'TYPO3\\CMS\\')) {
                $this->extensionName = $classNameParts[2];
            } else {
                $this->extensionName = $classNameParts[1];
            }
        }
        $this->iconFactory = GeneralUtility::makeInstance(IconFactory::class);
        $this->uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        $this->backendFrameUtility = GeneralUtility::makeInstance(BackendFrameUtility::class);
    }

    protected function initializeConfiguration(ServerRequestInterface $request)
    {
        if($this->extensionName) {
            $this->mountPointPages = $this->backendFrameUtility->getSysFolderByModule( $this->extensionName );
        }

        $userTsConfig = $this->getBackendUser()->getTSConfig();
        $this->hiddenRecords = GeneralUtility::intExplode(
            ',',
            $userTsConfig['options.']['hideRecords.'][$this->tableName] ?? '',
            true
        );
        $this->backgroundColors = $userTsConfig['options.']['categoryTree.']['backgroundColor.'] ?? [];
        $this->addIdAsPrefix = (bool)($userTsConfig['options.']['categoryTree.']['showPageIdWithTitle'] ?? false);
        $this->showMountPathAboveMounts = (bool)($userTsConfig['options.']['categoryTree.']['showPathAboveMounts'] ?? false);
        $backendUserConfiguration = GeneralUtility::makeInstance(BackendUserConfiguration::class);
        $backendUserCategoryTreeState = $backendUserConfiguration->get('BackendComponents.States.Categorytree.' . $this->extensionName);

        //var_dump($backendUserCategoryTreeState);die();
        if (is_object($backendUserCategoryTreeState) && is_object($backendUserCategoryTreeState->stateHash)) {
            $this->expandedState = (array)$backendUserCategoryTreeState->stateHash;
        } else {
            $stateHash = $backendUserCategoryTreeState['stateHash'] ?? [];
            $this->expandedState = is_array($stateHash) ? $stateHash : [];
        }
    }

    /**
     * Returns page tree configuration in JSON
     *
     * @return ResponseInterface
     */
    public function fetchConfigurationAction(): ResponseInterface
    {
        $extensionNameLowerCase = strtolower($this->extensionName);

        $jsConfirmationClass = '';
        if (VersionNumberUtility::convertVersionNumberToInteger(VersionNumberUtility::getNumericTypo3Version()) >= 13000000) {
            $jsConfirmationClass = \TYPO3\CMS\Core\Authentication\JsConfirmation::class;
        } else {
            $jsConfirmationClass = \TYPO3\CMS\Core\Type\Bitmask\JsConfirmation::class;
        }

        $configuration = [
            'extensionName' => $this->extensionName,
            'allowDragMove' => false,
            'doktypes' => $this->getDokTypes(),
            'displayDeleteConfirmation' => $this->getBackendUser()->jsConfirmation($jsConfirmationClass::DELETE),
            'showIcons' => true,
            'dataUrl' => (string)$this->uriBuilder->buildUriFromRoute('ajax_' . $extensionNameLowerCase . '_' . $this->type . '_tree_data'),
            'filterUrl' => (string)$this->uriBuilder->buildUriFromRoute('ajax_' . $extensionNameLowerCase .'_' . $this->type . '_tree_filter'),
            'itemType' => $this->tableName,
            'actions' => ['itemType' => $this->tableName,],
        ];
        return new JsonResponse($configuration);
    }

    public function fetchReadOnlyConfigurationAction(ServerRequestInterface $request): ResponseInterface
    {
        $entryPoints = $request->getQueryParams()['alternativeEntryPoints'] ?? '';
        $entryPoints = GeneralUtility::intExplode(',', $entryPoints, true);
        $additionalArguments = [
            'readOnly' => 1,
        ];
        if (!empty($entryPoints)) {
            $additionalArguments['alternativeEntryPoints'] = $entryPoints;
        }
        $extensionNameLowerCase = strtolower($this->extensionName);
        $jsConfirmationClass = '';
        if (VersionNumberUtility::convertVersionNumberToInteger(VersionNumberUtility::getNumericTypo3Version()) >= 13000000) {
            $jsConfirmationClass = \TYPO3\CMS\Core\Authentication\JsConfirmation::class;
        } else {
            $jsConfirmationClass = \TYPO3\CMS\Core\Type\Bitmask\JsConfirmation::class;
        }
        $configuration = [
            'extensionName' => $this->extensionName,
            'displayDeleteConfirmation' => $this->getBackendUser()->jsConfirmation($jsConfirmationClass::DELETE),
            'showIcons' => true,
            'dataUrl' => (string)$this->uriBuilder->buildUriFromRoute('ajax_' . $extensionNameLowerCase .'_' . $this->type . '_tree_data', $additionalArguments),
            'filterUrl' => (string)$this->uriBuilder->buildUriFromRoute('ajax_' . $extensionNameLowerCase . '_' . $this->type . '_tree_filter', $additionalArguments),
            'itemType' => $this->tableName,
        ];
        return new JsonResponse($configuration);
    }

    /**
     * Returns the list of doktypes to display in page tree toolbar drag area
     *
     * Note: The list can be filtered by the user TypoScript
     * option "options.categoryTree.doktypesToShowInNewPageDragArea".
     *
     * @return array
     */
    protected function getDokTypes(): array
    {
        return [];
    }

    /**
     * Returns JSON representing page tree
     *
     * @param ServerRequestInterface $request
     * @return ResponseInterface
     */
    public function fetchDataAction(ServerRequestInterface $request): ResponseInterface
    {
        $backendUserConfiguration = GeneralUtility::makeInstance(BackendUserConfiguration::class);
        $backendUserCategoryTreeState = $backendUserConfiguration->get('BackendComponents.States.Categorytree.' . $this->extensionName);
        //var_dump($request);die();
        //var_dump($request->getQueryParams());die();
        $this->initializeConfiguration($request);
        /*$context = GeneralUtility::makeInstance(Context::class);
        DebuggerUtility::var_dump(Environment::getContext());die();*/

        $items = [];

        $siblingsPosition = 0;
        $parentParam = $request->getQueryParams()['parent'] ?? null;
        // Handle mount point identifiers (e.g., 'm12' -> 0)
        if (is_string($parentParam) && str_starts_with($parentParam, 'm')) {
            $parent = 0; // Mount points have category uid=0
        } else {
            $parent = (int)$parentParam;
        }
        $pid = (int)($request->getQueryParams()['pid'] ?? null);
        // Wenn parent nicht gesetzt, hole rootkategorien by mountpointPage
        // Sonst hole unterkategorien by parent in pid
        if(!$parent) {
            if (empty($this->mountPointPages)) {
                GeneralUtility::makeInstance(LogManager::class)->getLogger(self::class)->warning('Error no system folder found for extention "{extensionName}". Needed for table "{table}. Select module in sys folder or define mountPointPages in TreeController.php"', [
                    'extensionName' => $this->extensionName,
                    'table' => $this->tableName,
                ]);
            }
            foreach($this->mountPointPages as $mountPointPage) {
                $item = $this->getCategoriesByMountPoint($mountPointPage);
                $items = array_merge($items, $this->categoriesToFlatArray($item, $mountPointPage['uid']));
            }
        } else {
            $mountPoint = [];
            foreach($this->mountPointPages as $mountPointPage) {
                if($pid == $mountPointPage['uid']) {
                    $mountPoint = $mountPointPage;
                }
            }
            $item = $this->getCategoriesByMountPoint($mountPoint, $parent);
            $parentDepth = 0;
            $items = array_merge($items, $this->categoriesToFlatArray($item, $pid, $parent, $parentDepth));
        }

        return new JsonResponse($items);
    }

    /**
     * Converts nested tree structure produced by CategoryTreeRepository to a flat, one level array
     * and also adds visual representation information to the data.
     *
     * @param array $page
     * @param int $entryPoint
     * @param int $parent
     * @param int $depth
     * @param array $inheritedData
     * @return array
     */
    protected function categoriesToFlatArray(array $category, int $entryPoint, int $parent = 0, int $depth = 0, array $inheritedData = []): array
    {
        $backendUser = $this->getBackendUser();
        $categoryId = (int)$category['uid'];
        if (in_array($categoryId, $this->hiddenRecords, true)) {
            return [];
        }
        if ($entryPoint === 0 && !$backendUser->isAdmin()) {
            return [];
        }

        $stopCategoryTree = !empty($category['php_tree_stop']) && $depth > 0;
        $identifier = $entryPoint . '_' . $categoryId;
        $expanded = !empty($category['expanded'])
            || (isset($this->expandedState[$identifier]) && $this->expandedState[$identifier])
            || $this->expandAllNodes;

        $backgroundColor = !empty($this->backgroundColors[$categoryId]) ? $this->backgroundColors[$categoryId] : ($inheritedData['backgroundColor'] ?? '');

        $suffix = '';
        $prefix = '';
        $nameSourceField = $this->titleField;
        $visibleText = $category[$this->titleField];
        if($category['uid'] > 0 && isset($GLOBALS['TCA'][$this->tableName]['ctrl']['label'])) {
            $nameSourceField = $GLOBALS['TCA'][$this->tableName]['ctrl']['label'];
            $visibleText = $category[$nameSourceField];

            if(isset($GLOBALS['TCA'][$this->tableName]['ctrl']['label_alt']) && isset($GLOBALS['TCA'][$this->tableName]['ctrl']['label_alt_force'])) {
                if($GLOBALS['TCA'][$this->tableName]['ctrl']['label_alt_force'] === true) {
                    $labelAlts = GeneralUtility::trimExplode(',', $GLOBALS['TCA'][$this->tableName]['ctrl']['label_alt']);
                    foreach($labelAlts as $labelAlt) {
                        if($category[$labelAlt]) {
                            $visibleText .= ', ' . $category[$labelAlt];
                        }
                    }
                }
            }
        }

        $tooltip = $visibleText;

        if($category['uid'] == 0 && $GLOBALS['TCA']['pages']['ctrl']['typeicon_classes']['contains-' . lcfirst($this->extensionName)] ?? false) {
            $icon = $this->iconFactory->getIcon($GLOBALS['TCA']['pages']['ctrl']['typeicon_classes']['contains-' . lcfirst($this->extensionName)], Icon::SIZE_SMALL);
        } else if($category['uid'] == 0) {
            $icon = $this->iconFactory->getIconForRecord('pages', $category, Icon::SIZE_SMALL);
        } else {
            $icon = $this->iconFactory->getIconForRecord($this->tableName, $category, Icon::SIZE_SMALL);
        }

        if (is_string($visibleText) && trim($visibleText) === '') {
            $visibleText = htmlspecialchars('[' . $this->getLanguageService()->sL('LLL:EXT:core/Resources/Private/Language/locallang_core.xlf:labels.no_title') . ']');
        }

        $lockInfo = BackendUtility::isRecordLocked($this->tableName, $categoryId);
        if (is_array($lockInfo)) {
            $tooltip .= ' - ' . $lockInfo['msg'];
        }
        if ($this->addIdAsPrefix) {
            $prefix = htmlspecialchars('[' . $categoryId . '] ');
        }

        $items = [];
        // For mount points (uid=0), use the entryPoint as identifier to avoid duplicates
        $uniqueIdentifier = $categoryId === 0 ? 'm' . $entryPoint : (string)$categoryId;

        $item = [
            // Used to track if the tree item is collapsed or not
            'stateIdentifier' => $identifier,
            // identifier is not only used for pages, therefore it's a string
            'identifier' => $uniqueIdentifier,
            'pid' => $entryPoint,
            'depth' => $depth,
            'tip' => is_string($tooltip) ? htmlspecialchars($tooltip) : '',
            'icon' => $icon->getIdentifier(),
            'name' => $visibleText,
            'labels' => [$visibleText],
            'statusInformation' => [],
            'type' => (int)($page['doktype'] ?? 0),
            'nameSourceField' => $nameSourceField,
            //'workspaceId' => !empty($page['t3ver_oid']) ? $page['t3ver_oid'] : $pageId,
            'siblingsCount' => $page['siblingsCount'] ?? 1,
            'siblingsPosition' => $page['siblingsPosition'] ?? 1,
            'extensionName' => $this->extensionName,
        ];

        if ((isset($category['_children']) && !empty($category['_children'])) || $this->getCategoryTreeRepository()->hasChildren($categoryId, $entryPoint)) {
            $item['hasChildren'] = true;
            if ($depth >= $this->levelsToFetch) {
                $category = $this->getCategoryTreeRepository()->getTreeLevels($category, 1, $entryPoint);
            }
        }
        if(isset($category['_children']) && is_array($category['_children']) && empty($category['_children'])) {
            $item['hasChildren'] = false;
        }
        if (!empty($prefix)) {
            $item['prefix'] = htmlspecialchars($prefix);
        }
        if (!empty($suffix)) {
            $item['suffix'] = htmlspecialchars($suffix);
        }
        if (is_array($lockInfo)) {
            $item['locked'] = true;
        }
        if ($icon->getOverlayIcon()) {
            $item['overlayIcon'] = $icon->getOverlayIcon()->getIdentifier();
        }
        if ($expanded && isset($category['_children']) && is_array($category['_children']) && !empty($category['_children'])) {
            $item['expanded'] = $expanded;
        }
        if ($backgroundColor) {
            $item['backgroundColor'] = htmlspecialchars($backgroundColor);
        }
        if ($stopCategoryTree) {
            $item['stopCategoryTree'] = $stopCategoryTree;
        }
        $class = $this->resolvePageCssClassNames($category);
        if (!empty($class)) {
            $item['class'] = $class;
        }
        /*if ($depth === 0) {
            $item['isMountPoint'] = true;

            if ($this->showMountPathAboveMounts) {
                $item['readableRootline'] = $this->getMountPointPath($categoryId);
            }
        }*/

        $items[] = $item;
        if (!$stopCategoryTree && isset($category['_children']) && is_array($category['_children']) && !empty($category['_children']) && ($depth < $this->levelsToFetch || $expanded)) {
            $siblingsCount = count($category['_children']);
            $siblingsPosition = 0;
            $items[key($items)]['loaded'] = true;
            foreach ($category['_children'] as $child) {
                $child['siblingsCount'] = $siblingsCount;
                $child['siblingsPosition'] = ++$siblingsPosition;
                $items = array_merge($items, $this->categoriesToFlatArray($child, $entryPoint, (int)$item['identifier'], $depth + 1, ['backgroundColor' => $backgroundColor]));
            }
        }
        return $items;
    }

    public function getCategoriesByMountPoint($mountPoint, $parent = 0, $searchQuery = '')
    {
        $categoryRepository = $this->getCategoryTreeRepository();
        if($searchQuery !== '') {
            return $categoryRepository->fetchFilteredTree($searchQuery, $mountPoint, [$mountPoint['uid']]);
        }
        return $categoryRepository->getTree($mountPoint, $parent);
        //$categoryRepository->find
    }

    /**
     * Returns JSON representing page tree filtered by keyword
     *
     * @param ServerRequestInterface $request
     * @return ResponseInterface
     */
    public function filterDataAction(ServerRequestInterface $request): ResponseInterface
    {
        $searchQuery = $request->getQueryParams()['q'] ?? '';
        if (trim($searchQuery) === '') {
            return new JsonResponse([]);
        }

        $this->initializeConfiguration($request);
        $this->expandAllNodes = true;

        $items = [];
        $parentParam = $request->getQueryParams()['parent'] ?? null;
        // Handle mount point identifiers (e.g., 'm12' -> 0)
        if (is_string($parentParam) && str_starts_with($parentParam, 'm')) {
            $parent = 0; // Mount points have category uid=0
        } else {
            $parent = (int)$parentParam;
        }
        $pid = (int)($request->getQueryParams()['pid'] ?? null);
        // Wenn parent nicht gesetzt, hole rootkategorien by mountpointPage
        // Sonst hole unterkategorien by parent in pid
        if (!$parent) {
            if (empty($this->mountPointPages)) {
                GeneralUtility::makeInstance(LogManager::class)->getLogger(self::class)->warning('Error no system folder found for extention "{extensionName}". Needed for table "{table}. Select module in sys folder or define mountPointPages in TreeController.php"', [
                    'extensionName' => $this->extensionName,
                    'table' => $this->tableName,
                ]);
            }
            $seenIdentifiers = [];
            foreach ($this->mountPointPages as $mountPointPage) {
                $item = $this->getCategoriesByMountPoint($mountPointPage, $parent, $searchQuery);
                $flatItems = $this->categoriesToFlatArray($item, $mountPointPage['uid']);
                // Deduplicate based on identifier to avoid showing the same category multiple times
                foreach ($flatItems as $flatItem) {
                    if (!isset($seenIdentifiers[$flatItem['identifier']])) {
                        $seenIdentifiers[$flatItem['identifier']] = true;
                        $items[] = $flatItem;
                    }
                }
            }
        } else {
            $mountPoint = [];
            foreach ($this->mountPointPages as $mountPointPage) {
                if ($pid == $mountPointPage['uid']) {
                    $mountPoint = $mountPointPage;
                }
            }
            $item = $this->getCategoriesByMountPoint($mountPoint, $parent, $searchQuery);
            $parentDepth = 0;
            $items = array_merge($items, $this->categoriesToFlatArray($item, $pid, $parent, $parentDepth));
        }

        return new JsonResponse($items);
    }

    /**
     * Sets a temporary mount point
     *
     * @param ServerRequestInterface $request
     * @return ResponseInterface
     * @throws RuntimeException
     */
    public function setTemporaryMountPointAction(ServerRequestInterface $request): ResponseInterface
    {
        if (empty($request->getParsedBody()['pid'])) {
            throw new RuntimeException(
                'Required "pid" parameter is missing.',
                1511792197
            );
        }
        $pid = (int)$request->getParsedBody()['pid'];

        $this->getBackendUser()->uc['categoryTree_temporaryMountPoint'] = $pid;
        $this->getBackendUser()->writeUC();
        $response = [
            'mountPointPath' => $this->getMountPointPath($pid),
        ];
        return new JsonResponse($response);
    }

    protected function getCategoryTreeRepository(): CategoryTreeRepository
    {
        $backendUser = $this->getBackendUser();
        $userTsConfig = $backendUser->getTSConfig();

        $additionalQueryRestrictions = [];

        return GeneralUtility::makeInstance(
            CategoryTreeRepository::class,
            $this->tableName,
            (int)$backendUser->workspace,
            [],
            $additionalQueryRestrictions,
            //$this->tableName,
            $this->titleField,
            $this->parentField
        );
    }

    /**
     * Returns the mount point path for a temporary mount or the given id
     *
     * @param int $uid
     * @return string
     */
    protected function getMountPointPath(int $uid): string
    {
        if ($uid <= 0) {
            return '';
        }
        $rootline = array_reverse(BackendUtility::BEgetRootLine($uid));
        array_shift($rootline);
        $path = [];
        foreach ($rootline as $rootlineElement) {
            $record = BackendUtility::getRecordWSOL($this->tableName, $rootlineElement['uid'], 'title', '', true, true);
            $text = $record['title'];
            $path[] = htmlspecialchars($text);
        }
        return '/' . implode('/', $path);
    }

    /**
     * Fetches possible css class names to be used when a record was modified in a workspace
     *
     * @param array $page Page record (workspace overlaid)
     * @return string CSS class names to be applied
     */
    protected function resolvePageCssClassNames(array $page): string
    {
        $classes = [];

        if ($page['uid'] === 0) {
            return '';
        }
        $workspaceId = (int)$this->getBackendUser()->workspace;
        if ($workspaceId > 0 && ExtensionManagementUtility::isLoaded('workspaces')) {
            if ((int)$page['t3ver_wsid'] === $workspaceId
                && ((int)$page['t3ver_oid'] > 0 || (int)$page['t3ver_state'] === VersionState::NEW_PLACEHOLDER)
            ) {
                $classes[] = 'ver-element';
                $classes[] = 'ver-versions';
            } elseif (
                $this->getWorkspaceService()->hasPageRecordVersions(
                    $workspaceId,
                    $page['t3ver_oid'] ?: $page['uid']
                )
            ) {
                $classes[] = 'ver-versions';
            }
        }

        return implode(' ', $classes);
    }

    /**
     * Check if drag-move in the svg tree is allowed for the user
     *
     * @return bool
     */
    protected function isDragMoveAllowed(): bool
    {
        $backendUser = $this->getBackendUser();
        return $backendUser->isAdmin()
            || ($backendUser->check('tables_modify', $this->tableName) && $backendUser->checkLanguageAccess(0));
    }

    /**
     * Get allowed mountpoints. Returns temporary mountpoint when temporary mountpoint is used.
     *
     * @return int[]
     */
    protected function getAllowedMountPoints(): array
    {
        $mountPoints = (int)($this->getBackendUser()->uc['categoryTree_temporaryMountPoint'] ?? 0);
        if (!$mountPoints) {
            if (!empty($this->alternativeEntryPoints)) {
                return $this->alternativeEntryPoints;
            }
            $mountPoints = array_map('intval', $this->getBackendUser()->returnWebmounts());
            return array_unique($mountPoints);
        }
        return [$mountPoints];
    }

    /**
     * @return WorkspaceService
     */
    protected function getWorkspaceService(): WorkspaceService
    {
        return GeneralUtility::makeInstance(WorkspaceService::class);
    }

    /**
     * @return BackendUserAuthentication
     */
    protected function getBackendUser(): BackendUserAuthentication
    {
        return $GLOBALS['BE_USER'];
    }

    /**
     * @return LanguageService|null
     */
    protected function getLanguageService(): ?LanguageService
    {
        return $GLOBALS['LANG'] ?? null;
    }
}
