<?php

declare(strict_types=1);

namespace Internetgalerie\IgBackendHelpers\View;


use TYPO3\CMS\Backend\Tree\View\AbstractTreeView;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\QueryHelper;
use TYPO3\CMS\Core\Database\Query\Restriction\DeletedRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\WorkspaceRestriction;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

/**
 * Browse pages in Web module
 *
 * @internal This class is a TYPO3 Backend implementation and is not considered part of the Public TYPO3 API.
 */
class TableListView extends AbstractTreeView
{
    public $fieldArray = null;
    //public $defaultList = 'uid,pid,tstamp,sorting,deleted,parent';
    public $treeName = null; // @todo must be unique -> set from tablename
    //    public $expandAll=true;

    /**
     * Name of the main Module, to store the current ids/url parameter
     */
    public $mainModule = 'web';
    /**
     * @var string|null
     */
    /**
     * Field for ORDER BY. Is set by function init.
     */
    public $orderByFields = null;
    /**
     * PageId
     */
    protected $pageUid = null;
    protected $argumentName = 'entryId'; // maybe default should be empty, name for url query parameter

    /**
     * Calls init functions
     *
     * @param string $clause        Additional clause for selecting pages.
     * @param string $orderByFields record ORDER BY field
     */
    public function __construct(string $clause = '', string $orderByFields = '')
    {
        parent::__construct();
        $this->init($clause, $orderByFields);
    }
    /**
     * Initialize, setting what is necessary for browsing pages.
     * Using the current user.
     *
     * @param string $clause        Additional clause for selecting pages.
     * @param string $orderByFields record ORDER BY field
     */
    public function init($clause = '', $orderByFields = '')
    {
        // Setting BE_USER by default
        $this->BE_USER = $GLOBALS['BE_USER'];
        // Setting clause
        $this->clause = ' AND deleted=0 AND sys_language_uid IN (0,-1) ' .$clause;
        if ($orderByFields) {
            $this->orderByFields = $orderByFields;
        }
        if (! is_array($this->MOUNTS)) {
            // Dummy
            $this->MOUNTS = [0 => 0];
        }
        $this->title = $GLOBALS['TYPO3_CONF_VARS']['SYS']['sitename'];

        // This will hide records from display - it has nothing to do with user rights!!
        //$clauseExcludePidList = '';
        // This is very important for making trees of pages: Filtering out deleted pages, pages with no access to and sorting them correctly:
        //parent::init(' AND deleted=0 AND sys_language_uid IN (0,-1) ' . $clause . $clauseExcludePidList, $orderByFields);
        //$this->treeName = str_replace('_', '', $this->treeName ?: $this->table);

        //$this->MOUNTS = [0];//$backendUser->returnWebmounts();
    }

    public function setMounts(array $rootUids): void
    {
        $this->MOUNTS = $rootUids;
    }
    public function setPageUid($pageUid): void
    {
        $this->pageUid = $pageUid;
    }
    public function setMainModule($mainModule): void
    {
        $this->mainModule = $mainModule;
    }
    public function setArgumentName($argumentName): void
    {
        $this->argumentName = $argumentName;
    }
    public function setTable($table): void
    {
        $this->table = $table;
        if ($this->treeName === null) {
            $this->treeName = str_replace('_', '', $this->table);
        }

        if ($this->orderByFields === null) {
            $this->orderByFields = $GLOBALS['TCA'][$table]['ctrl']['sortby'] ?? '';
        }
        if ($this->fieldArray === null) {
            $this->fieldArray = [
                'uid',
                'pid',
                'hidden',
                'starttime',
                'endtime',
                //'parent',
                $GLOBALS['TCA'][$table]['ctrl']['label'],
            ];
            if ($GLOBALS['TCA'][$table]['ctrl']['label_alt']) {
                $this->fieldArray = array_merge(
                    $this->fieldArray,
                    GeneralUtility::trimExplode(',', $GLOBALS['TCA'][$table]['ctrl']['label_alt'], true)
                );
            }
            if ($GLOBALS['TCA'][$table]['ctrl']['sortby']) {
                $this->fieldArray[] = $GLOBALS['TCA'][$table]['ctrl']['sortby'];
            }
        }
    }
    public function setOrderBy($orderBy): void
    {
        $this->orderByFields = $orderBy;
    }
    public function setWhere($where): void
    {
        $this->clause = $where;
    }
    public function setFieldArray($fieldArray): void
    {
        $this->fieldArray = $fieldArray;
    }
    public function setTreeName($treeName): void
    {
        $this->treeName = $treeName;
    }




   /**
     * Getting the tree data: Selecting/Initializing data pointer to items for a certain parent id.
     * For tables: This will make a database query to select all children to "parent"
     * For arrays: This will return key to the ->dataLookup array
     *
     * @param int $pid page id
     *
     * @return mixed Data handle (Tables: An sql-resource, arrays: A parentId integer. -1 is returned if there were NO subLevel.)
     * @internal
     */
    public function getDataInit($pid)
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($this->table);
        $queryBuilder->getRestrictions()
                ->removeAll()
                ->add(GeneralUtility::makeInstance(DeletedRestriction::class))
                ->add(GeneralUtility::makeInstance(WorkspaceRestriction::class, (int)$this->BE_USER->workspace));
        $queryBuilder
                ->select(...$this->fieldArray)
                ->from($this->table)
                ->where(
                    /*                    $queryBuilder->expr()->eq(
                        'pid',
                        $queryBuilder->createNamedParameter($pid, \PDO::PARAM_INT)
                        ),*/
                    QueryHelper::stripLogicalOperatorPrefix($this->clause)
                );

        foreach (QueryHelper::parseOrderBy($this->orderByFields) as $orderPair) {
            [$fieldName, $order] = $orderPair;
            $queryBuilder->addOrderBy($fieldName, $order);
        }

        return $queryBuilder->execute();
    }


   /**
     * Fetches the data for the tree
     *
     * @param int $uid item id for which to select subitems (parent id)
     * @param int $limit max entries
     * @param string $depthData HTML-code prefix for recursive calls.
     * @return int The count of items on the level
     */
    public function getTree($uid, $limit = 1000, $depthData = '')
    {
        // Buffer for id hierarchy is reset:
        $this->buffer_idH = [];
        // Init vars
        $depth = (int)$limit;
        $limit = (int)$limit;
        $HTML = '';
        $a = 0;
        $res = $this->getDataInit($uid);
        $c = $this->getDataCount($res);
        $idH = [];
        // Traverse the records:
       while ($limit > 0 && ($row = $this->getDataNext($res))) {
            if (!$this->getBackendUser()->isInWebMount($this->table === 'pages' ? $row : $row['pid'])) {
                // Current record is not within web mount => skip it
                continue;
            }

            $a++;
            $limit--;
            $newID = $row['uid'];
            // Reserve space.
            $this->tree[] = [];
            end($this->tree);
            // Get the key for this space
            $treeKey = key($this->tree);
            // If records should be accumulated, do so
            if ($this->setRecs) {
                $this->recs[$row['uid']] = $row;
            }
            // Accumulate the id of the element in the internal arrays
            $this->ids[] = ($idH[$row['uid']]['uid'] = $row['uid']);
            $this->ids_hierarchy[$depth][] = $row['uid'];
            $this->orig_ids_hierarchy[$depth][] = $row['_ORIG_uid'] ?: $row['uid'];

            // Make a recursive call to the next level
            $nextLevelDepthData = $depthData . '<span class="treeline-icon treeline-icon-' . ($a === $c ? 'clear' : 'line') . '"></span>';
            $hasSub = $this->expandNext($newID) && !$row['php_tree_stop'];

            if ($depth > 1 && $hasSub) {
                $nextCount = $this->getTree($newID, $depth - 1, $nextLevelDepthData);
                if (!empty($this->buffer_idH)) {
                    $idH[$row['uid']]['subrow'] = $this->buffer_idH;
                }
                // Set "did expand" flag
                $isOpen = 1;
            } else {
                $nextCount = $this->getCount($newID);
                // Clear "did expand" flag
                $isOpen = 0;
            }
            // Set HTML-icons, if any:
            if ($this->makeHTML) {
                $HTML = $this->PMicon($row, $a, $c, $nextCount, $isOpen) . $this->wrapStop($this->getIcon($row), $row);
            }
            // Finally, add the row/HTML content to the ->tree array in the reserved key.
            $this->tree[$treeKey] = [
                'row' => $row,
                'HTML' => $HTML,
                'invertedDepth' => $depth,
                'depthData' => $depthData,
                'bank' => $this->bank,
                'hasSub' => $nextCount && $hasSub,
                'isFirst' => $a === 1,
                'isLast' => $a === $c,
            ];
        }

        $this->getDataFree($res);
        $this->buffer_idH = $idH;
        return $c;
    }


    /**
     * Will create and return the HTML code for a browsable list
     * Is based on the mounts found in the internal array ->MOUNTS (set in the constructor)
     *
     * @param  int $limit Max entries to show
     *
     * @return string HTML code for the browsable list
     */
    public function getBrowsableList(int $limit = 1000): string
    {

        // Get stored tree structure AND updating it if needed according to incoming PM GET var.
        $this->initializePositionSaving();
        // Init done:
        $treeArr = [];
        // Traverse mounts:
        $firstHtml = '';
        foreach ($this->MOUNTS as $idx => $uid) {
            // Set first:
            $this->bank = $idx;
            $isOpen = $this->stored[$idx][$uid] || $this->expandFirst || $uid === '0' || $uid === 0;

            // Save ids while resetting everything else.
            $curIds = $this->ids;
            $this->reset();
            $this->ids = $curIds;
            // Only, if not for uid 0
            if ($uid) {
                // Set PM icon for root of mount:
                $cmd = $this->bank . '_' . ($isOpen ? '0_' : '1_') . $uid . '_' . $this->treeName;
                $firstHtml = '<a class="list-tree-control list-tree-control-' . ($isOpen ? 'open' : 'closed')
                    . '" href="' . htmlspecialchars($this->getThisScript() . 'PM=' . $cmd) . '"><i class="fa"></i></a>';
            }
            // Preparing rootRec for the mount
            if ($uid) {
                $rootRec = $this->getRecord($uid);
                $firstHtml .= $this->getIcon($rootRec);
            } else {
                // Artificial record for the tree root, id=0
                $rootRec = $this->getRootRecord();
                $firstHtml .= $this->getRootIcon($rootRec);
            }
            if (is_array($rootRec)) {
                // In case it was swapped inside getRecord due to workspaces.
                $uid = $rootRec['uid'];
                // Add the root of the mount to ->tree
                $this->tree[] = ['HTML' => $firstHtml, 'row' => $rootRec, 'bank' => $this->bank, 'hasSub' => true, 'invertedDepth' => 1000];
                // If the mount is expanded, go down:

                if ($isOpen) {
                    // Set depth:
                    if ($this->addSelfId) {
                        $this->ids[] = $uid;
                    }
                    $this->getTree($uid, $limit);
                }
                // Add tree:
                $treeArr = array_merge($treeArr, $this->tree);
            }
        }
        return $this->printTree($treeArr);
    }

    /**
     * Returns the title for the input record. If blank, a "no title" label (localized) will be returned.
     * Do NOT htmlspecialchar the string from this function - has already been done.
     *
     * @param  array $row      The input row array (where the key "title" is used for the title)
     * @param  int   $titleLen Title length (30)
     *
     * @return string The title.
     */
    public function getTitleStr($row, $titleLen = 30)
    {
        return is_array($row) && isset($row['name']) ? substr(trim($row['name']), 0, $titleLen) : LocalizationUtility::translate('LLL:EXT:felogin/Resources/Private/Language/Database.xlf:tt_content.pi_flexform.groupSelectmode_showAll');
    }

    /**
     * Wrapping $title in a-tags.
     *
     * @param    string $title Title string
     * @param    array  $row   Item record
     * @param    int    $bank  Bank pointer (which mount point number)
     *
     * @internal
     */
    public function wrapTitle($title, $row, $bank = 0)
    {
        //if($row['uid']>0) {var_dump($row);exit(0);} console.log(document.getElementById(\'pageUid\').value);
        // $row['pid'] ? GeneralUtility::quoteJSvalue($row['pid']) :
        if($this->pageUid) {
            $aOnClick = 'return jumpTo(' . GeneralUtility::quoteJSvalue($row['pid'] ?? $this->pageUid) . ',' . GeneralUtility::quoteJSvalue($this->getJumpToParam($row)) . ',this,' . GeneralUtility::quoteJSvalue($this->domIdPrefix . $this->getId($row)) . ',' . $bank . ');';
        } else {
            $aOnClick = 'return jumpTo(' . ('document.getElementById(\'pageUid\').value') . ',' . GeneralUtility::quoteJSvalue($this->getJumpToParam($row)) . ',this,' . GeneralUtility::quoteJSvalue($this->domIdPrefix . $this->getId($row)) . ',' . $bank . ');';
        }
        return '<a href="#" onclick="' . htmlspecialchars($aOnClick) . '">' . $title . '</a>';
    }
    public function getDragDropCode($hlClass)
    {
        return '
            var currentState = ModuleStateStorage.current(\'' .  $this->mainModule . '\');
            Tree.highlightClass = "' . $hlClass . '";
            Tree.highlightActiveItem("", currentState.selection);
        ';
    }
    public function getJumpToScript($cMR)
    {
        return '
        // Function, loading the list frame from navigation tree:
        function jumpTo(pid,uid, linkObj, highlightID, bank) {
            var currentState = ModuleStateStorage.current(\'' .  $this->mainModule . '\');
            var theUrl = top.currentSubScript;
            if (theUrl.indexOf("?") != -1) {
                theUrl += "&id=" + pid
            } else {
                theUrl += "?id=" + pid
            }
                theUrl += "&' . $this->argumentName . '=" + uid
            ModuleStateStorage.update(\'' .  $this->mainModule . '\', \'mount\', bank);
            top.TYPO3.Backend.ContentContainer.setUrl(theUrl);
            ModuleStateStorage.update(\'' .  $this->mainModule . '\', pid, pid +"&' . $this->argumentName . '=" + uid);
            Tree.highlightActiveItem("' . $this->mainModule . '", highlightID + "_" + bank);

            if (linkObj) { linkObj.blur(); }
            return false;
        }
        ';
    }
}
