<?php

declare(strict_types=1);

namespace Internetgalerie\IgBackendHelpers\ViewHelpers\Link;

use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Core\Imaging\Icon;
use TYPO3\CMS\Core\Imaging\IconFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper;

/**
 * Use this ViewHelper to provide edit links to records. The ViewHelper will
 * pass the uid and table to FormEngine.
 *
 * The uid must be given as a positive integer.
 * For new records, use the :ref:`<be:link.newRecordViewHelper> <typo3-backend-link-newrecord>`.
 *
 * Examples
 * ========
 *
 * Link to the record-edit action passed to FormEngine::
 *
 *    <be:link.editRecord uid="42" table="a_table" redirect="foo/bar" />
 *
 * Output::
 *
 *    <a href="/typo3/index.php?route=/record/edit&edit[a_table][42]=edit&redirect=foo/bar">
 *        Edit record
 *    </a>
 *
 * Link to edit page uid=3 and then return back to the BE module "web_MyextensionList"::
 *
 *    <be:link.editRecord uid="3" table="pages" redirect="{f:be.uri(route: 'web_MyextensionList')}">
 */
class MoveRecordViewHelper extends AbstractTagBasedViewHelper
{
    protected $tagName = 'a';

    public function initializeArguments(): void
    {
        parent::initializeArguments();
        $this->registerUniversalTagAttributes();
        $this->registerArgument('uid', 'int', 'uid of record to be edited', true);
        $this->registerArgument('table', 'string', 'target database table', true);
        $this->registerArgument('move', 'int', 'pid or -uid to move', true);
        $this->registerArgument('direction', 'string', 'direction "up" or "down" dor data attribute and icon', false);
        $this->registerArgument('redirect', 'string', 'return to this URL after closing the edit dialog', false, '');
        $this->registerArgument('redirectAdditionalParams', 'array', 'AdditionalParams for redirect (used if redirect is not set)', false);
        $this->registerArgument('size', 'string', 'Desired size of the icon. All values of the Icons.sizes enum are allowed, these are: "small", "default", "large" and "overlay".', false, Icon::SIZE_SMALL);
        $this->overrideArgument('class', 'string', 'css class', false, 'btn btn-default');
    }

    /**
     * @throws \TYPO3\CMS\Backend\Routing\Exception\RouteNotFoundException
     */
    public function render(): string
    {
        if ($this->arguments['uid'] < 1) {
            throw new \InvalidArgumentException('Uid must be a positive integer, ' . $this->arguments['uid'] . ' given.', 1526127158);
        }
        if ($this->arguments['move'] === 0) {
            throw new \InvalidArgumentException(',move , ' . $this->arguments['move'] . ' given, 0 is not allowed.', 1526127159);
        }
        //$this->tag->addAttribute('data-params', ' data[' . $this->arguments['table'] . '][' . $this->arguments['uid'] . '][hidden]='. intval(!$hidden));
        //data-state="visible" data-params="{tableRow.links.disable}" title="" data-toggle-title="<f:translate key='LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:unHide' />" data-original-title="<f:translate key='LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:hide' />">

        // title="" data-original-title="<f:translate key='LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:edit' />"

        //$this->tag->addAttribute('class', trim($this->arguments['class'] . ' t3js-record-hide'));
        if (empty($this->arguments['redirect'])) {
            $this->arguments['redirect'] = GeneralUtility::getIndpEnv('REQUEST_URI');
            if ($this->arguments['redirectAdditionalParams']) {
                $this->arguments['redirect'] .= '&'  . http_build_query($this->arguments['redirectAdditionalParams']);
            }
        }
        $params = [
            'cmd['.$this->arguments['table'].'][' . $this->arguments['uid'] . '][move]' => $this->arguments['move'],
            'redirect' => $this->arguments['redirect'],
        ];
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        $uri = (string) $uriBuilder->buildUriFromRoute('tce_db', $params);
        // data-action="sort" data-direction="down" title="" data-original-title="<f:translate key='LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:moveDown' />" moveUp
        if ($this->arguments['direction']) {
            $this->tag->addAttribute('data-action', 'sort');
            $this->tag->addAttribute('data-direction', $this->arguments['direction']);
        }
        $content = $this->renderChildren();
        if ($content === null) {
            $size = $this->arguments['size'];
            $iconFactory = GeneralUtility::makeInstance(IconFactory::class);
            $content = $iconFactory->getIcon($this->arguments['direction'] === 'up' ? 'actions-move-up' : 'actions-move-down', $size, null, null)->render(null);
        }

        if ($this->arguments['direction'] === 'up') {
            $this->tag->addAttribute('title', LocalizationUtility::translate('LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:moveUp', 'core'));
        } elseif ($this->arguments['direction'] === 'down') {
            $this->tag->addAttribute('title', LocalizationUtility::translate('LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:moveDown', 'core'));
        }

        $this->tag->addAttribute('href', $uri);
        $this->tag->setContent($content);
        $this->tag->forceClosingTag(true);
        return $this->tag->render();
    }
}
