<?php

declare(strict_types=1);

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace Internetgalerie\IgBackendHelpers\Tree\Repository;

use PDO;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Backend\Utility\BackendUtility;
use TYPO3\CMS\Core\Authentication\BackendUserAuthentication;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\QueryHelper;
use TYPO3\CMS\Core\Database\Query\Restriction\DeletedRestriction;
use TYPO3\CMS\Core\DataHandling\PlainDataResolver;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Versioning\VersionState;

class CategoryTreeRepository
{
    /**
     * Fields to be queried from the database
     *
     * @var string[]
     */
    protected $fields = [
        'uid',
        'pid',
        'starttime',
        'endtime',
        'hidden',
        't3ver_oid',
        't3ver_wsid',
        't3ver_state',
        't3ver_stage',
    ];

    /**
     * The workspace ID to operate on
     *
     * @var int
     */
    protected $currentWorkspace = 0;

    /**
     * Full category tree when selected without permissions applied.
     *
     * @var array
     */
    protected $fullCategoryTree = [];

    /**
     * @var array
     */
    protected $additionalQueryRestrictions = [];

    /**
     * @param int $workspaceId the workspace ID to be checked for.
     * @param array $additionalFieldsToQuery an array with more fields that should be accessed.
     * @param array $additionalQueryRestrictions an array with more restrictions to add
     */
    public function __construct(protected $tableName, int $workspaceId = 0, array $additionalFieldsToQuery = [], array $additionalQueryRestrictions = [], protected $titleField = 'name', protected $parentField = 'parent')
    {
        $this->currentWorkspace = $workspaceId;
        if (!empty($additionalFieldsToQuery)) {
            $this->fields = array_merge($this->fields, $additionalFieldsToQuery);
        }

        if (!empty($additionalQueryRestrictions)) {
            $this->additionalQueryRestrictions = $additionalQueryRestrictions;
        }

        if($this->titleField) {
            $this->fields[] = $this->titleField;
        }
        if ($this->parentField) {
            $this->fields[] = $this->parentField;
        }
    }

    /**
     * Main entry point for this repository, to fetch the tree data for a category.
     * Basically the category record, plus all child categories and their child categories recursively, stored within "_children" item.
     *
     * @param array $mountPoint the category ID to fetch the tree for
     * @param int $parent
     * @param callable $callback a callback to be used to check for permissions and filter out categories not to be included.
     * @param array $dbMounts
     * @param bool $resolveUserPermissions
     * @return array
     */
    public function getTree(
        array $mountPoint,
        int $parent = 0,
        callable $callback = null,
        array $dbMounts = [],
        $resolveUserPermissions = false
    ): array {
        $this->fetchAllCategories($mountPoint, $parent);
        if ($parent === 0) {
            $tree = $this->fullCategoryTree;
        } else {
            $tree = $this->findInCategoryTree($parent, $this->fullCategoryTree);
        }
        if (!empty($tree) && $callback !== null) {
            $this->applyCallbackToChildren($tree, $callback);
        }
        //        var_dump($this->fullCategoryTree, $tree);exit(0);
        return $tree;
    }

    /**
     * Removes items from a tree based on a callback, usually used for permission checks
     *
     * @param array $tree
     * @param callable $callback
     */
    protected function applyCallbackToChildren(array &$tree, callable $callback)
    {
        if (!isset($tree['_children'])) {
            return;
        }
        foreach ($tree['_children'] as $k => &$childCategory) {
            if (!$callback($childCategory)) {
                unset($tree['_children'][$k]);
                continue;
            }
            $this->applyCallbackToChildren($childCategory, $callback);
        }
    }

    /**
     * Get the category tree based on a given category record and a given depth
     *
     * @param array $categoryTree The category record of the top level category you want to get the category tree of
     * @param int $depth Number of levels to fetch
     * @param int $pid
     * @return array An array with category records and their children
     */
    public function getTreeLevels(array $categoryTree, int $depth, $pid = 0): array
    {
        $parentIds = [$categoryTree['uid']];
        $groupedAndSortedCategoriesByPid = [];
        for ($i = 0; $i < $depth; $i++) {
            if (empty($parentIds)) {
                break;
            }
            $categoryRecords = $this->getChildCategoryRecords($parentIds, $pid);

            $groupedAndSortedCategoriesByPid = $this->groupAndSortCategories($categoryRecords, $groupedAndSortedCategoriesByPid);

            $parentIds = array_column($categoryRecords, 'uid');
        }
        $this->addChildrenToCategory($categoryTree, $groupedAndSortedCategoriesByPid);
        return $categoryTree;
    }

    /**
     * Retrieve the category records based on the given parent category ids
     *
     * @param array $parentCategoryIds
     * @param int $pid
     * @return array
     */
    protected function getChildCategoryRecords(array $parentIds, $pid = 0): array
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getQueryBuilderForTable($this->tableName);
        $queryBuilder->getRestrictions()
            ->removeAll()
            ->add(GeneralUtility::makeInstance(DeletedRestriction::class));

        if (!empty($this->additionalQueryRestrictions)) {
            foreach ($this->additionalQueryRestrictions as $additionalQueryRestriction) {
                $queryBuilder->getRestrictions()->add($additionalQueryRestriction);
            }
        }

        $queryBuilder
            ->select(...$this->fields)
            ->from($this->tableName)
            ->where(
                $queryBuilder->expr()->eq('pid', $queryBuilder->createNamedParameter($pid, Connection::PARAM_INT)),
            );
            if($this->parentField) {
                $queryBuilder->andWhere($queryBuilder->expr()->in($this->parentField, $queryBuilder->createNamedParameter($parentIds, Connection::PARAM_INT_ARRAY)));
            }
            if($GLOBALS['TCA'][$this->tableName]['ctrl']['languageField'] ?? false) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq($GLOBALS['TCA'][$this->tableName]['ctrl']['languageField'], $queryBuilder->createNamedParameter(0, Connection::PARAM_INT)));
            }
            /*->andWhere(
                QueryHelper::stripLogicalOperatorPrefix($GLOBALS['BE_USER']->getCategoryPermsClause(Permission::PAGE_SHOW))
            )*/
        $categoryRecords = $queryBuilder->executeQuery()
            ->fetchAllAssociative();

        // This is necessary to resolve all IDs in a workspace
        if ($this->currentWorkspace !== 0 && !empty($categoryRecords)) {
            $liveCategoryIds = [];
            $movedCategories = [];
            foreach ($categoryRecords as $categoryRecord) {
                $liveCategoryIds[] = (int)$categoryRecord['uid'];
                if ((int)$categoryRecord['t3ver_state'] === VersionState::MOVE_POINTER) {
                    $movedCategories[$categoryRecord['t3ver_oid']] = [
                        'pid' => (int)$categoryRecord['pid'],
                        'sorting' => (int)($categoryRecord['sorting'] ?? 0),
                    ];
                }
            }

            // Resolve placeholders of workspace versions
            $resolver = GeneralUtility::makeInstance(
                PlainDataResolver::class,
                $this->tableName,
                $liveCategoryIds
            );
            $resolver->setWorkspaceId($this->currentWorkspace);
            $resolver->setKeepDeletePlaceholder(false);
            $resolver->setKeepMovePlaceholder(false);
            $resolver->setKeepLiveIds(false);
            $recordIds = $resolver->get();

            if (!empty($recordIds)) {
                $queryBuilder->getRestrictions()->removeAll();
                $categoryRecords = $queryBuilder
                    ->select(...$this->fields)
                    ->from($this->tableName)->where($queryBuilder->expr()->in('uid', $queryBuilder->createNamedParameter($recordIds, Connection::PARAM_INT_ARRAY)))->executeQuery()
                    ->fetchAllAssociative();

                foreach ($categoryRecords as &$categoryRecord) {
                    if ((int)$categoryRecord['t3ver_state'] === VersionState::MOVE_POINTER && !empty($movedCategories[$categoryRecord['t3ver_oid']])) {
                        $categoryRecord['uid'] = $categoryRecord['t3ver_oid'];
                        $categoryRecord['sorting'] = (int)$movedCategories[$categoryRecord['t3ver_oid']]['sorting'];
                        $categoryRecord['pid'] = (int)$movedCategories[$categoryRecord['t3ver_oid']]['pid'];
                    } elseif ((int)$categoryRecord['t3ver_oid'] > 0) {
                        $liveRecord = BackendUtility::getRecord($this->tableName, $categoryRecord['t3ver_oid']);
                        $categoryRecord['sorting'] = (int)$liveRecord['sorting'];
                        $categoryRecord['uid'] = (int)$liveRecord['uid'];
                        $categoryRecord['pid'] = (int)$liveRecord['pid'];
                    }
                }
                unset($categoryRecord);
            } else {
                $categoryRecords = [];
            }
        }
        foreach ($categoryRecords as &$categoryRecord) {
            $categoryRecord['uid'] = (int)$categoryRecord['uid'];
        }

        return $categoryRecords;
    }

    public function hasChildren(int $parent, $pid = 0): bool
    {
        $categoryRecords = $this->getChildCategoryRecords([$parent], $pid);
        return !empty($categoryRecords);
    }

    /**
     * Fetch all non-deleted categories, regardless of permissions. That's why it's internal.
     * @return array the full category tree of the whole installation
     */
    protected function fetchAllCategories(array $mountPoint, $parent = 0): array
    {
        if ($GLOBALS['TCA'][$this->tableName]['ctrl']['sorting'] ?? false) {
            $this->fields[] = $GLOBALS['TCA'][$this->tableName]['ctrl']['sorting'];
        }
        $pid = $mountPoint['uid'];
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getQueryBuilderForTable($this->tableName);
        $queryBuilder->getRestrictions()
            ->removeAll()
            ->add(GeneralUtility::makeInstance(DeletedRestriction::class));

        if (!empty($this->additionalQueryRestrictions)) {
            foreach ($this->additionalQueryRestrictions as $additionalQueryRestriction) {
                $queryBuilder->getRestrictions()->add($additionalQueryRestriction);
            }
        }

        $query = $queryBuilder
            ->select('*')
            ->from($this->tableName)
            ->where($queryBuilder->expr()->eq('pid', $queryBuilder->createNamedParameter($pid, Connection::PARAM_INT)));


        if($GLOBALS['TCA'][$this->tableName]['ctrl']['languageField'] ?? false) {
            $queryBuilder->andWhere($queryBuilder->expr()->in($GLOBALS['TCA'][$this->tableName]['ctrl']['languageField'], [0, -1]));
        }

        if($parent) {
            $queryBuilder->andWhere(
                $queryBuilder->expr()->eq($this->parentField, $queryBuilder->createNamedParameter($parent, Connection::PARAM_INT))
            );
        }

        //if ($resolveUserPermissions) {
            /*$query->andWhere(
                QueryHelper::stripLogicalOperatorPrefix($GLOBALS['BE_USER']->getCategoryPermsClause(Permission::PAGE_SHOW))
            );*/
        //}
        $categoryRecords = $query->executeQuery()->fetchAllAssociative();

        // Now set up sorting, nesting (tree-structure) for all categories based on pid+sorting fields
        $groupedAndSortedCategoriesByPid = [];
        foreach ($categoryRecords as $categoryRecord) {
            $parentCategoryId = $this->parentField ? (int)$categoryRecord[$this->parentField] : null;
            $sorting = isset($categoryRecord['sorting']) ? (int)$categoryRecord['sorting'] : 0;
            while (isset($groupedAndSortedCategoriesByPid[$parentCategoryId][$sorting])) {
                $sorting++;
            }
            $groupedAndSortedCategoriesByPid[$parentCategoryId][$sorting] = $categoryRecord;
        }

        $this->fullCategoryTree = [
            'uid' => 0,
            'pid' => $pid,
            $this->titleField => $mountPoint['title'],
        ];
        $this->addChildrenToCategory($this->fullCategoryTree, $groupedAndSortedCategoriesByPid);
        return $this->fullCategoryTree;
    }

    /**
     * Adds the property "_children" to a category record with the child categories
     *
     * @param array $category
     * @param array[] $groupedAndSortedCategoriesByPid
     */
    protected function addChildrenToCategory(array &$category, array &$groupedAndSortedCategoriesByPid)
    {
        $category['_children'] = $groupedAndSortedCategoriesByPid[(int)$category['uid']] ?? [];
        ksort($category['_children']);
        foreach ($category['_children'] as &$child) {
            $this->addChildrenToCategory($child, $groupedAndSortedCategoriesByPid);
        }
    }

    /**
     * Looking for a category by traversing the tree
     *
     * @param int $categoryId the category ID to search for
     * @param array $categories the category tree to look for the category
     * @return array Array of the tree data, empty array if nothing was found
     */
    protected function findInCategoryTree(int $categoryId, array $categories): array
    {
        foreach ($categories['_children'] as $childCategory) {
            if ((int)$childCategory['uid'] === $categoryId) {
                return $childCategory;
            }
            $result = $this->findInCategoryTree($categoryId, $childCategory);
            if (!empty($result)) {
                return $result;
            }
        }
        return [];
    }

    /**
     * Retrieve the category tree based on the given search filter
     *
     * @param string $searchFilter
     * @param array $allowedMountPointCategoryIds
     * @param string $additionalWhereClause
     * @return array
     */
    public function fetchFilteredTree(string $searchFilter, $mountPoint, array $allowedMountPointCategoryIds, string $additionalWhereClause = ''): array
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)
            ->getQueryBuilderForTable($this->tableName);
        $queryBuilder->getRestrictions()
            ->removeAll()
            ->add(GeneralUtility::makeInstance(DeletedRestriction::class));

        if (!empty($this->additionalQueryRestrictions)) {
            foreach ($this->additionalQueryRestrictions as $additionalQueryRestriction) {
                $queryBuilder->getRestrictions()->add($additionalQueryRestriction);
            }
        }

        $expressionBuilder = $queryBuilder->expr();

        if ($this->currentWorkspace === 0) {
            // Only include records from live workspace
            $workspaceIdExpression = $expressionBuilder->eq('t3ver_wsid', 0);
        } else {
            // Include live records PLUS records from the given workspace
            $workspaceIdExpression = $expressionBuilder->in(
                't3ver_wsid',
                [0, $this->currentWorkspace]
            );
        }

        $queryBuilder = $queryBuilder
            ->select(...$this->fields)
            ->from($this->tableName)
            ->where(
                // Only show records in default language
                $workspaceIdExpression,
                QueryHelper::stripLogicalOperatorPrefix($additionalWhereClause)
            );


        if($GLOBALS['TCA'][$this->tableName]['ctrl']['languageField'] ?? false) {
            $queryBuilder->andWhere($queryBuilder->expr()->eq($GLOBALS['TCA'][$this->tableName]['ctrl']['languageField'], $queryBuilder->createNamedParameter(0, Connection::PARAM_INT)));
        }

        $searchParts = $expressionBuilder->or();
        if (is_numeric($searchFilter) && $searchFilter > 0) {
            $searchParts = $searchParts->with($expressionBuilder->eq('uid', $queryBuilder->createNamedParameter($searchFilter, Connection::PARAM_INT)));
        }
        $searchFilter = '%' . $queryBuilder->escapeLikeWildcards($searchFilter) . '%';

        $searchWhereAlias = $expressionBuilder->like(
            $this->titleField,
            $queryBuilder->createNamedParameter($searchFilter, Connection::PARAM_STR)
        );
        $searchParts = $searchParts->with($searchWhereAlias);

        $queryBuilder->andWhere($searchParts);
        $categoryRecords = $queryBuilder
            ->executeQuery()
            ->fetchAllAssociative();

        $liveCategoryPids = [];
        if ($this->currentWorkspace !== 0 && !empty($categoryRecords)) {
            $liveCategoryIds = [];
            foreach ($categoryRecords as $categoryRecord) {
                $liveCategoryIds[] = (int)$categoryRecord['uid'];
                $liveCategoryPids[(int)$categoryRecord['uid']] = (int)$categoryRecord['pid'];
                if ((int)$categoryRecord['t3ver_oid'] > 0) {
                    $liveCategoryPids[(int)$categoryRecord['t3ver_oid']] = (int)$categoryRecord['pid'];
                }
                if ((int)$categoryRecord['t3ver_state'] === VersionState::MOVE_POINTER) {
                    $movedCategories[$categoryRecord['t3ver_oid']] = [
                        'pid' => (int)$categoryRecord['pid'],
                        'sorting' => (int)$categoryRecord['sorting'],
                    ];
                }
            }
            // Resolve placeholders of workspace versions
            $resolver = GeneralUtility::makeInstance(
                PlainDataResolver::class,
                $this->tableName,
                $liveCategoryIds
            );
            $resolver->setWorkspaceId($this->currentWorkspace);
            $resolver->setKeepDeletePlaceholder(false);
            $resolver->setKeepMovePlaceholder(false);
            $resolver->setKeepLiveIds(false);
            $recordIds = $resolver->get();

            $categoryRecords = [];
            if (!empty($recordIds)) {
                $queryBuilder->getRestrictions()->removeAll();
                $queryBuilder
                    ->select(...$this->fields)
                    ->from($this->tableName)
                    ->where(
                        $queryBuilder->expr()->in('uid', $queryBuilder->createNamedParameter($recordIds, Connection::PARAM_INT_ARRAY))
                    );
                $queryBuilder->andWhere($searchParts);
                $categoryRecords = $queryBuilder
                    ->executeQuery()
                    ->fetchAllAssociative();
            }
        }

        $categories = [];
        $parentid = 0;
        foreach ($categoryRecords as $categoryRecord) {
            $parentid = $categoryRecord['pid'];
            // In case this is a record from a workspace
            // The uid+pid of the live-version record is fetched
            // This is done in order to avoid fetching records again (e.g. via BackendUtility::workspaceOL()
            if ((int)$categoryRecord['t3ver_oid'] > 0) {
                // This probably should also remove the live version
                if ((int)$categoryRecord['t3ver_state'] === VersionState::DELETE_PLACEHOLDER) {
                    continue;
                }
                // When a move pointer is found, the pid+sorting of the versioned record be used
                if ((int)$categoryRecord['t3ver_state'] === VersionState::MOVE_POINTER && !empty($movedCategories[$categoryRecord['t3ver_oid']])) {
                    $parentCategoryId = (int)$movedCategories[$categoryRecord['t3ver_oid']]['pid'];
                    $categoryRecord['sorting'] = (int)$movedCategories[$categoryRecord['t3ver_oid']]['sorting'];
                } else {
                    // Just a record in a workspace (not moved etc)
                    $parentCategoryId = (int)$liveCategoryPids[$categoryRecord['t3ver_oid']];
                }
                // this is necessary so the links to the modules are still pointing to the live IDs
                $categoryRecord['uid'] = (int)$categoryRecord['t3ver_oid'];
                $categoryRecord['pid'] = $parentCategoryId;
            }
            $categories[(int)$categoryRecord['uid']] = $categoryRecord;
        }
        //unset($categoryRecords);

        //$categories = $this->filterCategoriesOnMountPoints($categories, $allowedMountPointCategoryIds);

        $groupedAndSortedCategoriesByPid = $this->groupAndSortCategories($categoryRecords);

        $this->fullCategoryTree = [
            'uid' => 0,
            'pid' => $mountPoint['uid'],
            $this->titleField => $mountPoint['title'],
        ];
        $this->addChildrenToCategory($this->fullCategoryTree, $groupedAndSortedCategoriesByPid);
        //var_dump($this->fullCategoryTree);
        return $this->fullCategoryTree;
    }

    /**
     * Filter all records outside of the allowed mount points
     *
     * @param array $categories
     * @param array $mountPoints
     * @return array
     */
    protected function filterCategoriesOnMountPoints(array $categories, array $mountPoints): array
    {
        foreach ($categories as $key => $categoryRecord) {
            $rootline = BackendUtility::BEgetRootLine(
                $categoryRecord['uid'],
                '',
                $this->currentWorkspace !== 0,
                $this->fields
            );
            $rootline = array_reverse($rootline);
            if (!in_array(0, $mountPoints, true)) {
                $isInsideMountPoints = false;
                foreach ($rootline as $rootlineElement) {
                    if (in_array((int)$rootlineElement['uid'], $mountPoints, true)) {
                        $isInsideMountPoints = true;
                        break;
                    }
                }
                if (!$isInsideMountPoints) {
                    unset($categories[$key]);
                    //skip records outside of the allowed mount points
                    continue;
                }
            }

            $inFilteredRootline = false;
            $amountOfRootlineElements = count($rootline);
            for ($i = 0; $i < $amountOfRootlineElements; ++$i) {
                $rootlineElement = $rootline[$i];
                $rootlineElement['uid'] = (int)$rootlineElement['uid'];
                $isInWebMount = false;
                if ($rootlineElement['uid'] > 0) {
                    $isInWebMount = (int)$this->getBackendUser()->isInWebMount($rootlineElement);
                }

                if (!$isInWebMount
                    || ($rootlineElement['uid'] === (int)$mountPoints[0]
                        && $rootlineElement['uid'] !== $isInWebMount)
                ) {
                    continue;
                }
                if ($this->getBackendUser()->isAdmin() || ($rootlineElement['uid'] === $isInWebMount && in_array($rootlineElement['uid'], $mountPoints, true))) {
                    $inFilteredRootline = true;
                }
                if (!$inFilteredRootline) {
                    continue;
                }

                if (!isset($categories[$rootlineElement['uid']])) {
                    $categories[$rootlineElement['uid']] = $rootlineElement;
                }
            }
        }
        // Make sure the mountpoints show up in category tree even when parent categories are not accessible categories
        foreach ($mountPoints as $mountPoint) {
            if ($mountPoint !== 0) {
                if (!array_key_exists($mountPoint, $categories)) {
                    $categories[$mountPoint] = BackendUtility::getRecordWSOL($this->tableName, $mountPoint);
                    $categories[$mountPoint]['uid'] = (int)$categories[$mountPoint]['uid'];
                }
                $categories[$mountPoint]['pid'] = 0;
            }
        }

        return $categories;
    }

    /**
     * Group categories by parent category and sort categories based on sorting property
     *
     * @param array $categories
     * @param array $groupedAndSortedCategoriesByPid
     * @return array
     */
    protected function groupAndSortCategories(array $categories, $groupedAndSortedCategoriesByPid = []): array
    {
        foreach ($categories as $key => $categoryRecord) {
            $parentCategoryId = isset($categoryRecord['parent']) ? (int)$categoryRecord['parent'] : null;//pid is mount point
            $sorting = isset($categoryRecord['sorting']) ? (int)$categoryRecord['sorting'] : 0;
            while (isset($groupedAndSortedCategoriesByPid[$parentCategoryId][$sorting])) {
                $sorting++;
            }
            $groupedAndSortedCategoriesByPid[$parentCategoryId][$sorting] = $categoryRecord;
        }

        return $groupedAndSortedCategoriesByPid;
    }

    /**
     * @return BackendUserAuthentication
     */
    protected function getBackendUser(): BackendUserAuthentication
    {
        return $GLOBALS['BE_USER'];
    }
}
