<?php

declare(strict_types=1);

namespace Internetgalerie\IgBackendHelpers\ViewHelpers\Link;

use InvalidArgumentException;
use TYPO3\CMS\Backend\Routing\Exception\RouteNotFoundException;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Core\Imaging\Icon;
use TYPO3\CMS\Core\Imaging\IconFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper;

/**
 * Use this ViewHelper to provide edit links to records. The ViewHelper will
 * pass the uid and table to FormEngine.
 *
 * The uid must be given as a positive integer.
 * For new records, use the :ref:`<be:link.newRecordViewHelper> <typo3-backend-link-newrecord>`.
 *
 * Examples
 * ========
 *
 * Link to the record-edit action passed to FormEngine::
 *
 *    <be:link.editRecord uid="42" table="a_table" redirect="foo/bar" />
 *
 * Output::
 *
 *    <a href="/typo3/index.php?route=/record/edit&edit[a_table][42]=edit&redirect=foo/bar">
 *        Edit record
 *    </a>
 *
 * Link to edit page uid=3 and then return back to the BE module "web_MyextensionList"::
 *
 *    <be:link.editRecord uid="3" table="pages" redirect="{f:be.uri(route: 'web_MyextensionList')}">
 */
class MoveUpRecordViewHelper extends AbstractTagBasedViewHelper
{
    protected $tagName = 'a';

    public function initializeArguments(): void
    {
        parent::initializeArguments();
        $this->registerUniversalTagAttributes();
        $this->registerArgument('uid', 'int', 'uid of record to be edited', true);
        $this->registerArgument('pid', 'int', 'pid of current page', false);
        $this->registerArgument('table', 'string', 'target database table', true);
        $this->registerArgument('objects', 'array', 'all objects (sorting according the output)', true);
        $this->registerArgument('redirect', 'string', 'return to this URL after closing the edit dialog', false, '');
        $this->registerArgument('redirectAdditionalParams', 'array', 'AdditionalParams for redirect (used if redirect is not set)', false);
        $this->registerArgument('size', 'string', 'Desired size of the icon. All values of the Icons.sizes enum are allowed, these are: "small", "default", "large" and "overlay".', false, Icon::SIZE_SMALL);
        $this->overrideArgument('class', 'string', 'css class', false, 'btn btn-default');
    }

    /**
     * @throws RouteNotFoundException
     */
    public function render(): string
    {
        if ($this->arguments['uid'] < 1) {
            throw new InvalidArgumentException('Uid must be a positive integer, ' . $this->arguments['uid'] . ' given.', 1526127158);
        }
        if (! is_iterable($this->arguments['objects'])) {
            throw new InvalidArgumentException(',objects must be iterable (e.x. array).', 1526127160);
        }
        /*
        if (!is_array($this->arguments['objects'])) {
            throw new \InvalidArgumentException(',objects must be an array.', 1526127160);
        }
        */
        $objects = $this->arguments['objects'];
        $uid = $this->arguments['uid'];
        $size = $this->arguments['size'];

        if (empty($this->arguments['redirect'])) {
            $this->arguments['redirect'] = GeneralUtility::getIndpEnv('REQUEST_URI');
            if ($this->arguments['redirectAdditionalParams']) {
                $this->arguments['redirect'] .= '&'  . http_build_query($this->arguments['redirectAdditionalParams']);
            }
        }

        $iconFactory = GeneralUtility::makeInstance(IconFactory::class);
        //$content=$this->renderChildren();
        //if( $content )
        //IconState::STATE_DEFAULT
        //$size=Icon::SIZE_SMALL;
        //$state = IconState::cast($arguments['state']);

        $found = false;
        $move = 0;
        $prevUid = $this->arguments['pid'] ?? $GLOBALS['TYPO3_REQUEST']->getQueryParams()['id'] ?? null;//pageUid ist dies korrekt?
        $prevPrevUid = 0;
        if (is_array($this->arguments['objects'])) {
            foreach ($objects as $object) {
                //echo('test:' . $object['uid'] .'=='. $uid .'<br />');
                if ($object['uid'] === $uid) {
                    $found = true;
                    $move = $prevPrevUid;
                    break;
                }
                $prevPrevUid = $prevUid;
                $prevUid = -$object['uid'];
            }
        } else {
            foreach ($objects as $object) {
                //echo('test:' . $object['uid'] .'=='. $uid .'<br />');
                if ($object->getUid() === $uid) {
                    $found = true;
                    $move = $prevPrevUid;
                    break;
                }
                $prevPrevUid = $prevUid;
                $prevUid = -$object->getUid();
            }
        }
        //die('found='. $move);
        if ($move !== 0) {
            $content = $iconFactory->getIcon('actions-move-up', $size, null, null)->render(null);
            $params = [
                'cmd['.$this->arguments['table'].'][' . $uid . '][move]' => $move,
                'redirect' => $this->arguments['redirect'],
            ];
            $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
            $uri = (string) $uriBuilder->buildUriFromRoute('tce_db', $params);
            $this->tag->addAttribute('data-action', 'sort');
            $this->tag->addAttribute('data-direction', 'up');
            $class = $this->arguments['class'];
            $this->tag->addAttribute('title', LocalizationUtility::translate('LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:moveUp', 'core'));
            $this->tag->addAttribute('href', $uri);
        } else {
            // move == 0 , found==false -> errror (objekt nicht in liste)
            $this->tag->setTagName('span');
            $content = $iconFactory->getIcon('empty-empty', $size, null, null)->render(null);
            $class = trim($this->arguments['class'] . ' disabled');
        }

        $renderChildrenContent = $this->renderChildren();
        if ($renderChildrenContent !== null) {
            $content = $renderChildrenContent;
        }

        $this->tag->addAttribute('class', $class);
        $this->tag->setContent($content);
        $this->tag->forceClosingTag(true);
        return $this->tag->render();
    }
}
