<?php

declare(strict_types=1);

namespace Internetgalerie\IgBackendHelpers\ViewHelpers\Link;

use InvalidArgumentException;
use TYPO3\CMS\Backend\Routing\Exception\RouteNotFoundException;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Core\Imaging\Icon;
use TYPO3\CMS\Core\Imaging\IconFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper;

/**
 * Use this ViewHelper to provide edit links to records. The ViewHelper will
 * pass the uid and table to FormEngine.
 *
 * The uid must be given as a positive integer.
 * For new records, use the :ref:`<be:link.newRecordViewHelper> <typo3-backend-link-newrecord>`.
 *
 * Examples
 * ========
 *
 * Link to the record-delete action passed to FormEngine::
 *
 *    <be:link.deleteRecord uid="42" table="a_table" redirect="foo/bar" />
 *
 * Output::
 *
 *    <a href="/typo3/index.php?route=/record/commit&cmd[a_table][42][delete]=1&redirect=foo/bar">
 *        Delete record
 *    </a>
 *
 * Link to delete page uid=3 and then return back to the BE module "web_MyextensionList"::
 *
 *    <be:link.deleteRecord uid="3" table="pages" redirect="{f:be.uri(route: 'web_MyextensionList')}">
 */
class DeleteRecordViewHelper extends AbstractTagBasedViewHelper
{
    protected $tagName = 'a';

    public function initializeArguments(): void
    {
        parent::initializeArguments();
        $this->registerUniversalTagAttributes();
        $this->registerArgument('uid', 'int', 'uid of record to be deleteed', true);
        $this->registerArgument('table', 'string', 'target database table', true);
        $this->registerArgument('redirect', 'string', 'redirect to this URL after deleting', false, '');
        $this->registerArgument('redirectAdditionalParams', 'array', 'AdditionalParams for returnUrl (used if returnUrl is not set)', false);
        $this->registerArgument('additionalParams', 'array', 'query parameters to be attached to the resulting URI', false, []);
        $this->registerArgument('size', 'string', 'Desired size of the icon. All values of the Icons.sizes enum are allowed, these are: "small", "default", "large" and "overlay".', false, Icon::SIZE_SMALL);
        $this->overrideArgument('class', 'string', 'css class', false, 'btn btn-default');
    }

    /**
     * @throws RouteNotFoundException
     */
    public function render(): string
    {
        if ($this->arguments['uid'] < 1) {
            throw new InvalidArgumentException('Uid must be a positive integer, ' . $this->arguments['uid'] . ' given.', 1526127158);
        }
        if (empty($this->arguments['redirect'])) {
            $this->arguments['redirect'] = GeneralUtility::getIndpEnv('REQUEST_URI');
            if ($this->arguments['redirectAdditionalParams']) {
                $this->arguments['redirect'] .= '&'  . http_build_query($this->arguments['redirectAdditionalParams']);
            }
        }
        // title="" data-original-title="<f:translate key='LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:delete' />\" data-l10nparent="{tableRow.product.l10nParent}" data-message="<f:translate key='LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:clip_deleteMarkedWarning' arguments='{0: \'{tableName}\'}' />" data-params="{tableRow.links.delete}" title="" data-original-title=""
        $params = $this->arguments['additionalParams'];
        $params['cmd['.$this->arguments['table'].'][' . $this->arguments['uid'] . '][delete]'] = 1;
        $params['redirect'] = $this->arguments['redirect'];
        
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        $uri = (string) $uriBuilder->buildUriFromRoute('tce_db', $params);

        $content = $this->renderChildren();
        if ($content === null) {
            $size = $this->arguments['size'];
            $iconFactory = GeneralUtility::makeInstance(IconFactory::class);
            $content = $iconFactory->getIcon('actions-edit-delete', $size, null, null)->render(null);
        }
        $this->tag->addAttribute('title', LocalizationUtility::translate('LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:delete', 'core'));

        $this->tag->addAttribute('href', $uri);
        $this->tag->setContent($content);
        $this->tag->forceClosingTag(true);
        return $this->tag->render();
    }
}
