<?php

declare(strict_types=1);

namespace Internetgalerie\IgBackendHelpers\ViewHelpers\Link;

use InvalidArgumentException;
use TYPO3\CMS\Backend\Routing\Exception\RouteNotFoundException;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Core\Imaging\Icon;
use TYPO3\CMS\Core\Imaging\IconFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper;

/**
 * Use this ViewHelper to provide edit links to records. The ViewHelper will
 * pass the uid and table to FormEngine.
 *
 * The uid must be given as a positive integer.
 * For new records, use the :ref:`<be:link.newRecordViewHelper> <typo3-backend-link-newrecord>`.
 *
 * Examples
 * ========
 *
 * Link to the record-edit action passed to FormEngine::
 *
 *    <be:link.editRecord uid="42" table="a_table" returnUrl="foo/bar" />
 *
 * Output::
 *
 *    <a href="/typo3/index.php?route=/record/edit&edit[a_table][42]=edit&returnUrl=foo/bar">
 *        Edit record
 *    </a>
 *
 * Link to edit page uid=3 and then return back to the BE module "web_MyextensionList"::
 *
 *    <be:link.editRecord uid="3" table="pages" returnUrl="{f:be.uri(route: 'web_MyextensionList')}">
 */
class EditRecordViewHelper extends AbstractTagBasedViewHelper
{
    protected $tagName = 'a';

    public function initializeArguments(): void
    {
        parent::initializeArguments();
        $this->registerUniversalTagAttributes();
        $this->registerArgument('uid', 'int', 'uid of record to be edited', true);
        $this->registerArgument('table', 'string', 'target database table', true);
        $this->registerArgument('fields', 'string', 'Edit only these fields (comma separated list)');
        $this->registerArgument('returnUrl', 'string', 'return to this URL after closing the edit dialog', false, '');
        $this->registerArgument('returnUrlAdditionalParams', 'array', 'AdditionalParams for returnUrl (used if returnUrl is not set)', false);
        $this->registerArgument('additionalParams', 'array', 'query parameters to be attached to the resulting URI', false, []);
        $this->registerArgument('size', 'string', 'Desired size of the icon. All values of the Icons.sizes enum are allowed, these are: "small", "default", "large" and "overlay".', false, Icon::SIZE_SMALL);
        $this->overrideArgument('class', 'string', 'css class', false, 'btn btn-default');
    }

    /**
     * @throws RouteNotFoundException
     */
    public function render(): string
    {
        if ($this->arguments['uid'] < 1) {
            throw new InvalidArgumentException('Uid must be a positive integer, ' . $this->arguments['uid'] . ' given.', 1526127158);
        }
        if (empty($this->arguments['returnUrl'])) {
            $this->arguments['returnUrl'] = GeneralUtility::getIndpEnv('REQUEST_URI');
            if ($this->arguments['returnUrlAdditionalParams']) {
                $this->arguments['returnUrl'] .= '&'  . http_build_query($this->arguments['returnUrlAdditionalParams']);
            }
        }
        $params = $this->arguments['additionalParams'];
        $params['edit'] = [$this->arguments['table'] => [$this->arguments['uid'] => 'edit']];
        $params['returnUrl'] = $this->arguments['returnUrl'];

        if ($this->arguments['fields'] ?? false) {
            $params['columnsOnly'] = [
                $this->arguments['table'] => GeneralUtility::trimExplode(',', $this->arguments['fields'], true),
            ];
        }
        // Add extra parameter for `recurrence[update_scope]`
        //$params['recurrence'] = ['update_scope' => 'this'];
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        $uri = (string) $uriBuilder->buildUriFromRoute('record_edit', $params);

        $content = $this->renderChildren();
        if ($content === null) {
            $size = $this->arguments['size'];
            $iconFactory = GeneralUtility::makeInstance(IconFactory::class);
            $content = $iconFactory->getIcon('actions-open', $size, null, null)->render(null);
        }
        if (empty($this->arguments['title'])) {
            $this->tag->addAttribute('title', LocalizationUtility::translate('LLL:EXT:core/Resources/Private/Language/locallang_mod_web_list.xlf:edit', 'core'));
        }

        $this->tag->addAttribute('href', $uri);
        $this->tag->setContent($content);
        $this->tag->forceClosingTag(true);
        return $this->tag->render();
    }
}
