<?php

declare(strict_types=1);

namespace Ig\IgContainer\DataProcessing;

use B13\Container\DataProcessing\ContainerDataProcessingFailedException;
use B13\Container\Domain\Factory\ContainerFactory;
use B13\Container\Domain\Factory\Exception;
use B13\Container\Domain\Model\Container;
use Psr\Http\Message\ServerRequestInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Information\Typo3Version;
use Ig\IgResponsiveImages\Domain\Model\ResponsiveImagesConfiguration;
use Ig\IgResponsiveImages\Utility\ResponsiveImagesUtility;
use TYPO3\CMS\Core\Context\LanguageAspect;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Frontend\ContentObject\ContentDataProcessor;
use TYPO3\CMS\Frontend\ContentObject\ContentObjectRenderer;
use TYPO3\CMS\Frontend\ContentObject\DataProcessorInterface;
use TYPO3\CMS\Frontend\ContentObject\RecordsContentObject;

class ContainerProcessor implements DataProcessorInterface
{
    /**
     * @var ContainerFactory
     */
    protected $containerFactory;

    /**
     * @var ContentDataProcessor
     */
    protected $contentDataProcessor;

    protected Context $context;
    
    public function __construct(ContainerFactory $containerFactory, ContentDataProcessor $contentDataProcessor, Context $context)
    {
        $this->containerFactory = $containerFactory;
        $this->contentDataProcessor = $contentDataProcessor;
        $this->context = $context;
    }

    /**
     * @param ContentObjectRenderer $cObj
     * @param array $contentObjectConfiguration
     * @param array $processorConfiguration
     * @param array $processedData
     * @return array
     */
    public function process(
        ContentObjectRenderer $cObj,
        array $contentObjectConfiguration,
        array $processorConfiguration,
        array $processedData
    ) {
        // BEGIN IG
        $flexFormString = $cObj->data['pi_flexform'];
        $flexForm = GeneralUtility::makeInstance(\TYPO3\CMS\Core\Service\FlexFormService::class)
            ->convertFlexFormContentToArray($flexFormString);
        $imageRenderingClass = $flexForm['image_rendering_class'] ?? null;
        if ($imageRenderingClass && $processorConfiguration['igresponsiveimages.']) {
            $processorConfiguration['igresponsiveimages.']['cols'] = $imageRenderingClass;
        }
        if ($processorConfiguration['igresponsiveimages.'] && $processorConfiguration['igresponsiveimages.']['cols'] == 'X') {
            $colWidthInt = (int)str_replace('ym-g', '', $flexForm['width_column_normal_1']);
            if ($colWidthInt > 0) {
                $colsInt = round(100 / $colWidthInt);

                if ($colsInt > 10) {
                    $colsInt = 10;
                }
                if ($colsInt < 1) {
                    $colsInt = 1;
                }
                $processorConfiguration['igresponsiveimages.']['colWidth'] = (string)$colWidthInt;
                $processorConfiguration['igresponsiveimages.']['cols'] = (string)$colsInt;
            }
        }
        // END IG

        if (isset($processorConfiguration['if.']) && !$cObj->checkIf($processorConfiguration['if.'])) {
            return $processedData;
        }
        if ($processorConfiguration['contentId.'] ?? false) {
            $contentId = (int)$cObj->stdWrap($processorConfiguration['contentId'], $processorConfiguration['contentId.']);
        } elseif ($processorConfiguration['contentId'] ?? false) {
            $contentId = (int)$processorConfiguration['contentId'];
        } else {
            $contentId = (int)$cObj->data['uid'];
        }

        try {
            $container = $this->containerFactory->buildContainer($contentId);
        } catch (Exception $e) {
            // do nothing
            return $processedData;
        }

        if (empty($processorConfiguration['colPos']) && empty($processorConfiguration['colPos.'])) {
            $allColPos = $container->getChildrenColPos();
            foreach ($allColPos as $colPos) {
                $processedData = $this->processColPos(
                    $cObj,
                    $container,
                    $colPos,
                    'children_' . $colPos,
                    $processedData,
                    $processorConfiguration
                );
            }
        } else {
            if ($processorConfiguration['colPos.'] ?? null) {
                $colPos = (int)$cObj->stdWrap($processorConfiguration['colPos'], $processorConfiguration['colPos.']);
            } else {
                $colPos = (int)$processorConfiguration['colPos'];
            }
            $as = 'children';
            if ($processorConfiguration['as']) {
                $as = $processorConfiguration['as'];
            }
            $processedData = $this->processColPos(
                $cObj,
                $container,
                $colPos,
                $as,
                $processedData,
                $processorConfiguration
            );
        }
        return $processedData;
    }

    protected function processColPos(
        ContentObjectRenderer $cObj,
        Container $container,
        int $colPos,
        string $as,
        array $processedData,
        array $processorConfiguration
    ): array {
        $children = $container->getChildrenByColPos($colPos);

        $typo3Version = GeneralUtility::makeInstance(Typo3Version::class);
        if ($typo3Version->getMajorVersion() >= 13) {
            $contentRecordRenderer = $cObj->getContentObject('RECORDS');
            if ($contentRecordRenderer === null) {
                throw new ContainerDataProcessingFailedException('RECORDS content object not available.', 1691483526);
            }
        } elseif ($typo3Version->getMajorVersion() < 12) {
            $contentRecordRenderer = new RecordsContentObject($cObj);
        } else {
            $contentRecordRenderer = new RecordsContentObject(); // $GLOBALS['TSFE']
            $contentRecordRenderer->setRequest($this->getRequest());
            $contentRecordRenderer->setContentObjectRenderer($cObj);
        }

        $conf = [
            'tables' => 'tt_content'
        ];
        /** @var LanguageAspect $languageAspect */
        $doOverlays = false;
        $languageAspect = $this->context->getAspect('language');
        $doOverlays = $languageAspect->doOverlays();;
        $validChildren = [];
        foreach ($children as &$child) {
            if ($child['hidden'] > 0) {
                continue;
            }
            if ($child['l18n_parent'] > 0 && $doOverlays) {
                $conf['source'] = $child['l18n_parent'];
            } else {
                $conf['source'] = $child['uid'];
            }
            if ($child['t3ver_oid'] > 0) {
                $conf['source'] = $child['t3ver_oid'];
            }
            // IG BEGIN
            $responsiveSet = false;
            if (!empty($processorConfiguration['igresponsiveimages.'])) {
                $responsiveImageUtility = GeneralUtility::makeInstance(ResponsiveImagesUtility::class);

                $igresponsiveimages = $processorConfiguration['igresponsiveimages.'];
                $configName = $igresponsiveimages['config'] ?? $igresponsiveimages['cols'] ?? 'default';

                $colWidth = $igresponsiveimages['colWidth'] ?? null;
                /**
                 * @todo respect all with media query
                 * $child['tx_igcontainer_large_width']
                 * $child['tx_igcontainer_medium_width']
                 * $child['tx_igcontainer_small_width']
                 */
                if ($child['tx_igcontainer_large_width'] > 0) {
                    $colWidth = $child['tx_igcontainer_large_width'];
                }
                $responsiveImagesConfiguration = GeneralUtility::makeInstance(ResponsiveImagesConfiguration::class);
                $responsiveImagesConfiguration->setProcessorConfiguration($igresponsiveimages);
                $responsiveImagesConfiguration->setColWidth($colWidth);
                $responsiveImagesConfiguration->setConfigName($configName);
                $responsiveImageUtility->pushContainer($responsiveImagesConfiguration);
                $responsiveSet = true;
            }
            // IG END
            $child['renderedContent'] = $cObj->render($contentRecordRenderer, $conf);
            /** @var ContentObjectRenderer $recordContentObjectRenderer */
            $recordContentObjectRenderer = GeneralUtility::makeInstance(ContentObjectRenderer::class);
            $recordContentObjectRenderer->start($child, 'tt_content');
            $child = $this->contentDataProcessor->process($recordContentObjectRenderer, $processorConfiguration, $child);
            // IG BEGIN
            if ($responsiveSet) {
                $responsiveImageUtility->popContainer($responsiveImagesConfiguration);
            }
            $validChildren[] = $child;
            // IG END
        }
        $processedData[$as] = $validChildren;
        return $processedData;
    }

    protected function getRequest(): ServerRequestInterface
    {
        return $GLOBALS['TYPO3_REQUEST'];
    }
}
