class IgConsentManager {


    constructor(config) {
        const containerId = '#ig-consent-container';
        this.confirmed = false; // true if the user actively confirmed his/her consent
        this.changed = false // true if the service config changed compared to the cookie
        this.config = config;
        this.purposes = {};
        this.services = {};
        this.translations = {};
        this.consents = {};
        this.watchers = new Set([]);
        this.states = {} // keep track of the change (enabled, disabled) of individual services
        this.initialized = {} // keep track of which services have been initialized already
        this.executedOnce = {} //keep track of which services have been executed at least once
        this.cookieBannerElement = document.querySelector(containerId);
        if (this.cookieBannerElement == null) {
            console.error('no config Url found (no cookie banner found)');
            return;
        }
        const configUrl = this.cookieBannerElement.dataset.configUrl;
        // load config
        fetch(configUrl)
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.json();
            })
            .then(data => {
                this.config = Object.assign({}, data.config, config);
                this.services = data.services;
                this.purposes = data.purposes;
                this.translations = data.translations;
                this.loadConsents();
            })
            .catch(error => {
                console.error('Error making AJAX call:', error);
            });
    }

    defaultConsents() {
        const consents = {}
        for (let i = 0; i < this.services.length; i++) {
            const service = this.services[i]
            consents[service.name] = this.getDefaultConsent(service)
        }
        return consents
    }
    getDefaultConsent(service) {
        let consent = service.default || service.required
        if (consent === undefined)
            consent = false
        return consent
    }

    getCookies() {
        //return document.cookie.split(';').reduce((ac, str) => Object.assign(ac, {[str.split('=')[0].trim()]: str.split('=')[1]}), {});
        const cookiesArray = document.cookie.split('; ');
        const cookies = [];
        cookiesArray.forEach(cookie => {
            const [cookeName, cookieValue] = cookie.split('=');
            cookies.push({ name: cookeName, value: cookieValue });
        });
        return cookies;
    }

    deleteCookie(name, path, domain) {
        document.cookie = name + "=" +
            ((path) ? ";path=" + path : "") +
            ((domain) ? ";domain=" + domain : "") +
            ";expires=Thu, 01 Jan 1970 00:00:01 GMT";
    }

    loadConsents() {
        const consentData = localStorage.getItem(this.config.storageName);
        this.consents = {};
        this._checkConsents();
        this.renderContextualConsentNotices();
        // wait for async cookies
        setTimeout(() => {
            this.applyConsents(true);
        }, 0);
        this.notify(this.consents)

        //this.services = JSON.parse(this.cookieBannerElement.dataset.services);

        if (consentData !== null) {
            this.consents = JSON.parse(decodeURIComponent(consentData))
            this.confirmed = true;
        }
        return this.consents
    }

    resetConsents() {
        this.consents = {};
        this.states = {}
        this.services.forEach((element, index, array) => { this.consents[element.name] = false; });
        this.confirmed = false;
    }
    changeAll(value) {
        let changedServices = 0
        this.services.filter(service => !service.contextualConsentOnly).map(service => {
            if (service.required || value) {
                // console.log(service.name, value);
                if (this.updateConsent(service.name, true))
                    changedServices++
            } else {
                if (this.updateConsent(service.name, false))
                    changedServices++
            }
        });
        return changedServices
    }

    saveAndApplyConsents() {
        this.saveConsents()
        this.applyConsents()
    }

    saveConsents() {
        // console.log(this.consents);
        const v = encodeURIComponent(JSON.stringify(this.consents));
        localStorage.setItem(this.config.storageName, v);
        this.confirmed = true;
        this.changed = false;
    }

    // set constent to value (true/false)
    updateConsent(name, value) {
        const changed = (this.consents[name] || false) !== value;
        this.consents[name] = value;
        this.confirmed = false;
        this.notify(this.consents);
        return changed;
    }
    executeHandler(handler, opts) {
        if (handler === undefined || handler === '')
            return
        let handlerFunction
        if (typeof handler === 'function') {
            handlerFunction = handler
        } else {
            // eslint-disable-next-line no-new-func
            handlerFunction = new Function('opts', handler)
        }
        return handlerFunction(opts);
    }

    applyConsents(noStorageUpdate) {
        for (const service of this.services) {
            if (!this.initialized[service.name]) {
                this.initialized[service.name] = true;
                const handlerOpts = { manager: this, service: service }
                this.executeHandler(service.onInit, handlerOpts);
            }
        }

        for (const service of this.services) {
            const consent = this.consents[service.name] ? true : false;
            const state = this.states[service.name];
            const changed = state !== consent;
            const handlerOpts = { manager: this, service: service, consents: this.consents, changed: changed }
            this.executeHandler(consent ? service.onAccept : service.onDecline, handlerOpts)
            this.updateServiceElements(service, consent);
            if (noStorageUpdate) {
                continue;
            }
            this.updateServiceStorage(service, consent);
            this.states[service.name] = consent;
        }
        this.notify(this.consents);
    }

    getService(name) {
        return this.services.find((service) => service.name === name);
    }
    // hide pupose in UI if no service is shown (all are contextualConsentOnly) and hideServices is not set
    getUsedPurposes() {
        let usedPurposes = [];
        for (const purpose of this.purposes) {
            if (!purpose.hideServices) {
                let purposeIsUsed = false;
                for (const service of this.services) {
                    if (service.purpose == purpose.name && (!service.contextualConsentOnly || this.getConsent(service.name))) {
                        purposeIsUsed = true;
                        break;
                    }
                }
                if (!purposeIsUsed) {
                    continue;
                }
            }
            usedPurposes.push(purpose);
        };
        return usedPurposes;
    }

    watch(watcher) {
        if (!this.watchers.has(watcher)) {
            this.watchers.add(watcher);
        }
        if (watcher.hasOwnProperty('init')) {
            watcher.init(this.consents);
        }
    }
    unwatch(watcher) {
        if (this.watchers.has(watcher)) {
            this.watchers.delete(watcher);
        }
    }
    update() {
        this.loadConsents();
        this.notify(this.consents);
    }
    notify(data) {
        this.watchers.forEach((watcher) => {
            watcher.update(data)
        });
    }
    getConsent(name) {
        return this.consents[name] || false;
    }



    // src/lib.js -> renderContextualConsentNotices
    renderContextualConsentNotices() {
        for (const service of this.services) {
            if (!service.contextualText && !service.contextualButton)
                continue
            const consent = this.getConsent(service.name) && this.confirmed;
            const elements = document.querySelectorAll("[data-consent='" + service.name + "']");
            for (const element of elements) {
                if (element.dataset.type === 'placeholder')
                    continue

                if (element.tagName === 'IFRAME' || element.tagName === 'DIV') {
                    let placeholderElement = element.previousElementSibling
                    if (placeholderElement !== null) {
                        if (placeholderElement.dataset.type !== "placeholder" || placeholderElement.dataset.name !== service.name)
                            placeholderElement = null
                    }
                    if (placeholderElement === null) {
                        placeholderElement = this.generatePlaceholder(service);
                        //if (consent)
                        //    placeholderElement.style.display = 'none';
                        let insertedPlaceholderElement = element.parentElement.insertBefore(placeholderElement, element);
                        // set height of placeholder to element height
                        let elementHeight = element.clientHeight || element.style.height || element.height || 0;
                        // only set height on bigger element than itself
                        if (elementHeight > insertedPlaceholderElement.clientHeight) {
                            insertedPlaceholderElement.style.height = elementHeight + 'px';
                        }
                    }
                }
            }
        }
    }

    generatePlaceholder(service) {
        let htmlParser = new DOMParser();
        let parsedHTML = htmlParser.parseFromString('<div class="ig-consent-container-placeholder ig-consent-container-placeholder-' + service.name + '" data-type="placeholder" data-consent="' + service.name + '">'
            + '<div>'
            + '<p>' + service.contextualText + '</p>'
            + (service.policyUrl ? '<div class="frame-space-after-small ig-consent-container-more"><a href="' + service.policyUrl + '" target="_blank" class="ig-consent-container-more">' + service.policyText + '</a></div>' : '')
            + '<div class="frame-space-after-small ig-consent-container-button"><a href="#" class="link-button ig-consent-button ig-consent-button-load">' + service.contextualButton + '</a></div>'
            + '<div class="ig-consent-container-always"><label><input type="checkbox" class="ig-consent-input-all" checked="1">' + (service.alwaysUnlockText || (this.translations['alwaysUnlockBefore'] ? this.translations['alwaysUnlockBefore'] + ' ' : '') + service.title + (this.translations['alwaysUnlockAfter'] ? '  ' + this.translations['alwaysUnlockAfter'] : '')) + '</input></label></div>'
            //+ '<a href="#" class="ig-consent-button ig-consent-button-settings">Settings</a>'
            + '</div>'
            + '</div>', 'text/html');
        let container = parsedHTML.body.firstChild;
        let loadLink = container.querySelector('a.ig-consent-button-load');
        /*
        let settingsLink = container.querySelector('a.ig-consent-button-settings');
        if (settingsLink) {
            settingsLink.addEventListener('click', (e) => {
                e.preventDefault();
                this.cookieBannerElement.removeAttribute('style');
                this.cookieBannerElement.querySelector('.ig_consent_settings_button').click();
            });
        }
        */
        if (loadLink) {
            loadLink.addEventListener('click', (e) => {
                e.preventDefault();
                let checkbox = container.querySelector('.ig-consent-input-all');
                let loadAll = checkbox.checked || false;
                if (loadAll) {
                    let isConfirmed = this.confirmed;
                    //console.log('load all element');
                    this.updateConsent(service.name, true);
                    if (isConfirmed) {// we permanently save the consent state
                        this.saveConsents();
                    }
                    this.applyConsents();
                } else {
                    //console.log('load one element');
                    this.updateConsent(service.name, true);
                    this.applyConsents();
                }

            });
        }


        return container;
    }

    // src/consent-manager.js -> updateServiceStorage
    // update/delete cookies
    updateServiceStorage(service, consent) {
        if (consent)
            return
        function escapeRegexStr(str) {
            return str.replace(/[-[\]/{}()*+?.\\^$|]/g, "\\$&");
        }

        if (service.cookies !== undefined && Object.keys(service.cookies).length > 0) {
            const cookies = this.getCookies();
            for (const key of Object.keys(service.cookies)) {
                const serviceCookie = service.cookies[key];
                let cookiePattern = serviceCookie.pattern;
                let cookiePath = serviceCookie.path;
                let cookieDomain = serviceCookie.domain;

                if (cookiePattern === undefined)
                    continue
                if (!(cookiePattern instanceof RegExp)) {
                    if (cookiePattern.startsWith('/^')) {// we assume this is already a regex
                        try {
                            const regexParts = cookiePattern.match(/^\/(.*?)\/([gimy]*)$/);
                            if (regexParts) {
                                const regexPattern = regexParts[1];
                                const regexFlags = regexParts[2];
                                cookiePattern = new RegExp(regexPattern, regexFlags);
                            } else {
                                console.log('Invalid regular expression format');
                            }
                        } catch (error) {
                            console.log('Invalid regular expression:', error.message);
                        }
                    } else {// we assume this is a normal string
                        cookiePattern = new RegExp('^' + escapeRegexStr(cookiePattern) + '$')
                    }
                }
                //console.log(cookies, cookies.length);

                for (let j = 0; j < cookies.length; j++) {
                    const cookie = cookies[j]
                    const match = cookiePattern.exec(cookie.name)
                    //console.log('cookie=' + cookie.name, cookiePattern, match);
                    if (match !== null) {
                        // eslint-disable-next-line no-console
                        /*
                        console.log("Deleting cookie:", cookie.name,
                            "Matched pattern:", cookiePattern,
                            "Path:", cookiePath,
                                      "Domain:", cookieDomain);
                        */
                        this.deleteCookie(cookie.name, cookiePath, cookieDomain)
                        // if no cookie domain is given, we also try to delete the cookie with
                        // domain '.[current domain]' as some services set cookies for this
                        // dotted domain explicitly (e.g. the Facebook pixel) and '[current fullHostName]'.
                        if (cookieDomain === undefined || cookieDomain === '') {
                            let fullHostname = window.location.hostname;
                            let parts = fullHostname.split('.');
                            let dottedDomain = `.${parts.slice(-2).join('.')}`;
                            //console.log('delete cookies for ' + fullHostname + ' and ' + dottedDomain);
                            this.deleteCookie(cookie.name, cookiePath, '.' + fullHostname)
                            this.deleteCookie(cookie.name, cookiePath, dottedDomain)
                        }
                    }
                }

            }
        }
    }

    // src/consent-manager.js -> updateServiceElements
    // update data-consent elements with data-src and data-href ... and contextual consent notice
    updateServiceElements(service, consent) {
        const elements = document.querySelectorAll("[data-consent='" + service.name + "']");
        //console.log('updateServiceElements for ' + service.name + '('+consent+')');

        // we make sure we execute this service only once if the option is set
        if (consent) {
            if (service.onlyOnce && this.executedOnce[service.name])
                return
            this.executedOnce[service.name] = true
        }

        for (let i = 0; i < elements.length; i++) {
            const element = elements[i];
            const parent = element.parentElement
            const { type, src, href } = element.dataset;
            const attrs = ['href', 'src', 'type'];
            if (type === 'placeholder') {
                if (consent) {
                    if (!element.dataset['originalDisplay']) {
                        element.dataset['originalDisplay'] = element.style.display;
                    }
                    element.style.display = 'none';
                }
                else {
                    element.style.display = element.dataset['originalDisplay'] || 'flex';
                }
                continue
            }

            if (element.tagName === 'IFRAME') {
                // this element is already active, we do not touch it...
                if (consent && element.src === src) {
                    // eslint-disable-next-line no-console
                    console.debug(`Skipping ${element.tagName} for service ${service.name}, as it already has the correct type...`)
                    continue
                }
                // we create a new script instead of updating the node in
                // place, as the script won't start correctly otherwise
                const newElement = document.createElement(element.tagName)
                for (const attribute of element.attributes) {
                    newElement.setAttribute(attribute.name, attribute.value)
                }
                newElement.innerText = element.innerText
                newElement.text = element.text

                if (consent) {
                    if (element.dataset['originalDisplay'] !== undefined)
                        newElement.style.display = element.dataset['originalDisplay'];
                    if (element.dataset.src !== undefined) {
                        newElement.src = element.dataset.src;
                    }
                } else {
                    newElement.src = ''
                    if (element.dataset['modified-consent-manager'] !== undefined && element.dataset['originalDisplay'] !== undefined) // this is already a placeholder
                        newElement.setAttribute('data-original-display', element.dataset['originalDisplay'])
                    else {// this is a new element we haven't touched before
                        if (element.style.display !== undefined)
                            newElement.setAttribute('data-original-display', element.style.display)
                        newElement.setAttribute('data-modified-consent-manager', '1')
                    }
                    newElement.style.display = 'none'
                }
                //we remove the original element and insert a new one
                parent.insertBefore(newElement, element)
                parent.removeChild(element)
            } else if (element.tagName === 'SCRIPT' || element.tagName === 'LINK') {
                // this element is already active, we do not touch it...
                if (consent && element.type === (type || "") && (element.src || "") === (src || "")) {
                    // eslint-disable-next-line no-console
                    console.debug(`Skipping ${element.tagName} for service ${service.name}, as it already has the correct type and src...`)
                    continue
                }
                // we create a new script instead of updating the node in
                // place, as the script won't start correctly otherwise
                const newElement = document.createElement(element.tagName)
                for (const attribute of element.attributes) {
                    newElement.setAttribute(attribute.name, attribute.value)
                }

                newElement.innerText = element.innerText
                newElement.text = element.text
		if (element.hasAttribute('async')) {
		    newElement.async = true;
		}

                if (consent) {
                    newElement.type = type || ""
                    if (src !== undefined)
                        newElement.src = src
                    if (href !== undefined)
                        newElement.href = href
                } else {
                    newElement.type = 'text/plain'
                }
                //we remove the original element and insert a new one
                parent.insertBefore(newElement, element)
                parent.removeChild(element)
            } else {
                // all other elements (images etc.) are modified in place...
                if (consent) {
                    for (const attr of attrs) {
                        const attrValue = element.dataset[attr]
                        const capitalizedAttr = attr.charAt(0).toUpperCase() + attr.slice(1);
                        if (attrValue === undefined)
                            continue
                        if (element.dataset['original' + capitalizedAttr] === undefined)
                            element.dataset['original' + capitalizedAttr] = element[attr]
                        element[attr] = attrValue
                    }
                    if (element.dataset.title !== undefined)
                        element.title = element.dataset.title
                    if (element.dataset['originalDisplay'] !== undefined) {
                        element.style.display = element.dataset['originalDisplay']
                    } else {
                        element.style.removeProperty('display')
                    }
                }
                else {
                    if (element.dataset.title !== undefined)
                        element.removeAttribute('title')
                    if (element.dataset['originalDisplay'] === undefined && element.style.display !== undefined)
                        element.dataset['originalDisplay'] = element.style.display
                    element.style.display = 'none'
                    for (const attr of attrs) {
                        const attrValue = element.dataset[attr]
                        const capitalizedAttr = attr.charAt(0).toUpperCase() + attr.slice(1);
                        if (attrValue === undefined)
                            continue
                        if (element.dataset['original' + capitalizedAttr] !== undefined)
                            element[attr] = element.dataset['original' + capitalizedAttr]
                        else
                            element.removeAttribute(attr)
                    }
                }
            }

        }
    }



    _checkConsents() {
        let complete = true
        const services = new Set(this.services.map((service) => { return service.name }))
        const consents = new Set(Object.keys(this.consents))
        // delete consent without service
        for (const key of Object.keys(this.consents)) {
            if (!services.has(key)) {
                delete this.consents[key]
            }
        }
        // set default
        for (const service of this.services) {
            if (!consents.has(service.name)) {
                this.consents[service.name] = this.getDefaultConsent(service)
                complete = false
            }
        }
        //this.confirmed = complete
        this.confirmed = true;
        if (!complete) {
            this.changed = true;
        }
    }
    updateServericeElement(name, content) {
        let service = this.getService(name);
        return this.consents[name] || false;
    }
    dispatch(name, handlerOpts = {}) {
        if (this.config.hasOwnProperty(name)) {
            handlerOpts.manager = this;
            this.executeHandler(this.config[name], handlerOpts);
        }
    }
}
const igConsentManager = new IgConsentManager();
// add IgConsentManager to window object
window.IgConsentManager = igConsentManager;
if (typeof window.igJs !== 'undefined') {
    window.igJs.add('IgConsentManager', igConsentManager);
}