<?php

namespace Internetgalerie\IgCrmAdmin\Controller;

use Exception;
use Internetgalerie\IgCrmAdmin\Utility\CompatibilityUtility;
use Internetgalerie\IgDatapoolFe\Controller\ActionController;
use Internetgalerie\IgsCrm\Domain\Model\Kurse;
use Internetgalerie\IgsCrm\Domain\Repository\KurseRepository;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Pagination\QueryResultPaginator;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

class KurseController extends ActionController
{
    protected $objectClass = Kurse::class;
    protected $kurseRepository = null;
    /**
     * Parameter name of the object we are manipulating.
     * Used for forms with steps
     */
    protected $objectParamName = 'kurse';

    public function injectKurseRepository(KurseRepository $kurseRepository): void
    {
        $this->kurseRepository = $kurseRepository;
    }

    /**
     * List Action
     * Renders a list of all Records
     * Useful repository methods:
     * $kurseRepository->findAll(); - all records, order as in DB
     * $kurseRepository->findAllWithSearch() - automagically search all properties. use searchform viewhelper, input name = property, respects sortheaders
     * ...write your own method for custom search, see repository class fo more!
     */
    public function listAction(): ResponseInterface
    {
        // Default search: if a formfield parses attribute (if the property exists and the formfield is not empty).
        // search vars inside extensionprefix[@search][formfield] = value
        $kurses = $this->kurseRepository->findBySearch();
        $currentPage = $this->request->hasArgument('currentPage') ? (int)$this->request->getArgument('currentPage') : 1;
        $paginator = new QueryResultPaginator($kurses, $currentPage, 10);
        $pagination = CompatibilityUtility::createPagination($paginator, 10); // zweiter Argument: maximal Anzahl Links
        $this->view->assign('kurses', $kurses);
        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);
        return $this->htmlResponse($this->view->render());
    }

    public function showAction(Kurse $kurse): ResponseInterface
    {
        // set page title (<title>-tag, page title output and search index), use your own methods.
        $GLOBALS['TSFE']->page['title'] = $GLOBALS['TSFE']->indexedDocTitle = (string) $kurse;

        $this->view->assign('kurse', $kurse);
        return $this->htmlResponse($this->view->render());
    }

    public function newAction(Kurse $kurse = null): ResponseInterface
    {
        //1. Access check: Am I allowed to create Records?
        //$this->access->loginUser();
        $kursArten = $this->kurseRepository->findAll();
        $this->view->assign('kursArten', $kursArten);
        $this->view->assign('kurse', $kurse);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * Initialize the Update Action to override Labels
     * This can be done globally by using  initializeAction(), for one specific Action (initializeSomeAction())
     * or only for a specific field (3rd param, useful for validators).
     *
     *  REMOVE _ TO ACTIVATE!
     */
    public function _initializeCreateAction(): ResponseInterface
    {
        $this->setLabel('validator.notempty.empty', 'my_custom_label_in_extension_xlf');
        $this->setLabel('validator.email.notvalid', 'Bitte tragen Sie Ihre E-Mail Adresse ein!', 'emailfeld');
        $this->setLabel('tx_myextension_domain_model_table.email', 'E-Mail / Benutzername');
        return $this->htmlResponse();
    }

    /**
     * action create
     */
    public function createAction(Kurse $kurse): ResponseInterface
    {
        // 1. Access check: Am I allowed to create Records?
        // $this->access->loginUser();

        // 2. You might need to set additional properties. E.g, the current frontend user.
        // $kurse -> setFrontendUser( $this->getCurrentFrontendUser()->getUid() );
        // $kurse -> setSomeThing ( 'someValue' );

        // 3. Display a message "Your record has been created."
        $this->addFlashMessage(
            LocalizationUtility::translate('global.created', 'ig_render'),
            '',
            ContextualFeedbackSeverity::OK
        );

        // 4. Add to repository
        // we need to do that first and actually add it to the dbif we want to generate an activate link later.
        $this->kurseRepository->add($kurse);
        // we need to make it persistent, for getting an uid
        //$persistenceManager = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance('TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager') -> persistAll();

        // 6. Redirect the user to the list action
        return $this->redirectWithSearch('list');
    }

    public function editAction(Kurse $kurse): ResponseInterface
    {
        // 1. Access check: Am I allowed to create Records?
        // $this->access->isRecordOwner( $kurse->getFrontendUser() );
        $kursArten = $this->kurseRepository->findAll();
        $this->view->assign('kursArten', $kursArten);
        $this->view->assign('kurse', $kurse);
        return $this->htmlResponse($this->view->render());
    }

    public function updateAction(Kurse $kurse): ResponseInterface
    {
        // 1. Access check: Am I allowed to update the Records. normally i have to be the owner! pass a FEUser Object or a UID to isRecordOwner
        // $this->access->isRecordOwner($kurse->getFrontendUser());

        // 2. display a Message "Your record has been updated."
        $this->addFlashMessage(
            LocalizationUtility::translate('global.updated', 'ig_render'),
            '',
            ContextualFeedbackSeverity::OK
        );

        // 3. update in the repository
        $this->kurseRepository->update($kurse);

        // 4. redirect the user to the list
        return $this->redirectWithSearch('list');
    }

    public function confirmDeleteAction(Kurse $kurse): ResponseInterface
    {
        // 1. Access check: Am I allowed to delete this record?
        // $this->access->isRecordOwner(##OBJECT###->getFrontendUser());

        $this->view->assign('objectLabel', '<strong>&laquo;' . $kurse->__toString() . '&raquo;</strong>');
        $this->view->assign('kurse', $kurse);
        return $this->htmlResponse($this->view->render());
    }

    public function deleteAction(Kurse $kurse): ResponseInterface
    {
        // 1. Access check: Am I allowed to update the Records. normally i have to be the owner! pass a FEUser Object or a UID to isRecordOwner
        // $this->access->isRecordOwner($object->getFrontendUser());

        // 1. display a Message "Your record has been deleted."
        $this->addFlashMessage(
            LocalizationUtility::translate('global.deleted', 'ig_render'),
            '',
            ContextualFeedbackSeverity::OK
        );

        // 2. remove it from the repository
        $this->kurseRepository->remove($kurse);

        // 3. redirect the user
        return $this->redirectWithSearch('list');
    }

    /**
     * Activate Action
     * Activate a record, this is used as a callback from an e-mail.
     * The key is the records UID with an HMAC appended
     */
    public function activateRecordAction(Kurse $kurse, $key): ResponseInterface
    {
        // 1. Access check: Is my key correct?
        $this->access->checkKey($kurse->getUid(), $key);

        throw new Exception(self::class . '->activateRecordAction: Implement this!');

        // 2. check if it is already activated
        if ($kurse->isActive()) {
            $this->addFlashMessage('label.alreadyActive', '', ContextualFeedbackSeverity::ERROR);
            return $this->redirectWithSearch('list');
        }

        // 3. activate it
        $kurse->setActive(true);

        // 2. display a Message "The Record was approved and set online."
        $this->addFlashMessage('label.recordApproved', ContextualFeedbackSeverity::OK);

        // 3. update in the repository
        $this->kurseRepository->update($kurse);

        // 4. redirect the user to the list
        return $this->redirectWithSearch('list');
    }

    public function exportAction($type = null): ResponseInterface
    {
        $kurses = $this->kurseRepository->findBySearch();

        if ($type === null) {
            $this->htmlResponse('');
        }

        $exporter = GeneralUtility::makeInstance(
            '\\Internetgalerie\\IgDatapoolFe\\Services\\Exporter\\' . ucfirst(
                strtolower((string) $type)
            ) . 'ExporterService',
            $type
        );

        $fieldsWithLabels = [];
        $options = [
            'dbLabel' => true,
        ];
        $exporter->setOptions($fieldsWithLabels, $options);

        $fieldOptions = [];
        $fieldOptions['geburtsdatum']['format'] = 'd.m.Y';
        $exporter->setFieldOptions($fieldOptions);

        // DO IT!
        $exporter->export($kurses);
        exit;
        return $this->htmlResponse($this->view->render());
    }
}
