<?php

declare(strict_types=1);

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace Internetgalerie\IgCrmAdmin\Controller;

use Internetgalerie\IgCrmWeb\Rest\Service\AuthRestService;
use Internetgalerie\IgCrmWeb\Utility\UserUtility;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Configuration\ExtensionConfiguration;
use TYPO3\CMS\Core\Crypto\PasswordHashing\PasswordHashFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;

/**
 * @internal this is a concrete TYPO3 implementation and solely used for EXT:felogin and not part of TYPO3's Core API.
 */
class PasswordRecoveryController extends \TYPO3\CMS\FrontendLogin\Controller\PasswordRecoveryController
{
    private $config = null;
    private $userUtility = null;

    public function initializeAction(): void
    {
        parent::initializeAction();
        $this->config = GeneralUtility::makeInstance(ExtensionConfiguration::class)->get('ig_crm_web');
        $this->authRestService = GeneralUtility::makeInstance(AuthRestService::class);
        $this->userUtility = GeneralUtility::makeInstance(UserUtility::class, $this->config);
    }
    /**
     * Shows the recovery form. If $userIdentifier is set an email will be sent, if the corresponding user exists
     */
    public function recoveryAction(string $userIdentifier = null): ResponseInterface
    {
        if (empty($userIdentifier)) {
            return $this->htmlResponse('');
        }
        // First, check if user exists with email address on external server
        $user = $this->authRestService->getUserByEmail($userIdentifier);

        if ($user) {
            $localEmail = $this->userRepository->findEmailByUsernameOrEmailOnPages(
                $userIdentifier,
                $this->getStorageFolders()
            );

            if (!$localEmail) {
                $this->userUtility->createLocalUser($user['username'], $user, $this->userUtility->getGroups($user));
            }
            // TODO: Check if user locally exists
            // If user locally exists, send PW Recovery Mail
            // If user doesn't exist locally, create like in ExternalAuthService and then send PW Recovery Mail
            return parent::recoveryAction($userIdentifier);
        }
        return $this->htmlResponse('');
    }

    /**
     * Change actual password. Hash $newPass and update the user with the corresponding $hash.
     */
    public function changePasswordAction(string $newPass, string $hash): ResponseInterface
    {
        $user = $this->userRepository->findOneByForgotPasswordHash(GeneralUtility::hmac($hash));
        $hashedPassword = GeneralUtility::makeInstance(PasswordHashFactory::class)
            ->getDefaultHashInstance('FE')
            ->getHashedPassword($newPass);

        $hashedPassword = $this->notifyPasswordChange($newPass, $hashedPassword, $hash);
        $this->userRepository->updatePasswordAndInvalidateHash(GeneralUtility::hmac($hash), $hashedPassword);
        $this->authRestService->updatePassword($user, $newPass);

        $this->addFlashMessage($this->getTranslation('change_password_done_message'));

        return $this->redirect('login', 'Login', 'felogin');
    }
}
