<?php

namespace Internetgalerie\IgCrmAdmin\Controller;

use Internetgalerie\IgsCrm\Domain\Model\Contact;
use Internetgalerie\IgsCrm\Domain\Model\FrontendUser;
use Internetgalerie\IgsCrm\Domain\Model\Person;
use Internetgalerie\IgsCrm\Domain\Repository\ContactTagRepository;
use Internetgalerie\IgsCrm\Domain\Repository\FrontendUserRepository;
use Internetgalerie\IgsCrm\Domain\Repository\PersonRepository;
use Internetgalerie\IgsCrm\Domain\Repository\TagFieldRepository;
use Internetgalerie\IgsCrm\Domain\Repository\TagRoleRepository;
use Internetgalerie\IgsCrm\Domain\Repository\VerbandRepository;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Crypto\PasswordHashing\PasswordHashFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Extbase\Property\PropertyMapper;
use TYPO3\CMS\Extbase\Property\PropertyMappingConfigurationBuilder;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3\CMS\Form\Mvc\Configuration\ConfigurationManager;
use OpenApi\Attributes as OA;

#[OA\Info(
    version: "1.0.0",
    title: "Person API",
    description: "API documentation for managing persons"
)]
#[OA\Server(url: "https://crm2.beta-version.ch", description: "Production server")]
            
class PersonApiController extends AbstractApiController
{
    /**
     * personRepository
     *
     * @var PersonRepository
     */
    protected $personRepository = null;

    /**
     * contactTagRepository
     *
     * @var ContactTagRepository
     */
    protected $contactTagRepository = null;

    /**
     * frontendUserRepository
     *
     * @var FrontendUserRepository
     */
    protected $frontendUserRepository = null;

    /**
     * verbandRepository
     *
     * @var VerbandRepository
     */
    protected $verbandRepository = null;

    public function injectPersonRepository(PersonRepository $personRepository): void
    {
        $this->personRepository = $personRepository;
    }

    public function injectContactTagRepository(ContactTagRepository $contactTagRepository): void
    {
        $this->contactTagRepository = $contactTagRepository;
    }

    public function injectFrontendUserRepository(FrontendUserRepository $frontendUserRepository): void
    {
        $this->frontendUserRepository = $frontendUserRepository;
    }

    public function injectVerbandRepository(VerbandRepository $verbandRepository): void
    {
        $this->verbandRepository = $verbandRepository;
    }

    /**
     * action list
     */
    #[OA\Get(
        path: '/api/persons',
        summary: 'Fetch list of persons',
        description: 'Returns a list of persons based on search criteria.'
    )]
    #[OA\Parameter(
        name: 'apiKey',
        in: 'query',
        description: 'API key for authentication',
        required: true,
        schema: new OA\Schema(type: 'string')
    )]
    #[OA\Response(
        response: 200,
        description: 'List of persons',
        content: new OA\JsonContent(
            type: 'object',
            properties: [
                new OA\Property(property: 'result', type: 'array', items: new OA\Items(type: 'object'))
            ]
        )
    )]
    #[OA\Response(
        response: 401,
        description: 'Unauthorized',
        content: new OA\JsonContent(
            type: 'object',
            properties: [new OA\Property(property: 'error', type: 'string')]
        )
    )]
    public function listAction(): ResponseInterface
    {
        $search = $this->getSearch() ?? [];
        $token = $this->request->hasArgument('apiKey') ? $this->request->getArgument('apiKey') : '';
        $this->personRepository->setToken($token);
        $persons = $this->personRepository->findBySearchDoctrine($search ?? []);
        $persons = $this->arrayUnderscoredToLowerCamelCase($persons);
        return $this->jsonResponse((string)json_encode([
                    'result' => $persons,
                ]));
    }

    /**
     * action show
     */
    public function showAction(): ResponseInterface
    {
        return $this->jsonResponse((string)json_encode([
                'show' => true,
            ]));
    }
    /**
     * action tagrole
     */
    public function tagroleAction(): ResponseInterface
    {
        $search = $this->getSearch() ?? [];
        $token = $this->request->hasArgument('apiKey') ? $this->request->getArgument('apiKey') : '';
        $this->personRepository->setToken($token);
        //$persons = $this->personRepository->findBySearch($search);
        $verbandUid = null; // @todo aus $token
        $tagUid = $search['tags'][0] ?? null;
        $persons = $this->personRepository->personRoleFindByVerbandRaw($verbandUid, $tagUid, $search);
        // Headers
        $headers = [];
        $headers[] = LocalizationUtility::translate('Name', 'igCrmWeb', null, $this->language->getLocale()->getLanguageCode());
        $headers[] = LocalizationUtility::translate(
            'function',
            'igCrmWeb',
            null,
            $this->language->getLocale()->getLanguageCode()
        );
        $tagId = $search['tags'][0] ?? null;
        $classes = ['td-name', 'td-tag-role'];
        $types = ['text', 'text'];
        if ($tagId) {
            //$tag = $this->tagRepository->findByUid($tagId);
            $tagFieldRepository = GeneralUtility::makeInstance(TagFieldRepository::class);
            //$textfields = $tagFieldRepository->findByTag($tagId);
            $textfields = $tagFieldRepository->findRawByTagAndLanguage($tagId, $this->language->getLanguageId());
            if ($textfields) {
                foreach ($textfields as $textfield) {
                    $headers[] = $textfield['title'];
                    $classes[] = 'td-person-tag';
                    $types[] = 'text';
                }
            }
        }
        //$headers[] = LocalizationUtility::translate('LLL:EXT:tt_address/Resources/Private/Language/locallang_db:tt_address_palette.contact', null, null, $this->language->getLocale()->getLanguageCode());
        $headers[] = LocalizationUtility::translate(
            'contact',
            'igCrmWeb',
            null,
            $this->language->getLocale()->getLanguageCode()
        );
        $classes[] = 'td-contact';
        $types[] = 'email';
        //('LLL:EXT:tt_address/Resources/Private/Language/locallang:label.email');

        $rows = [];
        foreach ($persons as $person) {
            $row = [];
            $contactTag = $this->contactTagRepository->findOneByContactTag($person['uid'], $tagId, true);
            //$tagRole = $contactTag ? $contactTag->getTagRole() : null;
            if (!empty($contactTag)) {
                $this->tagRoleRepository = GeneralUtility::makeInstance(TagRoleRepository::class);
                $tagRole = $this->tagRoleRepository->findRawByUidAndLanguage(
                    $contactTag['tagrole'],
                    $this->language->getLanguageId()
                );
                if ($this->language->getLanguageId() > 0) {
                    $tagRoleDefaultLangauge = $this->tagRoleRepository->findRawByUidAndLanguage(
                        $contactTag['tagrole'],
                        0
                    );
                } else {
                    $tagRoleDefaultLangauge = $tagRole;
                }
            } else {
                $tagRole = [];
                $tagRoleDefaultLangauge = $tagRole;
            }
            //var_dump($this->language->getLanguageId(),$tag,$contactTag['tagrole']); exit(0);
            $row[] = $person['me_firstname'] . ' ' . $person['me_lastname'];
            //$row[] = $person->getName();
            $row[] = $tagRole['name'] ?? '';
            if ($textfields) {
                foreach ($textfields as $textfield) {
                    if ($textfield['is_translated']) {
                        $langAttribute = $this->language->getLocale()->getLanguageCode();
                        $value = $contactTag['textfield' . $textfield['field_number'] . '_' . $langAttribute];
                        // language Fallback
                        if ($value == '') {
                            $value = $contactTag['textfield' . $textfield['field_number'] . '_de'];
                        }
                        /*
                        if ($value!='-') {
                            var_dump($contactTag,$contactTag['textfield' . $textfield['field_number'] . '_' . $langAttribute], $this->language->getLanguageId(), $langAttribute,$textfield['field_number'] );
                            die('code=' . $this->language->getLocale()->getLanguageCode() . ' mit wert=' . $value );
                        }
                        */
                    } else {
                        $value = $contactTag['textfield' . $textfield['field_number'] . '_de'];
                    }
                    $row[] = $value;
                }
            }
            $row[] = $tagRoleDefaultLangauge['show_mail'] ? $person['me_email'] : '';
            //$row[] = $tagRole && $tagRole->getShowMail() ? $person->getEmail() : '';
            $rows[] = $row;
        }

        $this->view->setVariablesToRender(['json']);
        $this->view->assignMultiple([
            'json' => [
                'result' => [
                    'headers' => $headers,
                    'rows' => $rows,
                    'classes' => $classes,
                    'types' => $types,
                ],
            ],
        ]);
        return $this->jsonResponse($this->view->render());
    }

    // Create fe_user-Entry
    public function createAction(): ResponseInterface
    {
        if (!$this->verband) {
            $error = true;
        }

        if (!$error) {
            // Storage PID herauslesen und an Repository zuweisen
            $configuration = $this->configurationManager->getConfiguration(
                ConfigurationManager::CONFIGURATION_TYPE_FRAMEWORK,
                'IgCrmAdmin'
            );
            $token = $this->request->hasArgument('apiKey') ? $this->request->getArgument('apiKey') : '';
            $useAcl = $this->personRepository->getUseAcl(); // useAcl is set false in setToken
            $this->personRepository->setToken($token);

            $storagePid = (int)$configuration['persistence']['classes'][Contact::class]['newRecordStoragePid'];
            if ($storagePid) {
                $this->personRepository->setStoragePids([$storagePid]);
            }
            $storagePid = (int)$configuration['persistence']['classes'][FrontendUser::class]['newRecordStoragePid'];
            if ($storagePid) {
                $this->frontendUserRepository->setStoragePageId($storagePid);
            }
            // Objekt generieren
            $personArray = $this->get['person'];

            $feUserArray = $personArray['frontendUser'];
            unset($personArray['frontendUser']);
            $hashInstance = GeneralUtility::makeInstance(PasswordHashFactory::class)->getDefaultHashInstance('FE');
            $hashedPassword = $hashInstance->getHashedPassword($feUserArray['password']);
            $feUserArray['password'] = $hashedPassword;

            $type = $personArray['type'] ?? 'Person';
            unset($personArray['type']);

            // TODO: Prüfen ob fe_user bereits existiert
            $propertyMappingConfigurationBuilder = GeneralUtility::makeInstance(
                PropertyMappingConfigurationBuilder::class
            );
            $mappingConfiguration = $propertyMappingConfigurationBuilder->build();
            $mappingConfiguration->allowProperties('frontendUser');
            $propertyMapper = GeneralUtility::makeInstance(PropertyMapper::class);
            $feUser = $propertyMapper->convert($feUserArray, FrontendUser::class);

            /*
            $obj = GeneralUtility::makeInstance(Person::class);
            foreach($personArray as $property => $value) {
                $obj->_setProperty($property, $value);
            }
            */
            $propertyMappingConfigurationBuilder = GeneralUtility::makeInstance(
                PropertyMappingConfigurationBuilder::class
            );
            $mappingConfiguration = $propertyMappingConfigurationBuilder->build();
            $mappingConfiguration->allowProperties('person');
            $propertyMapper = GeneralUtility::makeInstance(PropertyMapper::class);
            $person = $propertyMapper->convert($personArray, Person::class);
            // set hash for links/approval
            $hash = base64_encode(random_bytes(48));
            $person->setHash($hash);

            // @todo should be configurable
            $person->setHidden(true);
            $person->setMustReviewed(true);

            // set acl/permissions according verband
            if ($this->verband && !is_object($this->verband)) {
                $this->verband = $this->verbandRepository->findByUid($this->verband);
            }
            if (!$this->verband) {
                $this->verbandRepository = GeneralUtility::makeInstance(VerbandRepository::class);
                $this->verband = $this->verbandRepository->findDefault();
            }
            if ($this->verband) {
                $person->setAclOwner($this->verband->getAclOwner());
                $person->setAclWriteGroups($this->verband->getAclWriteGroups());
                $person->setAclReadGroups($this->verband->getAclReadGroups());
            } else {
                if ($useAcl) {
                    $error = [
                        'message' => 'no api key or default client set, but acl is used',
                    ];
                }
            }

            //var_dump($person);exit(0);
            if (!$error) {
                // Objekt in DB speichern
                if ($type == 'Person') {
                    $this->frontendUserRepository->add($feUser);
                    GeneralUtility::makeInstance(PersistenceManager::class)->persistAll();
                    $person->setFrontendUser($feUser);
                    $this->personRepository->add($person);
                } elseif ($type == 'Organisation') {
                    $error = [
                        'message' => 'Organisation not yet implemented',
                    ];
                    //$this->organisationRepository->add($person);
                }
                GeneralUtility::makeInstance(PersistenceManager::class)->persistAll();
            }


            /*
            $hash = hash('sha1', time() . $GLOBALS['TYPO3_CONF_VARS']['SYS']['encryptionKey'] . $person->getUid());
            $person->setHash($hash);
            $this->personRepository->update($person);
            */
        }

        $this->view->setVariablesToRender(['json']);
        if ($error) {
            $this->view->assign('json', [
                'error' => $error,
            ]);
        } else {
            $this->view->assign('json', [
                'person' => $person,
            ]);
        }
        return $this->jsonResponse($this->view->render());
        /*
        // UID und PID von Objekt holen
        $personArray['uid'] = $person->getUid();
        $personArray['pid'] = $person->getPid();
        $personArray['hash'] = $hash;
        return $this->jsonResponse(
            (string)json_encode($personArray)
        );
        */
    }

    // TODO: Hash
    public function approveAction(): ResponseInterface
    {
        $hash = $this->get['hash'];
        $personUid = $this->get['person'];

        $token = $this->request->hasArgument('apiKey') ? $this->request->getArgument('apiKey') : '';
        $this->personRepository->setToken($token);

        $person = $this->personRepository->findHiddenByUid($personUid);

        $error = true;
        if ($person && $person->getHash() == $hash) {
            $error = false;
            $person->setHidden(false);
            $person->setMustReviewed(false);
            $person->setHash('');
            $this->personRepository->update($person);
            $feUser = $person->getFrontendUser();

            foreach ($person->getAclReadGroups() as $group) {
                if (!$feUser->getUsergroup()->contains($group)) {
                    $feUser->addUsergroup($group);
                }
            }
            foreach ($person->getAclWriteGroups() as $group) {
                if (!$feUser->getUsergroup()->contains($group)) {
                    $feUser->addUsergroup($group);
                }
            }

            $this->frontendUserRepository->update($feUser);
            GeneralUtility::makeInstance(PersistenceManager::class)->persistAll();
        }

        $this->view->setVariablesToRender(['json']);
        if ($error) {
            $this->view->assign('json', [
                'error' => $error,
            ]);
        } else {
            $this->view->assign('json', [
                'person' => $person,
            ]);
        }
        return $this->jsonResponse($this->view->render());
    }

    // TODO: Hash
    public function unapproveAction(): ResponseInterface
    {
        $hash = $this->get['hash'];
        $personUid = $this->get['person'];

        $token = $this->request->hasArgument('apiKey') ? $this->request->getArgument('apiKey') : '';
        $this->personRepository->setToken($token);

        $person = $this->personRepository->findByUid($personUid);

        $status = 'error';
        if ($person && $person->getHash() == $hash) {
            $status = 'success';
            $person->setHash('');
            $this->personRepository->remove($person);
            GeneralUtility::makeInstance(PersistenceManager::class)->persistAll();
        }
        return $this->jsonResponse((string)json_encode([
                    'status' => $status,
                ]));
    }
}
