<?php

namespace Internetgalerie\IgCrmTemplate\Domain\Repository;

use Internetgalerie\IgCrmTemplate\Database\Query\TemplateLetterQueryBuilder;
use Internetgalerie\IgCrmTemplate\Domain\Model\AddressInterface;
use Internetgalerie\IgCrmTemplate\Domain\Model\TemplateObjectInterface;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;
use TYPO3\CMS\Extbase\Persistence\Generic\Typo3QuerySettings;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use TYPO3\CMS\Extbase\Persistence\Repository;

/**
 * The repository for TemplateLetter
 */
class TemplateLetterRepository extends Repository
{
    protected $tablename = 'tx_igcrmtemplate_domain_model_templateletter';

    protected $defaultOrderings = [
        'templateType' => QueryInterface::ORDER_ASCENDING,
        'name' => QueryInterface::ORDER_ASCENDING,
    ];
    /**
     * @var DataMapper
     **/
    protected $dataMapper;

    
    public function injectDataMapper(DataMapper $dataMapper): void
    {
        $this->dataMapper = $dataMapper;
    }

    public function initializeObject(): void
    {
        $querySettings = GeneralUtility::makeInstance(Typo3QuerySettings::class);
        $querySettings->setRespectStoragePage(false);
        $this->setDefaultQuerySettings($querySettings);
    }

    public function createTemplateLetterQueryBuilder()
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable($this->tablename);
        $queryBuilder = GeneralUtility::makeInstance(TemplateLetterQueryBuilder::class, $conn);
        $queryBuilder->joinTemplatetype();
        return $queryBuilder;
    }
    /**
     * @param tenant $tenantUid UID (Verband)
     * @param name $templateType of templateType table (e.g. Mail)
     * @param class $templateObjectType type of template object (e.g. Invoice)
     * @param sub $subTag tag (e.g. status of invoice) - meaning is depending from templateType/templateObjectType
     * @param class $contactType type of contact object
     * @param language $crmLanguageUid uid
     * @param main $mainTag tag (e.g. mitgliedschaft) - meaning is depending from templateType/templateObjectType
     */
    public function findTemplate(
        ?int $tenantUid,
        string $templateType,
        string $templateObjectType = null,
        int $subTag = null,
        string $contactType = null,
        int $crmLanguageUid = null,
        int $mainTag = null
    ) {
        $queryBuilder = $this->createTemplateLetterQueryBuilder();
        $queryBuilder->select($this->tablename . '.*')->from($this->tablename);
        $queryBuilder->addTemplateTypeByName($templateType);
        $constraints = [
            $queryBuilder->expr()
                         ->eq('active', 1),
            //$queryBuilder->expr()->eq('template_type', $templateType),
        ];
        if ($tenantUid !== null) {
            $constraints[] = $queryBuilder->expr()->eq('tenant_id', (int)$tenantUid);
        }
        if ($contactType !== null && $contactType !== '') {
            $constraints[] = $queryBuilder->expr()->or(
                $queryBuilder->expr()
                             ->eq('object_type', $queryBuilder->createNamedParameter($contactType)),
                $queryBuilder->expr()
                             ->eq('object_type', $queryBuilder->createNamedParameter('')),
                $queryBuilder->expr()
                             ->isNull('object_type')
            );
        }
        /*
          if ($templateObjectType !== null) {
          if ($templateObjectType === '') {
          $constraints[] = $queryBuilder->expr()->or(
          $queryBuilder->expr()->eq('template_object_type', $queryBuilder->createNamedParameter('')),
          $queryBuilder->expr()->eq('template_object_type', 0),
          $queryBuilder->expr()->isNull('template_object_type')
          );
          } else {
          $constraints[] = $queryBuilder->expr()->eq('template_object_type', $queryBuilder->createNamedParameter($templateObjectType));
          }
          }
        */
        if ($crmLanguageUid !== null) {
            $constraints[] = $queryBuilder->expr()->eq('crm_language', (int) $crmLanguageUid);
            /*
              $constraints[] = $query->logicalOr (
              $query->equals('crmLanguage', $crmLanguageUid),
              $query->equals('crmLanguage', null)
              );
            */
        } else {
            $constraints[] = $queryBuilder->expr()->or(
                $queryBuilder->expr()
                             ->isNull('crm_language'),
                $queryBuilder->expr()
                             ->eq('crm_language', 0)
            );
            /*
              $constraints[] = $query->logicalOr (
              $query->equals('crmLanguage', 0)
              );
            */
        }
        
        if ($mainTag !== null) {
            if ($mainTag === 0) {
                $constraints[] = $queryBuilder->expr()->or(
                    $queryBuilder->expr()
                                 ->eq('main_tags', $queryBuilder->createNamedParameter('')),
                    $queryBuilder->expr()
                                 ->eq('main_tags', $queryBuilder->createNamedParameter('0')),
                    $queryBuilder->expr()
                                 ->isNull('main_tags'),
                );
            } else {
                $constraints[] = $queryBuilder->expr()->inSet('main_tags', $mainTag);
            }
        }

        if ($subTag !== null) {
            if ($subTag === 0) {
                $constraints[] = $queryBuilder->expr()->or(
                    $queryBuilder->expr()
                                 ->eq('sub_tags', $queryBuilder->createNamedParameter('')),
                    $queryBuilder->expr()
                                 ->eq('sub_tags', $queryBuilder->createNamedParameter('0')),
                    $queryBuilder->expr()
                                 ->isNull('sub_tags'),
                );
            } else {
                $constraints[] = $queryBuilder->expr()->inSet('sub_tags', $subTag);
            }
        }
        //echo('tenant_id=' . $tenantUid . " AND object_type='" . $contactType . "' AND (crm_language=" .  $crmLanguageUid .' OR crm_language IS NULL)' );
        $queryBuilder->andWhere($queryBuilder->expr()->and(...$constraints));


        // null to end, default to near end
        $queryBuilder->addArrayOrderBy([
            'object_type' => 'DESC',
            'crm_language' => 'DESC',
            //'template_object_type' => 'DESC',
            'name' => 'ASC',
            //'sub_tag' => 'DESC',
        ]);
        $rows = $queryBuilder->executeQuery()
                             ->fetchAllAssociative();
        return $this->dataMapper->map($this->objectType, $rows);
    }
    public function findOneTemplate(
        ?int $tenantUid,
        string $templateType,
        string $templateObjectType = null,
        int $subTag = null,
        string $contactType = null,
        int $crmLanguageUid = null,
        int $mainTag = null
    ) {
        $templateLetters = $this->findTemplate(
            $tenantUid,
            $templateType,
            $templateObjectType,
            $subTag,
            $contactType,
            $crmLanguageUid,
            $mainTag
        );
        return !empty($templateLetters) ? $templateLetters[0] : null;
    }

    public function findOneTemplateForLanguages(
        ?int $tenantUid,
        string $templateType,
        $crmLanguageUid,
        string $templateObjectType = null,
        int $subTag = null,
        string $contactType = null,
        int $mainTag = null
    ) {
        $template = $this->findOneTemplate(
            $tenantUid,
            $templateType,
            $templateObjectType,
            $subTag,
            $contactType,
            $crmLanguageUid,
            $mainTag
        );
        if ($template !== null) {
            return $template;
        }
        return $this->findOneTemplate(
            $tenantUid,
            $templateType,
            $templateObjectType,
            $subTag,
            $contactType,
            null,
            $mainTag
        );
    }


    public function findByVerbandTypeLanguage(
        ?int $tenantUid,
        string $templateType,
        TemplateObjectInterface $templateObject = null,
        string $contactType = null,
        int $crmLanguageUid = null,
        int $mainTag = null
    ) {
        if ($templateObject && $templateObject->getType() !== '' && $templateObject->getType() != null) {
            $templateObjectType = $templateObject->getType();
        } else {
            $templateObjectType = null;
        }
        $subTag = $templateObject && $templateObject->getTemplateSubTag() ? $templateObject->getTemplateSubTag() : null;
        $templateLetters = $this->findTemplate(
            $tenantUid,
            $templateType,
            $templateObjectType,
            $subTag,
            $contactType,
            $crmLanguageUid,
            $mainTag
        );
        //var_dump($tenantUid, $templateType, $templateObjectType, $subTag, $contactType, $crmLanguageUid, $mainTag, $templateLetters );exit(0);

        // template found
        if (count($templateLetters) > 0) {
            return $templateLetters;
            //return $templateLetters[0];
        }
        // nothing found
        if ($subTag > 0) {
            // search without subTag
            $templateLetters = $this->findTemplate(
                $tenantUid,
                $templateType,
                $templateObjectType,
                0,
                $contactType,
                $crmLanguageUid,
                $mainTag
            );
            return $templateLetters;
            //return empty($templateLetters) ? null : $templateLetters;
        }
        if ($contactType) {
            // search without status and objectType
            $templateLetters = $this->findTemplate(
                $tenantUid,
                $templateType,
                $templateObjectType,
                0,
                '',
                $crmLanguageUid,
                $mainTag
            );
        }
        // nothing found
        return $templateLetters;
        //return empty($templateLetters) ? null : $templateLetters;
    }
    public function findByVerbandType(
        ?int $tenantUid,
        string $templateType,
        TemplateObjectInterface $templateObject = null,
        string $contactType = null
    ) {
        return $this->findByVerbandTypeLanguage($tenantUid, $templateType, $templateObject, $contactType, null);
    }
        

    public function findOneByVerbandTypeLanguage(
        ?int $tenantUid,
        string $templateType,
        TemplateObjectInterface $templateObject = null,
        string $contactType = null,
        int $crmLanguageUid = null
    ) {
        $templateLetters = $this->findByVerbandTypeLanguage(
            $tenantUid,
            $templateType,
            $templateObject,
            $contactType,
            $crmLanguageUid
        );
        return !empty($templateLetters) ? $templateLetters[0] : null;
    }
    
    public function findOneByVerbandTypeLanguageMainTag(
        ?int $tenantUid,
        string $templateType,
        int $mainTag,
        TemplateObjectInterface $templateObject = null,
        string $contactType = null,
        int $crmLanguageUid = null
    ) {
        $templateLetters = $this->findByVerbandTypeLanguage(
            $tenantUid,
            $templateType,
            $templateObject,
            $contactType,
            $crmLanguageUid,
            $mainTag
        );
        return null;
        if (!empty($templateLetters)) {
            return $templateLetters;
        }
        $templateLetters = $this->findByVerbandTypeLanguage(
            $tenantUid,
            $templateType,
            $templateObject,
            $contactType,
            $crmLanguageUid
        );
        return $templateLetters;
    }

    public function findByTenantUidTypeWithTemplateObject(
        ?int $tenantUid,
        string $templateType = null,
        TemplateObjectInterface $templateObject = null,
        AddressInterface $contact = null,
        ?int $crmLanguageUid = null
    ) {
        if ($templateObject !== null) {
            $mainTag = $templateObject->getTemplateMainTag(); // mitgliedschaft
            $subTag = $templateObject->getTemplateSubTag();
        } else {
            $mainTag = null;
            $subTag = null;
        }
        if ($contact !== null) {
            $contactType = $contact->getType();
        } else {
            $contactType = null;
        }
        //var_dump($tenantUid, $templateType, $mainTag , $subTag, $contactType, $crmLanguageUid);exit(0);
        $templateObjectType = $templateObject !== null ? $templateObject->getType() : null;
        
        if ($mainTag && $subTag) {
            $template = $this->findOneTemplateForLanguages(
                $tenantUid,
                $templateType,
                $crmLanguageUid,
                $templateObjectType,
                $subTag,
                $contactType,
                $mainTag
            );
            if ($template !== null) {
                return $template;
            }
        }
        // e.g. membership
        if ($mainTag) {
            $template = $this->findOneTemplateForLanguages(
                $tenantUid,
                $templateType,
                $crmLanguageUid,
                $templateObjectType,
                0,
                $contactType,
                $mainTag
            );
            if ($template !== null) {
                return $template;
            }
        }

        // e.g. invoice status
        if ($subTag) {
            $template = $this->findOneTemplateForLanguages(
                $tenantUid,
                $templateType,
                $crmLanguageUid,
                $templateObjectType,
                $subTag,
                $contactType,
                0
            );
            //var_dump($tenantUid, $templateType, $templateObjectType,$subTag);exit(0);
            if ($template !== null) {
                return $template;
            }
        }
        $template = $this->findOneTemplateForLanguages(
            $tenantUid,
            $templateType,
            $crmLanguageUid,
            $templateObjectType,
            0,
            $contactType,
            0
        );
        if ($template !== null) {
            return $template;
        }
        // templateObjectType will probably be delete in next version - not used anymore
        $template = $this->findOneTemplateForLanguages(
            $tenantUid,
            $templateType,
            $crmLanguageUid,
            '',
            0,
            $contactType,
            0
        );
        if ($template !== null) {
            return $template;
        }
        return null;
    }

    
    public function findFavoritesByVerbandType(
        ?int $tenantUid,
        string $templateType = null,
        string $contactType = null,
        bool $returnRawQueryResult = false
    ) {
        $queryBuilder = $this->createTemplateLetterQueryBuilder();
        $queryBuilder->select($this->tablename . '.*')->from($this->tablename);
        $constraints = [
            $queryBuilder->expr()
                         ->eq('active', 1),
            $queryBuilder->expr()
                         ->eq('favorite', 1), //$queryBuilder->createNamedParameter(1, \Connection::PARAM_INT)
        ];
        if ($tenantUid !== null) {
            $constraints[] = $queryBuilder->expr()->eq('tenant_id', (int)$tenantUid);
        }
        
        $queryBuilder->addTemplateTypeByName($templateType);
        if ($contactType !== null && $contactType !== '') {
            $constraints[] = $queryBuilder->expr()->or(
                $queryBuilder->expr()
                             ->eq('object_type', $queryBuilder->createNamedParameter($contactType)),
                $queryBuilder->expr()
                             ->eq('object_type', $queryBuilder->createNamedParameter('')),
                $queryBuilder->expr()
                             ->isNull('object_type')
            );
        }
        
        $queryBuilder->andWhere($queryBuilder->expr()->and(...$constraints));
        $rows = $queryBuilder->executeQuery()
                             ->fetchAllAssociative();
        
        if ($returnRawQueryResult) {
            return $rows;
        }
        return $this->dataMapper->map($this->objectType, $rows);
    }
}
