<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Controller;

use Internetgalerie\IgCrmTimeRecording\Domain\Model\Activity;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\ActivityGroupRepository;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\ActivityRepository;
use Internetgalerie\IgCrmTimeRecording\Utility\PersonUtility;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Extbase\Annotation\IgnoreValidation;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

class ActivityController extends ActionController
{
    /**
     * activityRepository
     *
     * @var ActivityRepository
     */
    protected ?PersonUtility $personUtility = null;
    protected $activityRepository = null;
    protected $activityGroupRepository = null;

    
    public function injectActivityRepository(ActivityRepository $activityRepository): void
    {
        $this->activityRepository = $activityRepository;
    }
    public function injectActivityGroupRepository(ActivityGroupRepository $activityGroupRepository): void
    {
        $this->activityGroupRepository = $activityGroupRepository;
    }
    public function injectPersonUtility(PersonUtility $personUtility): void
    {
        $this->personUtility = $personUtility;
    }


    /**
     * action list
     */
    public function listAction(): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $activities = $this->activityRepository->findAll();
        $this->view->assign('activities', $activities);
        return $this->htmlResponse();
    }

    /**
     * action show
     */
    public function showAction(Activity $activity): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->view->assign('activity', $activity);
        return $this->htmlResponse();
    }

    /**
     * action new
     */
    public function newAction(): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->assignFormFields();
        return $this->htmlResponse();
    }

    /**
     * action create
     */
    public function createAction(Activity $newActivity): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->addFlashMessage('Eintrag erstellt', '', ContextualFeedbackSeverity::INFO);
        $this->activityRepository->add($newActivity);
        return $this->redirect('list');
    }

    /**
     * action edit
     */
    #[IgnoreValidation([
        'argumentName' => 'activity',
    ])]
    public function editAction(Activity $activity): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->assignFormFields();
        $this->view->assign('activity', $activity);
        return $this->htmlResponse();
    }

    /**
     * action update
     */
    public function updateAction(Activity $activity): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->addFlashMessage('Eintrag aktualisiert', '', ContextualFeedbackSeverity::INFO);
        $this->activityRepository->update($activity);
        return $this->redirect('list');
    }

    /**
     * action confirmDelete,
     */
    public function confirmDeleteAction(Activity $activity): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->view->assign('activity', $activity);
        return $this->htmlResponse();
    }

    /**
     * action delete
     */
    public function deleteAction(Activity $activity): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->addFlashMessage('Eintrag gelöscht', '', ContextualFeedbackSeverity::INFO);
        $this->activityRepository->remove($activity);
        return $this->redirect('list');
    }

    protected function assignFormFields()
    {
        $activityGroups = $this->activityGroupRepository->findAll();
        $this->view->assign('activityGroups', $activityGroups);
        $items = $GLOBALS['TCA']['tx_igcrmtimerecording_domain_model_activity']['columns']['internal_purpose']['config']['items'];
        foreach ($items as &$item) {
            $item['label'] = str_starts_with((string) $item['label'], 'LLL:') ? LocalizationUtility::translate(
                $item['label']
            ) : $item['label'];
        }

        $this->view->assign('internalPurposeOptions', $items);
    }
}
