<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Controller;

use DateTime;
use Internetgalerie\IgCrmTemplate\Domain\Model\TemplatePageLayout;
use Internetgalerie\IgCrmTemplate\Utility\TemplateUtility;
use Internetgalerie\IgCrmTimeRecording\Domain\Model\Time;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\ActivityRepository;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\EmployeeRepository;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\TimeRepository;
use Internetgalerie\IgCrmTimeRecording\Utility\DateUtility;
use Internetgalerie\IgCrmTimeRecording\Utility\PersonUtility;
use Internetgalerie\IgDatapoolFe\Property\TypeConverter\TimeIntegerConverter;
use Internetgalerie\IgDoctrinePaginator\Pagination\DoctrinePaginator;
use Internetgalerie\IgRender\Utility\PdfUtility;
use IntlDateFormatter;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Http\PropagateResponseException;
use TYPO3\CMS\Core\Page\AssetCollector;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Annotation\IgnoreValidation;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;

class TimeController extends ActionController
{
    protected ?PersonUtility $personUtility = null;
    protected ?AssetCollector $assetCollector = null;
    protected ?TemplateUtility $templateUtility = null;
    protected $dateFormat = [
        'year' => [
            'col1' => '%Y',
            'col2' => '',
            'source' => 'date_min',
        ],
        'month' => [
            'col1' => '%Y',
            'col2' => '%B',
            'source' => 'date_min',
        ],
        'week' => [
            'col1' => '%Y',
            'col2' => 'KW %V',
            'source' => 'date_min',
        ],
        'day' => [
            'col1' => '%A',
            'col2' => '%d.%m.%Y',
            'source' => 'date_min',
        ],
    ];

    /**
     * activityRepository
     *
     * @var ActivityRepository
     */
    protected $activityRepository = null;

    protected ?EmployeeRepository $employeeRepository = null;

    /**
     * timeRepository
     *
     * @var TimeRepository
     */
    protected $timeRepository = null;

    public function injectTemplateUtility(TemplateUtility $templateUtility): void
    {
        $this->templateUtility = $templateUtility;
    }
    
    public function injectPersonUtility(PersonUtility $personUtility): void
    {
        $this->personUtility = $personUtility;
    }

    public function injectAssetCollector(AssetCollector $assetCollector): void
    {
        $this->assetCollector = $assetCollector;
    }

    public function injectEmployeeRepository(EmployeeRepository $employeeRepository): void
    {
        $this->employeeRepository = $employeeRepository;
    }


    public function injectActivityRepository(ActivityRepository $activityRepository): void
    {
        $this->activityRepository = $activityRepository;
    }

    public function injectTimeRepository(TimeRepository $timeRepository): void
    {
        $this->timeRepository = $timeRepository;
    }

    public function trackerAction(): ResponseInterface
    {
        $employee = $this->personUtility->getFrontendUserEmployee();
        $this->view->assign('employee', $employee);
        $this->assignEditFormfields();

        return $this->htmlResponse();
    }

    /**
     * action add
     */
    public function addAction(Time $newTime): ResponseInterface
    {
        $redirectAction = $this->getAssignRedirectAction('tracker');
        $search = $this->getSearch();
        $employeeId = (int)($search['employee'] ?? 0);
        $arguments = [
            'search' => $search,
        ];
        if ($this->personUtility->hasRoleAdmin() && $employeeId) {
            $employee = $this->employeeRepository->findByUid($employeeId);
        } else {
            $employee = $this->personUtility->getFrontendUserEmployee();
        }
        $newTime->setEmployeeId($employee->getUid());

        $timeActivity = $newTime->getActivity();
        $internalPurpose = $timeActivity ? $timeActivity->getInternalPurpose() : null;
        if (
            ($internalPurpose === PersonUtility::$activityInternalPurposeVacation || $internalPurpose === PersonUtility::$activityInternalPurposeSick)
            && !$this->personUtility->isWorkingDay($newTime->getStartDate())
        ) {
            $arguments['errorValue'] = $this->request->getArgument('newTime');
            $this->addFlashMessage(
                $newTime->getStartDate()->format(DateUtility::getDateFormat()) . ' ist kein Arbeitstag',
                '',
                ContextualFeedbackSeverity::ERROR
            );
            return $this->redirect($redirectAction, null, null, $arguments);
        }
        if ($this->timeRepository->isConflict($newTime)) {
            $this->addFlashMessage(
                $newTime->getStartDate()->format(
                    DateUtility::getDateTimeFormat()
                ) . ' bis ' . $newTime->getEndDate()->format(
                    DateUtility::getTimeFormat()
                ) . ' überschneidet mit vorhandenen Zeiten',
                '',
                ContextualFeedbackSeverity::ERROR
            );
            $arguments['errorValue'] = $this->request->getArgument('newTime');
            return $this->redirect($redirectAction, null, null, $arguments);
        }
        
        
        $this->addFlashMessage(
            'Zeiten gespeichert: ' . $newTime->getStartDate()->format(
                DateUtility::getDateTimeFormat()
            ) . ' bis ' . $newTime->getEndDate()->format('H:i:s') . ' (Dauer: ' . $newTime->getDurationTime() . ')',
            '',
            ContextualFeedbackSeverity::INFO
        );
        $this->timeRepository->addAndPersist($newTime);
        $arguments['lastEntry'] = $newTime;
        return $this->redirect($redirectAction, null, null, $arguments);
    }
    /**
     * action public
     */
    public function publicAction(): ResponseInterface
    {
        $this->assetCollector->addJavaScript(
            'internetgalerie/ig-crm-time-recording/time/api',
            'EXT:ig_crm_time_recording/Resources/Public/JavaScript/time/api.js?v=' . time()
        );
        $this->assetCollector->addJavaScript(
            'internetgalerie/ig-crm-time-recording/time/format',
            'EXT:ig_crm_time_recording/Resources/Public/JavaScript/time/format.js?v=' . time()
        );
        $this->assetCollector->addJavaScript(
            'internetgalerie/ig-crm-time-recording/timer',
            'EXT:ig_crm_time_recording/Resources/Public/JavaScript/public_tracker.js?v=' . time()
        );
        $this->assignFormfields();
        return $this->htmlResponse();
    }

    public function timeOffAction(Time $lastEntry = null): ResponseInterface
    {
        $this->assetCollector->addJavaScript(
            'internetgalerie/ig-crm-time-recording/event',
            'EXT:ig_crm_time_recording/Resources/Public/JavaScript/events.js',
        );
        $this->assetCollector->addJavaScript(
            'internetgalerie/ig-crm-time-recording/list',
            'EXT:ig_crm_time_recording/Resources/Public/JavaScript/list.js',
        );
        //var_dump($_GET, $lastEntry);exit(0);
        $year = (int) date('Y');
        $search = $this->getAssignSearch();
        $employeeId = (int)($search['employee'] ?? 0);
        if ($this->personUtility->hasRoleAdmin()) {
            if ($employeeId) {
                $employee = $this->employeeRepository->findByUid($employeeId);
            } else {
                $employee = $this->personUtility->getFrontendUserEmployee();
            }
        } else {
            $employee = $this->personUtility->getFrontendUserEmployee();
        }
        $this->view->assign('employee', $employee);
        $duration = $this->personUtility->getWorkingDayInSeconds();
        if ($lastEntry instanceof Time) {
            $activity = $lastEntry->getActivity()
->getUid();
            $startDate = clone $lastEntry->getStartDate();
            $this->personUtility->setNextWorkingDay($startDate);
        } else {
            $activity = null;
            $startDate = (new DateTime())->setTime(8, 0, 0);
            if ($this->request->hasArgument('errorValue')) {
                $errorValue = $this->request->getArgument('errorValue');
                if ($errorValue['activity'] ?? 0 > 0) {
                    $activity = (int)$errorValue['activity'];
                }
                if ($errorValue['startDate']['date'] ?? false) {
                    $startDate = $errorValue['startDate']['date'];
                }
                if ($errorValue['duration'] ?? false) {
                    $duration = DateUtility::timeToSeconds($errorValue['duration']);
                }
            }
        }
        $endDate = new DateTime();
        $endDate->setTime(8, 0, 0);
        $endDate->modify('+' . $duration . ' seconds');
        $default = [
            'startDate' => $startDate,
            'endDate' => $endDate,
            'duration' => $duration,
            'activity' => $activity,
        ];
        $this->view->assign('default', $default);
        $this->view->assign('lastEntry', $lastEntry);
        $internalPurposeIds = [
            PersonUtility::$activityInternalPurposeVacation,
            PersonUtility::$activityInternalPurposeSick,
            PersonUtility::$activityInternalPurposeEo,
            PersonUtility::$activityInternalPurposeEducation,
        ];
        $search = [
            //nternalPurpose' => PersonUtility::$activityInternalPurposeVacation,
            'internalPurposeIn' => $internalPurposeIds,
            'employeeId' => $employee ? $employee->getUid() : '',
            'startDate' => $year . '-01-01',
        ];
        $this->view->assign('showTimesAddHeader', true);
        //$this->assignEditFormfields(['hasInternalPurpose' => 1]);
        $this->assignEditFormfields([
            'internalPurposeIn' => $internalPurposeIds,
        ]);
        $this->assignTimes($search);
        $this->view->assign('redirectAction', 'timeOff');

        if ($employee) {
            $vacationCarryover = $employee->getHolidayCarryover();
            $vacationTotal = $employee->getTotalHolidayDaysInYear();
            $emploeeTakenVacationDaysRow = $this->timeRepository->getVacationTotalForYearByEmployeeId(
                $employee->getUid(),
                $year
            );
            $vacationTaken = ($emploeeTakenVacationDaysRow['timeInSeconds'] ?? 0) / $this->personUtility->getWorkingDayInSeconds();
            $vacationBalance = [
                'total' => $vacationTotal,
                'holidayCarryover' => $vacationCarryover,
                'taken' => $vacationTaken,
                'open' => $vacationTotal - $vacationTaken,
            ];
            $this->view->assign('vacationBalance', $vacationBalance);
        }
        //$times = $this->timeRepository->findBySearchExecute();
        //$this->view->assign('times', $times);
        return $this->htmlResponse();
    }

    
    /**
     * action list
     */
    public function listAction(): ResponseInterface
    {
        $this->assetCollector->addJavaScript(
            'internetgalerie/ig-crm-time-recording/list',
            'EXT:ig_crm_time_recording/Resources/Public/JavaScript/list.js',
        );
        $search = $this->getAssignSearch();
        $this->assignFormfields();
        $this->assignTimes($search);
        $this->view->assign('tenants', $this->getTenants());
        return $this->htmlResponse();
    }

    /**
     * action pdf
     */
    public function pdfAction(): ResponseInterface
    {
        $search = $this->getAssignSearch();
        $this->assignFormfields();
        $this->assignTimes($search, false);

        $view = $search['view'] ?? '';
        /*
          $templateUtility = GeneralUtility::makeInstance(TemplateUtility::class);
          $templateUtility->setTenand($search['tenantId'] ?? null);
          $template = $templateUtility->getTemplate('Time', $contact, TemplateInterface $object)
        */
        $header = 'Zeiterfassung';
        
        if ($view == 'month' && $search['startDate'] ?? false) {
            $datetime = new DateTime($search['startDate']);
            $formatter = IntlDateFormatter::create(
                'de-DE',
                IntlDateFormatter::FULL,
                IntlDateFormatter::NONE,
                null,
                null,
                'MMMM yyyy'
            );
            //$formatter->setPattern('MMMM yyyy');
            $header .= ' ' . $formatter->format($datetime);
            //$header = 'Monatsübersicht ' . $datetime->format('m');
        } elseif ($search['startDate']['date'] ?? false && $search['endDate']['date'] ?? false) {
            $startDatetime = new DateTime($search['startDate']['date']);
            $endDatetime = new DateTime($search['endDate']['date']);
            $header .= ' ' . $startDatetime->format('d.m.Y') . ' - ' . $endDatetime->format('d.m.Y');
        }
        
        $header .= ':';
        if ($search['employeeId'] ?? false) {
            $employee = $this->personUtility->getEmployeeByUid((int) $search['employeeId']);
            $header .= ' ' . $employee->getName();
            $this->view->assign('receiver', $employee->getContact());
        }
        $this->view->assign('header', $header);

        if (isset($search['tenantId'])) {
            $this->templateUtility->setTenantById((int)($search['tenantId']));
        } else {
            $tenants = $this->getTenants();
            if (count($tenants) === 1) {
                $this->templateUtility->setTenant($tenants[0]);
            } else {
                $this->templateUtility->setTenantById(null);
            }
        }
        //$this->templateUtility->setTenantById(0);
        $templatePageLayout = $this->templateUtility->getTemplatePageLayout(
            'Time',
            TemplatePageLayout::PAGE_FORMART_A4,
            TemplatePageLayout::PAGE_ORIENTATION_LANDSCAPE
        );

        $css = $templatePageLayout->getCss();
        //var_dump($templatePageLayout->getHeaderCenterPartial());exit(0);
        //var_dump($templatePageLayout->getUid(), $templatePageLayout->getHeaderRightTemplate(), $templatePageLayout);exit(0);
        $this->view->assign('templatePageLayout', $templatePageLayout);
        $template = $this->templateUtility->getTemplate('Time');
        /*
          if ($templatePageLayout) {
          $css = $templatePageLayout->getCss();
          } else {
          die('no default pageLayout Css');
          $css = $this->templateUtility->getPageLayoutCss();
          }
        */
        $this->view->assign('template', $template);

        $sender = $this->templateUtility->getTenant();

        $this->view->assign('sender', $sender);
        $this->view->assign('tenant', $sender);
        $html = $this->view->render();

        $pdf = new PdfUtility();
        $pdf->setFilename('Zeiterfassung-' . strftime('%Y-%m-%d') . '.pdf');
        $pdf->setTitle('Zeiterfassung');
        $pdf->setConverter(PdfUtility::CONVERTER_WEASYPRINT);
        //$pdf->setCssFile('EXT:ig_crm_admin/Resources/Public/Css/styles.css');
        $css .= $template->getCss();
        $pdf->setCss($css);
        /*
          $pdf->setCssFile('EXT:ig_fibu/Resources/Public/Css/weasyprint.css');
          $pdf->addCssFile('EXT:ig_fibu/Resources/Public/Css/box.css');
          $pdf->addCssFile('EXT:ig_fibu/Resources/Public/Css/pdf.css');
          //$pdf->addCssFile('EXT:ig_fibu/Resources/Public/Css/a4.css');
          //$pdf->addCssFile('EXT:ig_fibu/Resources/Public/Css/qr.css');
          $pdf->addCssFile('EXT:ig_crm_time_recording/Resources/Public/Css/pdf.css');
          $pdf->addCssFile('EXT:ig_layout/Resources/Public/Pdf/Css/styles.css');
        */
        if ($_GET['html'] ?? false) {
            $pdf->debug($html);
        }
        $pdfContent = $pdf->get($html);
        $response = $this->responseFactory->createResponse()
                                          ->withHeader('Expires', '0')
                                          ->withHeader('Cache-Control', 'private')
                                          ->withHeader('Content-Type', 'application/pdf; charset=utf-8')
                                          ->withHeader(
                                              'Content-Disposition',
                                              'attachment; filename="' . $pdf->getFilename() . '"'
                                          )
                                          ->withHeader('Content-Length', (string)strlen($pdfContent));
        $response->getBody()
                 ->write($pdfContent);
        throw new PropagateResponseException($response, 200);
    }

    /**
     * action overview
     */
    public function overviewAction(): ResponseInterface
    {
        $search = $this->getSearch([
            'year' => date('Y'),
            'month' => (int)date('m'),
            'view' => 'day',
        ]);

        $this->assignFormfields();
        $this->view->assign('dateFormat', $this->dateFormat[$search['view']]);
        
        $months = [];
        for ($m = 1; $m <= 12; $m++) {
            $months[$m] = $m;
        }
        $years = [];
        for ($y = date('Y'); $y >= 2022; $y--) {
            $years[$y] = $y;
        }
        $views = [
            'day' => 'Tag',
            'week' => 'Woche',
            'month' => 'Monat',
            'year' => 'Jahr',
        ];
        $this->view->assignMultiple([
            'years' => $years,
            'months' => $months,
            'views' => $views,
            'search' => $search,
        ]);

        if (isset($search['view'])) {
            $times = $this->timeRepository->findOverviewBySearch($search);
        } else {
            $times = [];
        }
        $this->view->assign('times', $times);
        if (!empty($times)) {
            $totalTime = 0;
            $totalAmount = 0;
            foreach ($times as $time) {
                $totalTime += $time['sum_duration'];
                $totalAmount += $time['sum_amount'];
            }
            $this->view->assign('total', [
                'time' => $totalTime,
                'amount' => $totalAmount,
            ]);
        }


        return $this->htmlResponse();
    }

    /**
     * action show
     */
    public function showAction(Time $time): ResponseInterface
    {
        $this->view->assign('time', $time);
        return $this->htmlResponse();
    }

    /**
     * action new
     */
    public function newAction(): ResponseInterface
    {
        $this->assignEditFormfields();
        $search = $this->getAssignSearch();
        $this->view->assign('tenants', $this->getTenants());

        return $this->htmlResponse();
    }
    /**
     * action create
     */
    public function createAction(Time $newTime)
    {
        if (!$this->personUtility->hasRoleAdmin() || $newTime->getEmployeeId() === null) {
            $employee = $this->personUtility->getFrontendUserEmployee();
            if ($employee) {
                $newTime->setEmployeeId($employee->getUid());
            }
        }
        
        $this->addFlashMessage(
            'Zeiten gespeichert: ' . $newTime->getStartDate()->format(
                'd.m.Y H:i:s'
            ) . ' bis ' . $newTime->getEndDate()->format('H:i:s') . ' (Dauer: ' . $newTime->getDurationTime() . ')',
            '',
            ContextualFeedbackSeverity::INFO
        );

        $this->timeRepository->add($newTime);
        $arguments = [
            'search' => $this->getSearch(),
        ];
        $redirectAction = $this->getAssignRedirectAction('list');
        return $this->redirect($redirectAction, null, null, $arguments);
    }

    /**
     * action edit
     */
    #[IgnoreValidation([
        'argumentName' => 'time',
    ])]
    public function editAction(Time $time): ResponseInterface
    {
        if (!$this->personUtility->hasPermissionWrite($time)) {
            die('no access');
        }
        $this->assignEditFormfields();

        $search = $this->getAssignSearch();
        $this->view->assign('tenants', $this->getTenants());
        $this->view->assign('time', $time);
        return $this->htmlResponse();
    }
    /**
     * action update
     */
    public function updateAction(Time $time)
    {
        if (!$this->personUtility->hasPermissionWrite($time)) {
            die('no access');
        }
        
        $this->addFlashMessage(
            'Eintrag aktualisiert: ' . $time->getStartDate()->format(
                'd.m.Y H:i:s'
            ) . ' bis ' . $time->getEndDate()->format('H:i:s') . ' (Dauer: ' . $time->getDurationTime() . ')',
            '',
            ContextualFeedbackSeverity::INFO
        );
        $this->timeRepository->update($time);
        
        $arguments = [
            'search' => $this->getSearch(),
        ];
        $redirectAction = $this->getAssignRedirectAction('list');
        return $this->redirect($redirectAction, null, null, $arguments);
    }
    /**
     * action confirmDelete
     */
    public function confirmDeleteAction(Time $time): ResponseInterface
    {
        if (!$this->personUtility->hasPermissionWrite($time)) {
            die('no access');
        }
        $this->view->assign('time', $time);
        $redirectAction = $this->getAssignRedirectAction();
        $search = $this->getAssignSearch();
        return $this->htmlResponse();
    }

    /**
     * action delete
     */
    public function deleteAction(Time $time)
    {
        if (!$this->personUtility->hasPermissionWrite($time)) {
            die('no access');
        }
        $this->addFlashMessage(
            'Eintrag wurde gelöscht: ' . $time->getStartDate()->format(
                'd.m.Y H:i:s'
            ) . ' bis ' . $time->getEndDate()->format('H:i:s') . ' (Dauer: ' . $time->getDurationTime() . ')',
            '',
            ContextualFeedbackSeverity::INFO
        );
        $this->timeRepository->remove($time);
        $arguments = [
            'search' => $this->getSearch(),
        ];
        $redirectAction = $this->getAssignRedirectAction('list');
        return $this->redirect($redirectAction, null, null, $arguments);
    }

    public function monthAction(): ResponseInterface
    {
        $search = $this->getAssignSearch();
        $this->assignFormfields();
        $this->assignTimes($search);
        return $this->htmlResponse();
    }
    protected function initializeAjaxAction()
    {
        $this->search = $this->getSearch();
        $this->tenantId = isset($this->search['tenant']) ? (int)$this->search['tenant'] : null;
        $this->templateUtility->setTenantById($this->tenantId);

        $pmc = $this->arguments->getArgument('newTime')
                               ->getPropertyMappingConfiguration();
        $tc = GeneralUtility::makeInstance(TimeIntegerConverter::class);
        $pmc->forProperty('duration')
            ->setTypeConverter($tc);
    }

    protected function initializeAddAction()
    {
        $pmc = $this->arguments->getArgument('newTime')
                               ->getPropertyMappingConfiguration();
        $tc = GeneralUtility::makeInstance(TimeIntegerConverter::class);
        $pmc->forProperty('duration')
            ->setTypeConverter($tc);
    }

    protected function initializeCreateAction()
    {
        $pmc = $this->arguments->getArgument('newTime')
                               ->getPropertyMappingConfiguration();
        $tc = GeneralUtility::makeInstance(TimeIntegerConverter::class);
        $pmc->forProperty('duration')
            ->setTypeConverter($tc);
    }

    protected function initializeUpdateAction()
    {
        $pmc = $this->arguments->getArgument('time')
                               ->getPropertyMappingConfiguration();
        $tc = GeneralUtility::makeInstance(TimeIntegerConverter::class);
        $pmc->forProperty('duration')
            ->setTypeConverter($tc);
    }
    protected function assignEditFormfields(array $activitySearch = [])
    {
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_crm_time_recording/Resources/Public/JavaScript/time/api.js');
        $pageRenderer->addJsFooterFile('EXT:ig_crm_time_recording/Resources/Public/JavaScript/time/format.js');
        $pageRenderer->addJsFooterFile('EXT:ig_crm_time_recording/Resources/Public/JavaScript/tracker.js');
        $redirectAction = $this->getAssignRedirectAction();
        $this->assignFormfields($activitySearch);
    }

    protected function assignFormfields(array $activitySearch = [])
    {
        if (empty($activitySearch)) {
            $activities = $this->activityRepository->findAll();
        } else {
            $activities = $this->activityRepository->findBySearch($activitySearch);
        }
        $this->view->assign('activities', $activities);
        $employees = $this->employeeRepository->findActive();
        $this->view->assign('employees', $employees);
    }
    
    protected function assignTimes(array $search, bool $usePageinator = true)
    {
        if (!empty($search)) {
            $totalTime = 0;
            $totalAmount = 0;
            $totalActivities = [];

            //$times = $this->timeRepository->findBySearchExecute($search);
            //$total = $this->timeRepository->getTotalByObjects($times);

            $queryBuilder = $this->timeRepository->findBySearchRaw($search);
            $currentPage = $this->request->hasArgument('currentPage') ? (int)$this->request->getArgument(
                'currentPage'
            ) : 1;
            $search['limit'] = (int)($search['limit'] ?? 1000);
            if ($search['limit'] > 10000 || $search['limit'] <= 0) {
                $search['limit'] = 10000;
            }
            //$paginator = new DoctrinePaginator($queryBuilder, $currentPage, 4, null, [$this->timeRepository, 'prepareRows']);
            $paginator = new DoctrinePaginator(
                $queryBuilder,
                $currentPage,
                $search['limit'],
                null,
                TimeRepository::prepareRows(
                    ...
                )
            );
            if ($usePageinator) {
                $pagination = $paginator->createSlidingWindowPagination(10); // maximal Anzahl Links
                $this->view->assign('pagination', [
                    'paginator' => $paginator,
                    'pagination' => $pagination,
                ]);
                $times = $paginator->getPaginatedItems();
                $total = $this->timeRepository->getTotalByArray($times);
            } else {
                $times = $this->timeRepository->findBySearchExecuteRaw($search);
                $this->view->assign('times', $times);
                $total = $this->timeRepository->getTotalByArray($times);
            }

            $confAttributes = $this->settings['conf']['attributes'] ?? [];
            $action = $this->request->getControllerActionName();
            if (!isset($this->settings['time'][$action]['attributes'])) {
                $action = 'list';
            }
            $settingsAttributes = $this->settings['time'][$action]['attributes'] ?? 'activity, description, amount, employee';

            $attributes = GeneralUtility::trimExplode(',', $settingsAttributes, true);
            $confTime = [];
            foreach ($attributes as $attribute) {
                $confTime[$attribute] = $confAttributes[$attribute] ?? [];
            }
            $conf = [
                'time' => [
                    'attributes' => $confTime,
                ],
            ];
            //var_dump($conf, $confAttributes);exit(0);
            $this->view->assign('conf', $conf);

            $this->view->assign('total', $total);
            //$this->view->assign('showActivities', count($total['activities']) > 1);
        }
    }
    protected function getSearch($search = [])
    {
        if ($this->request->hasArgument('@search')) {
            $search = array_merge($search, $this->request->getArgument('@search'));
        } elseif ($this->request->hasArgument('search')) {
            $search = array_merge($search, $this->request->getArgument('search'));
        }
        return $search;
    }
    protected function getAssignRedirectAction(?string $defaultRedirectAction = '')
    {
        if ($this->request->hasArgument('redirectAction') && $this->request->getArgument('redirectAction')) {
            $redirectAction = $this->request->getArgument('redirectAction');
        } else {
            $redirectAction = $defaultRedirectAction;
        }
        $this->view->assign('redirectAction', $redirectAction);
        return $redirectAction;
    }
    protected function getAssignSearch(array $defaultSearch = []): array
    {
        $search = $this->getSearch($defaultSearch);
        $this->view->assign('search', $search);
        return $search;
    }
    protected function getTenants(): array
    {
        return $this->templateUtility->getTenants();
    }
}
