<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Controller;

use DateTime;
use Exception;
use Internetgalerie\IgCrmTimeRecording\Domain\Model\Time;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\ActivityRepository;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\TimeRepository;
use Internetgalerie\IgCrmTimeRecording\Utility\PersonUtility;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Http\PropagateResponseException;
use TYPO3\CMS\Core\Page\AssetCollector;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

class ApiController extends ActionController
{
    // Minimal Time to add
    protected const MIN_TIME = 5;
    protected ?PersonUtility $personUtility = null;
    protected ?AssetCollector $assetCollector = null;


    protected $persistenceManager = null;
    /**
     * activityRepository
     *
     * @var ActivityRepository
     */
    protected $activityRepository = null;

    /**
     * timeRepository
     *
     * @var TimeRepository
     */
    protected $timeRepository = null;

    
    public function injectPersonUtility(PersonUtility $personUtility): void
    {
        $this->personUtility = $personUtility;
    }

    public function injectAssetCollector(AssetCollector $assetCollector): void
    {
        $this->assetCollector = $assetCollector;
    }

    public function injectActivityRepository(ActivityRepository $activityRepository): void
    {
        $this->activityRepository = $activityRepository;
    }

    public function injectTimeRepository(TimeRepository $timeRepository): void
    {
        $this->timeRepository = $timeRepository;
    }
    public function injectPersistenceManager(PersistenceManager $persistenceManager): void
    {
        $this->persistenceManager = $persistenceManager;
    }
    
    /**
     * action add
     */
    public function addAction(): ResponseInterface
    {
        $json = $this->request->getBody()
->__toString();
        try {
            $data = json_decode($json, true);
        } catch (Exception) {
            die('error');
        }
        $durationInSeconds = (int)($data['durationInSeconds'] ?? 0);
        $startDateString = $data['startDate'] ?? null;
        $description = $data['description'] ?? '';
        $employeeId = (int)($data['employeeId'] ?? null);
        $activityUid = (int)($data['activityUid'] ?? null);
        
        $responseData = [
            'status' => [
                'error' => 0,
                'message' => '',
            ],
        ];
        
        $activity = $activityUid > 0 ? $this->activityRepository->findOneBy([
            'uid' => $activityUid,
        ]) : $this->activityRepository->getDefault();
        if ($activity === null) {
            $responseData['status']['error'] = 1;
            $responseData['status']['message'] = LocalizationUtility::translate(
                'error.no-activity',
                'IgCrmTimeRecording'
            );
        } else {
            if ($durationInSeconds >= static::MIN_TIME && $startDateString) {
                $startDate = new DateTime($startDateString);
                $newTime = GeneralUtility::makeInstance(Time::class);
                $newTime->setStartDate($startDate);
                $newTime->setActivity($activity);
                $newTime->setDuration($durationInSeconds);
                $newTime->setDescription($description);
                if ($employeeId == 0) {
                    $employee = $this->personUtility->getFrontendUserEmployee();
                    $employeeId = $employee->getUid();
                }
                $newTime->setEmployeeId($employeeId);
                $this->timeRepository->add($newTime);
                $this->persistenceManager->persistAll();

                //$day = $this->timeRepository->getDayTotalByEmployeeId($employeeId);
                //$month = $this->timeRepository->getMonthTotalByEmployeeId($employeeId);
                $responseData['data'] = [
                    'uid' => $newTime->getUid(),
                    'startDate' => $newTime->getStartDate()
->format('c'),
                    'endDate' => $newTime->getEndDate()
->format('c'),
                    'activity' => [
                        'uid' => $activity->getUid(),
                        'name' => $activity->getName(),
                        'amount' => $activity->getAmount(),
                    ],
                    'duration' => $newTime->getDuration(),
                    'description' => $newTime->getDescription(),
                    'employeeId' => $newTime->getEmployeeId(),
                    
                ];
            } else {
                $responseData['status']['error'] = 1;
                $responseData['status']['message'] = LocalizationUtility::translate(
                    'error.min-time',
                    'IgCrmTimeRecording',
                    [static::MIN_TIME,
                ]);
            }
        }
        /*
        'data' => [
                'comment' => $comment,
                'durationInSeconds' => $durationInSeconds,
                'startDate' => $startDate->format('c'),
            ],
        */
        $json = json_encode($responseData);
        $response = $this->responseFactory->createResponse()
                                          ->withHeader('Content-Type', 'application/json; charset=utf-8')
                                          ->withHeader('Expires', '0')
                                          ->withHeader('Cache-Control', 'private');
        $response->getBody()
->write($json);
        throw new PropagateResponseException($response, 200);
    }

    
    /**
     * action info
     */
    public function infoAction(): ResponseInterface
    {
        $json = $this->request->getBody()
->__toString();
        try {
            $data = json_decode($json, true);
        } catch (Exception) {
            die('error');
        }
        $employeeId = (int)($data['employeeId'] ?? null);
        $infos = GeneralUtility::trimExplode(',', $data['info'] ?? '', true);
        
        $responseData = [
            'status' => [
                'error' => 0,
                'message' => '',
            ],
        ];
        if ($employeeId <= 0) {
            $responseData['status']['error'] = 1;
            $responseData['status']['message'] = LocalizationUtility::translate(
                'error.no-employee',
                'IgCrmTimeRecording'
            );
        } else {
            $day = $this->timeRepository->getDayTotalByEmployeeId($employeeId);
            $month = $this->timeRepository->getMonthTotalByEmployeeId($employeeId);
            $responseData['data'] = [
                'day' => $day,
                'month' => $month,
                    
            ];
        }
        $json = json_encode($responseData);
        $response = $this->responseFactory->createResponse()
                                          ->withHeader('Content-Type', 'application/json; charset=utf-8')
                                          ->withHeader('Expires', '0')
                                          ->withHeader('Cache-Control', 'private');
        $response->getBody()
->write($json);
        throw new PropagateResponseException($response, 200);
    }

    /**
     * action list
     */
    public function listAction(): ResponseInterface
    {
        $json = $this->request->getBody()
->__toString();
        try {
            $search = json_decode($json, true);
        } catch (Exception) {
            die('error');
        }
        $responseData = [
            'status' => [
                'error' => 0,
                'message' => '',
            ],
        ];

        // limit to own user
        if (false) {
            $employeeId = (int)$search['employeeId'];
            if ($this->personUtility->hasRoleAdmin() && $employeeId) {
                $employee = $this->employeeRepository->findByUid($employeeId);
            } else {
                $employee = $this->personUtility->getFrontendUserEmployee();
            }
            $search['employeeId'] = $employee->getUid();
        }
        $results = $this->timeRepository->findBySearchExecuteRaw($search, [], true);
        
        $responseData['results'] = $results;

        $json = json_encode($responseData);
        $response = $this->responseFactory->createResponse()
                                          ->withHeader('Content-Type', 'application/json; charset=utf-8')
                                          ->withHeader('Expires', '0')
                                          ->withHeader('Cache-Control', 'private');
        $response->getBody()
->write($json);
        throw new PropagateResponseException($response, 200);
    }
}
