<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Controller;

use Internetgalerie\IgCrmTimeRecording\Domain\Model\Holiday;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\HolidayRepository;
use Internetgalerie\IgCrmTimeRecording\Utility\PersonUtility;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;

class HolidayController extends ActionController
{
    protected ?HolidayRepository $holidayRepository = null;
    protected ?PersonUtility $personUtility = null;

    public function injectHolidayRepository(HolidayRepository $holidayRepository): void
    {
        $this->holidayRepository = $holidayRepository;
    }
    public function injectPersonUtility(PersonUtility $personUtility): void
    {
        $this->personUtility = $personUtility;
    }

    /**
     * action list
     */
    public function listAction(): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $holidays = $this->holidayRepository->findAll();
        $this->view->assign('holidays', $holidays);
        return $this->htmlResponse();
    }

    /**
     * action new
     */
    public function newAction(): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        return $this->htmlResponse();
    }

    /**
     * action create
     */
    public function createAction(Holiday $newHoliday)
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->addFlashMessage('Eintrag erstellt', '', ContextualFeedbackSeverity::INFO);
        $this->holidayRepository->add($newHoliday);
        return $this->redirect('list');
    }

    /**
     * action edit
     */
    #[TYPO3\CMS\Extbase\Annotation\IgnoreValidation([
        'argumentName' => 'holiday',
    ])]
    public function editAction(Holiday $holiday): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->view->assign('holiday', $holiday);
        return $this->htmlResponse();
    }

    /**
     * action update
     */
    public function updateAction(Holiday $holiday): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->addFlashMessage('Eintrag aktualisiert', '', ContextualFeedbackSeverity::INFO);
        $this->holidayRepository->update($holiday);
        return $this->redirect('list');
    }

    /**
     * action delete
     */
    public function deleteAction(Holiday $holiday): ResponseInterface
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            return $this->htmlResponse('no access');
        }
        $this->addFlashMessage('Eintrag gelöscht', '', ContextualFeedbackSeverity::INFO);
        $this->holidayRepository->remove($holiday);
        return $this->redirect('list');
    }

    public function targetTimesAction(int $year = null): ResponseInterface
    {
        //var_Dump($year);
        $year ??= (int)date('Y');
        $yearNav = [
            'prev' => $year - 1,
            'current' => $year,
            'next' => $year + 1,
        ];
        $this->view->assign('yearNav', $yearNav);
        
        $targetTimes = $this->holidayRepository->getTargetTimesByYear(
            $year,
            $this->personUtility->getWorkingDayInSeconds()
        );
        $this->view->assign('workingDayInSeconds', $this->personUtility->getWorkingDayInSeconds());
        $this->view->assign('workingDayInHHMM', $this->personUtility->getWorkingDayInHHMM());
        $this->view->assign('targetTimes', $targetTimes);
        $this->view->assign('percents', $this->personUtility->getTargetTimesPercents());
        $this->view->assign('timePerDay', $this->personUtility->getWorkingDayInSeconds());
        $this->view->assign('year', $year);
        //var_dump($targetTimes);exit(0);

        $holidays = $this->holidayRepository->findByYear($year);
        $this->view->assign('holidays', $holidays);
        return $this->htmlResponse();
    }
}
