<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Utility;

use DateInterval;
use DateTime;
use Internetgalerie\IgAcl\Utility\AclUtility;
use Internetgalerie\IgCrmTimeRecording\Domain\Model\Employee;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\EmployeeRepository;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\FrontendUserRepository;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\HolidayRepository;
use Internetgalerie\IgFrontendUser\Utility\SecurityUtility;
use Internetgalerie\IgsCrm\Domain\Repository\PersonRepository;
use Internetgalerie\IgsCrm\Utility\SecurityUtility as CrmSecurityUtility;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Persistence\Repository;

class PersonUtility implements SingletonInterface
{
    public const ACL_CONTEXT = 'crm.time';
    public static array $targetTimesPercents = [100, 90, 80, 70, 60, 50, 40, 30, 20, 10];
    //public static int $timePerDayHour = 8;
    //public static int $timePerDayMinute = 24;
    public static int $workingDayInSeconds;
    public static int $activityInternalPurposeVacation = 1;
    public static int $activityInternalPurposeSick = 2;
    public static int $activityInternalPurposeEo = 3;
    public static int $activityInternalPurposeBreak = 4;
    public static int $activityInternalPurposeEducation = 5;
    protected array $settings = [];
    protected ?array $adminUsergroupUid = null;
    protected ConfigurationManager $configurationManager;
    protected ?PersonRepository $personRepository = null;
    protected ?TenantRepository $verbandRepository = null;
    protected $securityUtility = null;
    protected ?FrontendUserRepository $frontendUserRepository = null;
    protected ?Repository $addressRepository = null;
    protected array $adminFrontendUserGroupUids = [];
    protected array $viewFrontendUserGroupUids = [];
    private static $frontendUserGroupIds = null;
    /*
      public function __construct(
      protected readonly PersonRepository $personRepository,
      protected readonly TenantRepository $verbandRepository,
      protected readonly SecurityUtility $securityUtility,
      ) {
      }
    */

    public function __construct(
        private readonly Context $context,
        private readonly EmployeeRepository $employeeRepository,
        private readonly AclUtility $aclUtility,
    ) {
        if (class_exists(FrontendUserRepository::class)) {
            $this->frontendUserRepository = GeneralUtility::makeInstance(FrontendUserRepository::class);
            $this->addressRepository = $this->frontendUserRepository;
        }
        if (class_exists(PersonRepository::class)) {
            $this->personRepository = GeneralUtility::makeInstance(PersonRepository::class);
            $this->addressRepository = $this->personRepository;
        }
        if (class_exists(VerbansRepository::class)) {
            $this->verbansRepository = GeneralUtility::makeInstance(VerbansRepository::class);
        }
        if (class_exists(CrmSecurityUtility::class)) {
            $this->securityUtility = GeneralUtility::makeInstance(CrmSecurityUtility::class);
        } elseif (class_exists(SecurityUtility::class)) {
            $this->securityUtility = GeneralUtility::makeInstance(SecurityUtility::class);
        }
    }

    public function injectConfigurationManager(ConfigurationManager $configurationManager): void
    {
        $this->configurationManager = $configurationManager;
        $this->settings = $this->configurationManager->getConfiguration(
            ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        )['plugin.']['tx_igcrmtimerecording.']['settings.'];
        $this->adminFrontendUserGroupUids = GeneralUtility::intExplode(
            ',',
            $this->settings['adminFrontendUserGroupUids'] ?? ''
        );
        $this->viewFrontendUserGroupUids = GeneralUtility::intExplode(
            ',',
            $this->settings['viewFrontendUserGroupUids'] ?? ''
        );
        $this->workingDayInSeconds = (int)($this->settings['time.']['workingDayInSeconds'] ?? 0);
    }
    /*
      public function injectPersonRepository(PersonRepository $personRepository): void
      {
      $this->personRepository = $personRepository;
      }
      public function injectTenantRepository(TenantRepository $verbandRepository): void
      {
      $this->tenantRepository = $verbandRepository;
      }
      public function injectSecurityUtility(SecurityUtility $securityUtility)
      {
      $this->securityUtility = $securityUtility;
      }
     */

    
    /**
     * get the Address/Contact of with the given id
     */
    public function findByUid(int $uid)
    {
        if ($this->addressRepository) {
            return $this->addressRepository->findByUid($uid);
        }
        return null;
    }

    public function getContact(int $uid)
    {
        if ($this->addressRepository) {
            return $this->addressRepository->findOneBy([
                'uid' => $uid,
            ]);
            //return $this->addressRepository->findOneByFrontendUserUsername($username);
        }
        return null;
    }
    public function getEmployeeByUid(int $uid)
    {
        return $this->employeeRepository->findByUid($uid);
    }

    public function findEmployeeByTenant()
    {
        if ($this->personRepository) {
            return $this->personRepository->findEmployeeByTenant();
        }
        if ($this->frontendUserRepository) {
            // storagepid setzen in dieser ext oder hier
            if (!empty($this->viewFrontendUserGroupUids)) {
                return $this->frontendUserRepository->findWithUsergroups($this->viewFrontendUserGroupUids);
            }
            // storagepid setzen in dieser ext oder hier
            return $this->frontendUserRepository->findAll();
        }
        return [];
    }
    public function findFrontendUserByTenant()
    {
        return $this->frontendUserRepository->findAll();
    }
    /**
     * get the tenant for sender in letter header
     */
    public function getTenant(int $tenantUid = 1)
    {
        return $this->tenantRepository ? $this->tenantRepository->findByUid($tenantUid) : null;
    }

    /**
     * get the Address/Contact of the current logged in frontend user
     */
    //public function getFrontendUserContact()
    //{
    //    return $this->securityUtility ? $this->securityUtility->getFrontendUserContact() : null;
    //}
    public function getFrontendUserId(): ?int
    {
        return $this->context->getPropertyFromAspect('frontend.user', 'id');
    }

    public function getFrontendUserEmployee(): ?Employee
    {
        return $this->employeeRepository->findOneBy([
            'frontendUser' => $this->getFrontendUserId(),
        ]);
    }

    public function getFrontendUserGroupIds()
    {
        if (static::$frontendUserGroupIds === null) {
            static::$frontendUserGroupIds = [];
            foreach ($this->context->getPropertyFromAspect('frontend.user', 'groupIds') as $groupId) {
                if ($groupId > 0) {
                    static::$frontendUserGroupIds[] = $groupId;
                }
            }
        }
        return static::$frontendUserGroupIds;
    }

    
    /**
     * has the current frontend user admin rights
     */
    public function hasRoleAdmin()
    {
        return $this->aclUtility->hasRoleAdmin(self::ACL_CONTEXT);
    }
    public function isOwner($object = null): bool
    {
        return $this->aclUtility->isOwner(self::ACL_CONTEXT);
    }

    //public function isOwner(EntityWithAcl||array $object = null)
    public function hasPermissionWrite($object = null): bool
    {
        return $this->aclUtility->hasPermissionWrite($object, self::ACL_CONTEXT);
    }
    
    public function getContactConfig(?int $pageUid = null): array
    {
        if (class_exists(PersonRepository::class)) {
            $contactConfig = [
                'action' => 'ajaxSearch',
                'controller' => 'Person',
                'pluginName' => 'Mi',
                'extensionName' => 'igCrmAdmin',
                'argumentNameUid' => 'uid',
            ];
            if ($pageUid) {
                $contactConfig['pageUid'] = $pageUid;
            }
        } else {
            return [];
        }
        return $contactConfig;
    }

    // getTimePerday
    public function getWorkingDayInSeconds()
    {
        return $this->workingDayInSeconds;
    }
    public function getWorkingDayInHHMM()
    {
        $dateTime = new DateTime('00:00');
        $interval = new DateInterval('PT' . $this->workingDayInSeconds . 'S');
        $dateTime->add($interval);

        return $dateTime->format('H:i');
    }
    
    public function getTargetTimesPercents()
    {
        return static::$targetTimesPercents;
    }

    //@todo should be in settings
    public function isWeekend(DateTime $date): bool
    {
        // Check if the new date is a weekend
        return $date->format('N') >= 6;
    }
    
    public function setNextWorkingDay(DateTime $date): void
    {
        do {
            $date->modify('+1day');

            // Check if the new date is a weekend
            if ($this->isWeekend($date)) {
                // If it's Saturday (6) or Sunday (7), move to the next Monday
                $time = $date->format('H:i:s');
                $date->modify('next monday');
                $date->modify($time);
            }
            // Check if the new date is a holiday
            $isHoliday = HolidayRepository::isHoliday($date);
        } while ($isHoliday);
    }
    public function isWorkingDay(DateTime $date): bool
    {
        //  is a weekend
        if ($this->isWeekend($date)) {
            return false;
        }
        // is a holiday
        if (HolidayRepository::isHoliday($date)) {
            return false;
        }
        return true;
    }
}
