<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Controller;

use DateTime;
use Internetgalerie\IgCrmTemplate\Domain\Model\TemplatePageLayout;
use Internetgalerie\IgCrmTemplate\Utility\TemplateUtility;
use Internetgalerie\IgCrmTimeRecording\Domain\Model\Employee;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\EmployeeRepository;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\HolidayRepository;
use Internetgalerie\IgCrmTimeRecording\Domain\Repository\TimeRepository;
use Internetgalerie\IgCrmTimeRecording\Utility\PersonUtility;
use Internetgalerie\IgCrmTimeRecording\Utility\SalaryUtility;
use Internetgalerie\IgDatapoolFe\Controller\ActionController;
use Internetgalerie\IgDatapoolFe\Domain\Repository\CountryRepository;
use Internetgalerie\IgFrontendUser\Domain\Repository\FrontendUserRepository;
use Internetgalerie\IgRender\Utility\PdfUtility;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Http\PropagateResponseException;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Property\TypeConverter\DateTimeConverter;

class EmployeeController extends ActionController
{
    protected const dateProperties = ['entryDate', 'exitDate', 'birthday'];
    protected const objectName = 'employee';
    protected const newObjectName = 'newEmployee';
    protected int $timePerDayHour = 8;
    protected int $timePerDayMinute = 0;

    protected ?PersonUtility $personUtility = null;
    protected ?TemplateUtility $templateUtility = null;
    protected $searchDefault = [
        'active' => 1,
    ];
    protected ?int $tenantUid = null;

    protected ?EmployeeRepository $employeeRepository = null;

    /**
     * holidayRepository
     *
     * @var HolidayRepository
     */
    protected $holidayRepository = null;

    /**
     * timeRepository
     *
     * @var TimeRepository
     */
    protected $timeRepository = null;
    

    /**
     * FrontendUserRepository
     */
    protected ?FrontendUserRepository $frontendUserRepository = null;

    public function injectTemplateUtility(TemplateUtility $templateUtility): void
    {
        $this->templateUtility = $templateUtility;
    }

    public function injectPersonUtility(PersonUtility $personUtility): void
    {
        $this->personUtility = $personUtility;
    }

    public function injectEmployeeRepository(EmployeeRepository $employeeRepository): void
    {
        $this->employeeRepository = $employeeRepository;
    }

    public function injectHolidayRepository(HolidayRepository $holidayRepository): void
    {
        $this->holidayRepository = $holidayRepository;
    }

    public function injectTimeRepository(TimeRepository $timeRepository): void
    {
        $this->timeRepository = $timeRepository;
    }

    public function injectFrontendUserRepository(FrontendUserRepository $frontendUserRepository): void
    {
        $this->frontendUserRepository = $frontendUserRepository;
    }

    public function initializeAction(): void
    {
        if (!$this->personUtility->hasRoleAdmin()) {
            die('no access');
        }
        $this->search = $this->getSearch();
        $this->tenantId = isset($this->search['tenant']) ? (int)$this->search['tenant'] : null;
        $this->templateUtility->setTenantById($this->tenantId);
        if (isset($this->arguments[self::objectName]) || isset($this->arguments[self::newObjectName])) {
            if (isset($this->arguments[self::objectName])) {
                $object = $this->arguments[self::objectName];
            } else {
                $object = $this->arguments[self::newObjectName];
            }
            //var_dump(self::dateProperties);exit();
            
            //var_dump(self::newObjectName, self::dateProperties, $object);die('a');
            // Set date format of date fields
            foreach (self::dateProperties as $dateField) {
                $object->getPropertyMappingConfiguration()
                       ->forProperty($dateField)
                       ->setTypeConverterOption(
                           DateTimeConverter::class,
                           DateTimeConverter::CONFIGURATION_DATE_FORMAT,
                           'Y-m-d'
                       );
            }
        }
    }


    
    /**
     * action list
     */
    public function listAction(): ResponseInterface
    {
        $this->view->assign('search', $this->search);
        $this->assignOptions();
        $employees = $this->employeeRepository->findBySearch($this->search);
        $this->view->assign('employees', $employees);
        return $this->htmlResponse();
    }

    /**
     * action wageStatement
     */
    public function wageStatementAction(): ResponseInterface
    {
        $employees = $this->employeeRepository->findActive();
        $this->view->assign('employees', $employees);

        $salaryUtility = GeneralUtility::makeInstance(SalaryUtility::class);
        $salaries = $salaryUtility->getSalaries($employees->toArray(), '');
        $this->view->assign('salaries', $salaries);
        $tenant = $this->templateUtility->getTenant();
        $this->view->assign('tenant', $tenant);
        $this->view->assign('sender', $tenant);
        $letterDate = new DateTime();
        $this->view->assign('letterDate', $letterDate);
        $wageStatementDate = new DateTime();
        $this->view->assign('wageStatementDate', $wageStatementDate);
        return $this->htmlResponse();
    }
    
    /**
     * Monatliche Zeiterfassung
     */
    public function monthlyTimeRecordingAction(): ResponseInterface
    {
        $outputPdf = (bool)($this->request->hasArgument('pdf') ? $this->request->getArgument('pdf') : false);

        $search = $this->request->hasArgument('search') ? $this->request->getArgument('search') : [];
        $nowYear = (int) date('Y');
        $nowMonth = (int) date('m');
        $search['year'] = (int)($search['year'] ?? $nowYear);
        $search['month'] = (int)($search['month'] ?? $nowMonth);
        $this->view->assign('search', $search);

        $date = new DateTime();
        $yearMonths = $this->holidayRepository->getNaviArray($nowYear - 1, $nowYear, $nowMonth);
        $date->setDate($search['year'], $search['month'], 1);
        $targetDate = clone $date;
        $thisMonth = $this->holidayRepository->getTargetTimes($date, $this->personUtility->getWorkingDayInSeconds());
        $nextMonth = $this->holidayRepository->getTargetTimes($date, $this->personUtility->getWorkingDayInSeconds());



        $employees = $this->employeeRepository->findActive();

        $timeentries = [];
        foreach ($employees as $employee) {
            $targetTimesYearUpToMonth = $this->holidayRepository->getTargetTimesUpToMonth(
                $search['year'],
                $search['month'],
                $this->personUtility->getWorkingDayInSeconds(),
                $employee->getEntryDate()
            );
            $targetTimesYearUpToMonthBegin = $this->holidayRepository->getTargetTimesUpToMonth(
                $search['year'],
                $search['month'] - 1,
                $this->personUtility->getWorkingDayInSeconds(),
                $employee->getEntryDate()
            );


            $timeAmount = $this->timeRepository->getMonthTotalByEmployee($employee, $targetDate);
            $yearUpToMonthEndTimeAmount = $this->timeRepository->getYearUpToMonthEndTotalByEmployee(
                $employee,
                $targetDate
            );
            $yearUpToMonthBeginTimeAmount = $this->timeRepository->getYearUpToMonthBeginTotalByEmployee(
                $employee,
                $targetDate
            );

            $timeInSecondsInMonth = $timeAmount['timeInSeconds'] ?? 0;
            $timeInSecondsInYearUpToMonthEnd = $yearUpToMonthEndTimeAmount['timeInSeconds'] ?? 0;
            $timeInSecondsInYearUpToMonthBegin = $yearUpToMonthBeginTimeAmount['timeInSeconds'] ?? 0;
            $employeeMonthTargetTime = $thisMonth['targetTime'] * $employee->getEmploymentInPercent() / 100;

            $employeeCurrentTargetTime = $thisMonth['targetTime'] * $employee->getEmploymentInPercent() / 100;

            //$employeeUpToMonthTime = 0 * $employee->getEmploymentInPercent() / 100 + $employee->getTimeCarryover();

            //$employeeVacationDays = [];
            $employeeTotelVacationDays = $employee->getTotalHolidayDaysInYear();
            $employeeTakenVacationDaysTotal = $this->timeRepository->getVacationTotalForYearByEmployeeId(
                $employee->getUid(),
                (int)$targetDate->format('Y')
            );
            $employeeVacationDays = $this->timeRepository->getVacationForYearByEmployeeId(
                $employee->getUid(),
                (int)$targetDate->format('Y')
            );

            $workingDayInSeconds = 3600 * 8.4;
            $employeeTakenVacationDays = ($employeeTakenVacationDaysTotal['timeInSeconds'] ?? 0) / $workingDayInSeconds;
            $employeeOpenVacationDays = $employeeTotelVacationDays - $employeeTakenVacationDays;

            $timeentries[] = [
                'employee' => $employee,
                'employeeMonthTargetTime' => $employeeMonthTargetTime,
                'employeeVacationDays' => $employeeVacationDays,
                'employeeTotelVacationDays' => $employeeTotelVacationDays,
                'employeeTakenVacationDays' => $employeeTakenVacationDays,
                'employeeOpenVacationDays' => $employeeOpenVacationDays,
                'timeInSeconds' => $timeInSecondsInMonth,
                'timeInSecondsInYearUpToMonthEnd' => $timeInSecondsInYearUpToMonthEnd,
                'timeInSecondsInYearUpToMonthBegin' => $timeInSecondsInYearUpToMonthBegin,
                'balance' => [
                    'current' => $timeInSecondsInMonth - $employeeCurrentTargetTime,
                    'monthBeginEnd' => $timeInSecondsInMonth - $employeeMonthTargetTime,
                    'monthEnd' => $timeInSecondsInMonth - $employeeMonthTargetTime,
                    'flexitime' => [
                        'start' => $timeInSecondsInYearUpToMonthBegin - $targetTimesYearUpToMonthBegin,
                        'end' => $timeInSecondsInYearUpToMonthEnd - $targetTimesYearUpToMonth,
                    ],
                    //$Xzeit_jahr - $Xsoll_zeit_jahr*double($Pma_anstellung)/100  + int($Pma_zeit_uebertrag)

                ],
                'timeYear' => $timeInSecondsInYearUpToMonthEnd,
            ];

            /*
            if ($frontendUserUid == 1501) {
                var_dump($timeInSecondsInYearUpToMonthEnd,$timeInSecondsInYearUpToMonthBegin);
                die();
            }
            */
        }
        //var_dump($timeentries);exit(0);
        //$this->view->assign('employees', $employees);
        $this->view->assign('timeentries', $timeentries);
        $this->view->assign('thisMonth', $thisMonth);
        //var_dump($thisMonth);exit(0);
        $this->view->assign('nextMonth', $nextMonth);
        $this->view->assign('yearMonths', $yearMonths);
        $this->view->assign('outputPdf', $outputPdf);
        $tenant = $this->templateUtility->getTenant();
        $this->view->assign('tenant', $tenant);
        if ($outputPdf) {
            $this->templateUtility->setLanguageUid(1);
            $template = $this->templateUtility->getTemplate('Time');

            // TemplatePageLayout
            $templatePageLayout = $this->templateUtility->getTemplatePageLayout(
                'Time',
                TemplatePageLayout::PAGE_FORMART_A4,
                TemplatePageLayout::PAGE_ORIENTATION_PORTRAIT,
            );

            // html and css of css page media margin boxes
            $renderingContext = $this->view->getRenderingContext();

            $htmlAndCss = $this->templateUtility->renderCssPagedMediaMarginBoxs(
                $templatePageLayout,
                $renderingContext
            );

            $pdf = new PdfUtility();
            $pdf->setFilename('Monatszeiten-' . strftime('%Y-%m-%d') . '.pdf');
            $pdf->setTitle('Monatszeiten');
            $pdf->setConverter(PdfUtility::CONVERTER_WEASYPRINT);
             
            $pdf->addCssFile('EXT:ig_crm_time_recording/Resources/Public/Css/pdf.css');
            $pdf->addCss($templatePageLayout->getCss());
             
            $html = $this->view->render();

            //$pdf->debug($htmlAndCss . $html);
             
            $pdfContent = $pdf->get($htmlAndCss . $html);
            $response = $this->responseFactory->createResponse()
                                              ->withHeader('Expires', '0')
                                              ->withHeader('Cache-Control', 'private')
                                              ->withHeader('Content-Type', 'application/pdf; charset=utf-8')
                                              ->withHeader(
                                                  'Content-Disposition',
                                                  'attachment; filename="' . $pdf->getFilename() . '"'
                                              )
                                              ->withHeader('Content-Length', (string)strlen($pdfContent));
            $response->getBody()
                     ->write($pdfContent);
            throw new PropagateResponseException($response, 200);
        }
        
        return $this->htmlResponse();
    }
    public function analyticsAction(): ResponseInterface
    {
        $currentYear = date('Y');
        $year = (int)($this->request->hasArgument('year') ? (int) $this->request->getArgument('year') : $currentYear);
        $firstYear = (int)($this->settings['firstYear'] ?? 0);

        $employees = $this->employeeRepository->findActiveWithYear($year);
        $this->view->assign('employees', $employees);
        $employeesData = [];
        $totals = [];
        
        foreach ($employees as $employee) {
            $data = $this->timeRepository->getInternalForYearByEmployeeId($employee->getUid(), $year);
            if ($data['active'] === false) {
                continue;
            }
            $this->view->assign('header', $data['conf']);
            $employeesData[] = [
                'employee' => $employee,
                'cols' => $data['cols'],
            ];
            foreach ($data['cols'] as $col) {
                $attribute = $col['attribute'];

                if (!isset($totals[$attribute])) {
                    $totals[$attribute] = [
                        'value' => 0,
                        'percent' => 0,
                        'isActivity' => $col['isActivity'],
                    ];
                }
                $totals[$attribute]['value'] += $col['value'];
            }
        }
        $totalTimeInSeconds = $totals['timeInSeconds']['value'] ?? 0;
        if ($totalTimeInSeconds > 0) {
            foreach ($totals as &$total) {
                $total['percent'] = $total['value'] / $totalTimeInSeconds * 100;
            }
        }
        
        $years = [];
        $startYear = $year - 10;
        if ($startYear < $firstYear) {
            $startYear = $firstYear;
        }
        for ($y = $startYear; $y <= $currentYear; $y++) {
            $years[] = $y;
        }
        $this->view->assign('years', $years);
        $this->view->assign('year', $year);
        $this->view->assign('employeesData', $employeesData);
        $this->view->assign('totals', $totals);
        return $this->htmlResponse();
    }



    /**
     * action show
     */
    public function showAction(Employee $employee): ResponseInterface
    {
        $this->view->assign('employee', $employee);
        return $this->htmlResponse();
    }
    /**
     * action new
     */
    public function newAction(): ResponseInterface
    {
        $this->assignFormFields();
        $this->view->assign('search', $this->search);
        return $this->htmlResponse();
    }

    /**
     * action create
     */
    public function createAction(Employee $newEmployee): ResponseInterface
    {
        $this->addFlashMessage('Eintrag erstellt', '', ContextualFeedbackSeverity::INFO);
        $this->employeeRepository->add($newEmployee);
        $this->searchPrefix = 'search';
        return $this->redirectWithSearch('list');
    }

    /**
     * action edit
     */
    #[TYPO3\CMS\Extbase\Annotation\IgnoreValidation([
        'argumentName' => 'employee',
    ])]
    public function editAction(Employee $employee): ResponseInterface
    {
        $this->assignFormFields();
        $this->view->assign('search', $this->search);
        $this->view->assign('employee', $employee);
        return $this->htmlResponse();
    }

    /**
     * action update
     */
    public function updateAction(Employee $employee): ResponseInterface
    {
        $this->addFlashMessage('Eintrag aktualisiert', '', ContextualFeedbackSeverity::INFO);
        $this->employeeRepository->update($employee);
        $this->searchPrefix = 'search';
        return $this->redirectWithSearch('list');
    }

    public function confirmDeleteAction(Employee $employee): ResponseInterface
    {
        $this->assignFormFields();
        $this->view->assign('search', $this->search);
        $this->view->assign('employee', $employee);
        $times = $this->timeRepository->getTotalByEmployeeIdBetweendDates($employee->getUid());
        $this->view->assign('times', $times);
        return $this->htmlResponse();
    }

    public function deleteAction(Employee $employee): ResponseInterface
    {
        $this->addFlashMessage('Eintrag gelöscht', '', ContextualFeedbackSeverity::INFO);
        $this->employeeRepository->remove($employee);
        $this->searchPrefix = 'search';
        return $this->redirectWithSearch('list');
    }
    protected function assignOptions()
    {
        $activeOptions = [
            '' => 'Alle',
            '1' => 'Aktive',
            '0' => 'Inaktive',
        ];
        $genderOptions = [
            'm' => 'Herr',
            'f' => 'Frau',
        ];
        $this->view->assign('activeOptions', $activeOptions);
        $this->view->assign('genderOptions', $genderOptions);
    }
    protected function getSearch()
    {
        $search = [];
        if ($this->request->hasArgument('search')) {
            $search = $this->request->getArgument('search');
        } else {
            $search = $this->searchDefault;
        }
        return $search;
    }
    protected function assignFormFields()
    {
        $frontendUsers = $this->frontendUserRepository->findAll()
->toArray();
        $this->view->assign('frontendUsers', $frontendUsers);
        $contactConfig = $this->personUtility->getContactConfig((int)($this->settings['personPid'] ?? 0));
        $this->view->assign('contactConfig', $contactConfig);
        $this->assignOptions();
        $countryRepository = GeneralUtility::makeInstance(CountryRepository::class);
        $countries = $countryRepository->getIsoCountryName();
        $this->view->assign('countries', $countries);
    }
}
