<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Database\Query;

use DateTime;
use Internetgalerie\IgCrmTimeRecording\Utility\PersonUtility;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\Query\QueryBuilder;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;

class TimeQueryBuilder extends QueryBuilder
{
    protected $tablename = 'tx_igcrmtimerecording_domain_model_time';
    protected $activityTablename = 'tx_igcrmtimerecording_domain_model_activity';
    protected $defaultOrderings = [
        'start_date' => QueryInterface::ORDER_DESCENDING,
        'duration' => QueryInterface::ORDER_ASCENDING,
    ];

    protected ?PersonUtility $personUtility = null;
    protected bool $activityJoined = false;
    protected bool $employeeJoined = false;


    /**
     * use acl for queries
     */
    protected bool $useAcl = true;

    /**
     * Select
     */
    protected string $sqlSelect = '*';

    public function __construct(
        Connection $connection,
        QueryRestrictionContainerInterface $restrictionContainer = null,
        \Doctrine\DBAL\Query\QueryBuilder $concreteQueryBuilder = null,
        array $additionalRestrictions = null
    ) {
        parent::__construct($connection, $restrictionContainer, $concreteQueryBuilder, $additionalRestrictions);
        $this->personUtility = GeneralUtility::makeInstance(PersonUtility::class);
    }

    public function getUseAcl()
    {
        return $this->useAcl;
    }
    public function setUseAcl(bool $useAcl)
    {
        return $this->useAcl = $useAcl;
    }
    
    public function setDefaultOrderings(array $defaultOrderings)
    {
        return $this->defaultOrderings = $defaultOrderings;
    }
    public function addUserRestriction()
    {
        $constraints = [];
        if (!$this->personUtility->hasRoleAdmin() && $this->useAcl) {
            $employee = $this->personUtility->getFrontendUserEmployee();
            $constraints[] = $this->expr()->eq('employee_id', $employee ? $employee->getUid() : 0);
        }
        if (!empty($constraints)) {
            $this->andWhere(...$constraints);
        }
        return $this;
    }

    public function resetSelect()
    {
        $this->sqlSelect = '*';
        return $this;
    }

    public function joinActivity()
    {
        if (!$this->activityJoined) {
            $this->activityJoined = true;
            return $this->leftJoin(
                'tx_igcrmtimerecording_domain_model_time',
                'tx_igcrmtimerecording_domain_model_activity',
                'tx_igcrmtimerecording_domain_model_activity',
                $this->expr()
->eq('activity', $this->quoteIdentifier('tx_igcrmtimerecording_domain_model_activity.uid')),
            );
        }
    }

    public function joinEmployee()
    {
        if (!$this->employeeJoined) {
            $this->employeeJoined = true;
            return $this->leftJoin(
                'tx_igcrmtimerecording_domain_model_time',
                'tx_igcrmtimerecording_domain_model_employee',
                'tx_igcrmtimerecording_domain_model_employee',
                $this->expr()
->eq('employee_id', $this->quoteIdentifier('tx_igcrmtimerecording_domain_model_employee.uid')),
            );
        }
    }

    
    public function addSearch($search)
    {
        $constrains = [];
        if (isset($search['uid']) && $search['uid'] !== '') {
            $constrains[] = $this->expr()->eq($this->tablename . '.uid', trim((string) $search['uid']));
        }
        if ($search['activity'] ?? false) {
            $constrains[] = $this->expr()->eq($this->tablename . '.activity', $search['activity']);
        }
        if ($search['employeeId'] ?? false) {
            $constrains[] = $this->expr()->eq($this->tablename . '.employee_id', $search['employeeId']);
        }
        if ($search['isInternal'] ?? false) {
            $constrains[] = $this->expr()->eq($this->activityTablename . '.is_internal', (int)$search['isInternal']);
        }
        if ($search['tenant'] ?? false) {
            $constrains[] = $this->expr()->eq($this->tablename . '.tenant_id', $search['tenant']);
        }
        if ($search['internalPurpose'] ?? false) {
            $this->joinActivity();
            $constrains[] = $this->expr()->eq(
                $this->activityTablename . '.internal_purpose',
                (int)$search['internalPurpose']
            );
        }
        if ($search['internalPurposeIn'] ?? false && !empty(\SEARCH['internalPurposeIn'])) {
            $this->joinActivity();
            $constrains[] = $this->expr()->in(
                $this->activityTablename . '.internal_purpose',
                $search['internalPurposeIn']
            );
        }
        if (isset($search['startDate'])) {
            $startDate = is_array($search['startDate']) ? $search['startDate']['date'] : $search['startDate'];
            if ($startDate) {
                $constrains[] = $this->expr()->gte(
                    $this->tablename . '.start_date',
                    $this->createNamedParameter($startDate)
                );
            }
        }
        if (isset($search['endDate'])) {
            $endDate = is_array($search['endDate']) ? $search['endDate']['date'] : $search['endDate'];
            if ($endDate) {
                $datetime = new DateTime($endDate);
                $datetime->modify('+1 day');
                $constrains[] = $this->expr()->lte(
                    $this->tablename . '.start_date',
                    $this->createNamedParameter($datetime->format('Y-m-d'))
                );
            }
        }



        $year = (int)($search['year'] ?? 0);
        if ($year > 1900) {
            $month = (int)($search['month'] ?? 0);
            if ($month > 0) {
                $date = new DateTime();
                $date->setDate($year, $month, 1);
                $constrains[] = $this->expr()->gte(
                    $this->tablename . '.start_date',
                    $this->createNamedParameter($date->format('Y-m-d'))
                );
                $date->modify('+1 month');
                $constrains[] = $this->expr()->lt(
                    $this->tablename . '.start_date',
                    $this->createNamedParameter($date->format('Y-m-d'))
                );
            } else {
                $constrains[] = $this->expr()->gte(
                    $this->tablename . '.start_date',
                    $this->createNamedParameter($year . '-01-01')
                );
                $constrains[] = $this->expr()->lt(
                    $this->tablename . '.start_date',
                    $this->createNamedParameter(($year + 1) . '-01-01')
                );
            }
        }

        if ($search['keyword'] ?? false) {
            $keywordWords = array_filter(explode(' ', (string) $search['keyword']));
            foreach ($keywordWords as $keyword) {
                $constrains[] = $this->expr()->or($this->expr()
                     ->like('description', $this->createNamedParameter('%' . $keyword . '%')));
            }
        }
        $this->addArrayOrderBy($this->defaultOrderings);
        
        if (!empty($constrains)) {
            $this->andWhere(...$constrains);
        }
        return $this;
    }
    public function addArrayOrderBy(array $orderBys)
    {
        foreach ($orderBys as $attribute => $order) {
            $this->addOrderBy($attribute, $order);
        }
        return $this;
    }
    public function addLimit($search)
    {
        $limit = $search['limit'] ?? 0;
        if ($limit > 0) {
            $this->setMaxResults($limit);
        }
        $limit = $search['offset'] ?? 0;
        if ($offset > 0) {
            $this->setFirstResult($offset);
        }
        return $this;
    }
}
