<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Domain\Repository;
use TYPO3\CMS\Core\Database\Connection;

use DateTime;
use Internetgalerie\IgCrmTimeRecording\Database\Query\TimeQueryBuilder;
use Internetgalerie\IgCrmTimeRecording\Domain\Model\Employee;
use Internetgalerie\IgCrmTimeRecording\Domain\Model\Time;
use Internetgalerie\IgCrmTimeRecording\Utility\DateUtility;
use Internetgalerie\IgCrmTimeRecording\Utility\PersonUtility;
use PDO;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use TYPO3\CMS\Extbase\Persistence\Repository;

/**
 * The repository for Times
 */
class TimeRepository extends Repository
{
    protected $tablename = 'tx_igcrmtimerecording_domain_model_time';
    protected $activityTablename = 'tx_igcrmtimerecording_domain_model_activity';
    protected $useAcl = false;


    protected $defaultOrderings = [
        'start_date' => QueryInterface::ORDER_DESCENDING,
        'duration' => QueryInterface::ORDER_ASCENDING,
    ];


    /**
     * @var DataMapper
     **/
    protected $dataMapper;

    
    public function injectDataMapper(DataMapper $dataMapper): void
    {
        $this->dataMapper = $dataMapper;
    }


    /**
     * return a queryBuilder for times
     **/
    public function createTimeQueryBuilder(): TimeQueryBuilder
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable($this->tablename);
        $queryBuilder = GeneralUtility::makeInstance(TimeQueryBuilder::class, $conn);
        $queryBuilder->setDefaultOrderings($this->defaultOrderings);
        $queryBuilder->setUseAcl($this->useAcl);
        return $queryBuilder;
    }
    
    public function findBySearch(array $search = [], array $orderBy = []): TimeQueryBuilder
    {
        $queryBuilder = $this->createTimeQueryBuilder();
        $queryBuilder->addUserRestriction();
        $queryBuilder->select($this->tablename . '.*');
        $queryBuilder->from($this->tablename);
        $queryBuilder->addSearch($search);
        return $queryBuilder;
    }
    public function findBySearchRaw(array $search = [], array $orderBy = []): TimeQueryBuilder
    {
        $queryBuilder = $this->findBySearch($search, $orderBy);
        $queryBuilder->joinActivity();
        $queryBuilder->select('tx_igcrmtimerecording_domain_model_time.*', 'name', 'amount');
        return $queryBuilder;
    }
    public static function prepareRows(array $rows, bool $dateAsString = false): array
    {
        $attributes = [
            'uid',
            'employee_id',
            'start_date',
            'duration',
            'activity',
            'description',
            'offer_id',
            'workflow_id',
        ];
        $employees = [];
        $results = [];
        foreach ($rows as $row) {
            $result = [];
            foreach ($attributes as $attribute) {
                $result[GeneralUtility::underscoredToLowerCamelCase($attribute)] = $row[$attribute];
            }
            $startDate = new DateTime($result['startDate']);
            $endDate = clone $startDate;
            $endDate->modify('+' . $row['duration'] . 'seconds');
            if ($dateAsString) {
                $result['startDate'] = $startDate->format('c');
                $result['endDate'] = $endDate->format('c');
            } else {
                $result['startDate'] = $startDate;
                $result['endDate'] = $endDate;
            }
            $result['durationTime'] = DateUtility::secondsToHMMSS($result['duration']);
            // @todo speedUp
            $activityUid = (int)$result['activity'];
            //$activity = $this->activityRepository->findOneRawByUid($activityUid);
            $result['activity'] = [
                'uid' => $activityUid,
                'name' => $row['name'],
                'amount' => $row['amount'],
            ];
            $employeeUid = (int)$result['employeeId'];
            if (!isset($employees[$employeeUid])) {
                $employees[$employeeUid] = EmployeeRepository::getRawByUid($employeeUid);
            }
            $result['employee'] = $employees[$employeeUid];
            $result['aclOwner'] = $result['employee']['frontend_user'] ?? null;
            $results[] = $result;
        }
        return $results;
    }
    
    public function findBySearchExecuteRaw(array $search = [], array $orderBy = [], bool $dateAsString = false)
    {
        $queryBuilder = $this->findBySearchRaw($search, $orderBy);
        $res = $queryBuilder->executeQuery();
        $rows = $res->fetchAllAssociative();
        $results = static::prepareRows($rows, true);
        return $results;
    }
    
    public function findBySearchExecute(array $search = [], $orderBy = [])
    {
        $queryBuilder = $this->findBySearch($search, $orderBy);
        $rows = $queryBuilder->executeQuery()
                             ->fetchAllAssociative();
        return $this->dataMapper->map($this->objectType, $rows);
    }

    public function addAndPersist(Time $time): void
    {
        $this->add($time);
        $this->persistenceManager->persistAll();
    }
        
    public function isConflict(Time $time): bool
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable($this->tablename);
        $res = $queryBuilder
            ->selectLiteral('1')
            ->from($this->tablename)
            ->where(
                'start_date < ' . $queryBuilder->createNamedParameter($time->getEndDate()->format('Y-m-d H:i:s')),
                'DATE_ADD(start_date, INTERVAL duration second) > ' . $queryBuilder->createNamedParameter(
                    $time->getStartDate()->format('Y-m-d H:i:s')
                ),
                'employee_id = ' . (int)$time->getEmployeeId(),
            )
             ->executeQuery();
        $row = $res->fetchFirstColumn();
        return (bool)($row[0] ?? false);
    }

    public function findOverviewBySearch(array $search = [])
    {
        $view = $search['view'] ?? 'day';
        $queryBuilder = $this->createTimeQueryBuilder();
        $queryBuilder->addUserRestriction();
        if ($view == 'year') {
            $dateFunction = 'YEAR';
        } elseif ($view == 'month') {
            $dateFunction = 'MONTH';
        } elseif ($view == 'week') {
            $dateFunction = 'WEEK';
        } else {
            $dateFunction = 'DATE';
        }
        
        $queryBuilder->selectLiteral(
            $queryBuilder->expr()
                         ->sum('duration', 'sum_duration'),
            'YEAR(start_date) AS date_year',
            $dateFunction . '(start_date) AS date_group',
            'DATE(MIN(start_date)) AS date_min',
            'DATE(MAX(start_date)) AS date_max',
            'SUM(duration /3600 * amount) AS sum_amount'
        );
        $queryBuilder->from($this->tablename);
        $queryBuilder->addSearch($search);
        $queryBuilder->leftJoin(
            'tx_igcrmtimerecording_domain_model_time',
            'tx_igcrmtimerecording_domain_model_activity',
            'tx_igcrmtimerecording_domain_model_activity',
            $queryBuilder->expr()
                         ->eq(
                             'activity',
                             $queryBuilder->quoteIdentifier('tx_igcrmtimerecording_domain_model_activity.uid')
                         )
        );
        $queryBuilder->groupBy('date_year');
        $queryBuilder->addGroupBy('date_group');
        $queryBuilder->orderBy('date_min', 'ASC');
        $rows = $queryBuilder->executeQuery()
                             ->fetchAllAssociative();
        return $rows;
    }
    public function getDayTotalByEmployeeId(int $employeeId, DateTime $date = null)
    {
        if ($date === null) {
            $date = new DateTime();
        }
        $queryBuilder = $this->createTimeQueryBuilder();
        //$queryBuilder->addUserRestriction();
        $queryBuilder->selectLiteral(
            $queryBuilder->expr()
->sum('duration', 'timeInSeconds'),
            'SUM(duration * amount / 3600) AS amount'
        );
        $queryBuilder->from($this->tablename);
        $queryBuilder->leftJoin(
            'tx_igcrmtimerecording_domain_model_time',
            'tx_igcrmtimerecording_domain_model_activity',
            'tx_igcrmtimerecording_domain_model_activity',
            $queryBuilder->expr()
->eq('activity', $queryBuilder->quoteIdentifier('tx_igcrmtimerecording_domain_model_activity.uid'))
        )
            
                     ->where(
                         $queryBuilder->expr()
->eq('employee_id', $queryBuilder->createNamedParameter($employeeId, Connection::PARAM_INT)),
                         'DATE(start_date) = ' . $queryBuilder->createNamedParameter($date->format('Y-m-d')),
                     );
        $row = $queryBuilder->executeQuery()
->fetchAssociative();
        return [
            'timeInSeconds' => (int)$row['timeInSeconds'],
            'amount' => (float)$row['amount'],
        ];
    }

    /**
     * get total time and amount for given employee and month
     */
    public function getMonthTotalByEmployeeId(int $employeeId, DateTime $date = null)
    {
        if ($date === null) {
            $date = new DateTime();
        }
        $firstDayOfMonth = $date->format('Y-m-01');
        $lastDayOfMonth = $date->format('Y-m-t');
        return $this->getTotalByEmployeeIdBetweendDates($employeeId, $firstDayOfMonth, $lastDayOfMonth);
    }
    /**
     * get total time and amount for given employee and month
     */
    public function getMonthTotalByEmployee(Employee $employee, DateTime $date = null)
    {
        if ($date === null) {
            $date = new DateTime();
        }
        $firstDayOfMonth = $date->format('Y-m-01');
        $lastDayOfMonth = $date->format('Y-m-t');
        $employeeEntryDate = $employee->getEntryDate();
        $employeeEntryDateString = $employeeEntryDate ? $employeeEntryDate->format('Y-m-d') : null;
        return $this->getTotalByEmployeeIdBetweendDates(
            $employee->getUid(),
            $firstDayOfMonth,
            $lastDayOfMonth,
            true,
            $employeeEntryDateString
        );
    }


    /**
     * get total time and amount for given employee and up and including month of given date in this year
     */
    public function getYearUpToMonthEndTotalByEmployee(Employee $employee, DateTime $date = null)
    {
        if ($date === null) {
            $date = new DateTime();
        }
        $firstDayOfYear = $date->format('Y-01-01');
        $lastDayOfMonth = $date->format('Y-m-t');
        $employeeEntryDate = $employee->getEntryDate();
        $employeeEntryDateString = $employeeEntryDate ? $employeeEntryDate->format('Y-m-d') : null;
        return $this->getTotalByEmployeeIdBetweendDates(
            $employee->getUid(),
            $firstDayOfYear,
            $lastDayOfMonth,
            true,
            $employeeEntryDateString
        );
    }
    /**
     * get total time and amount for given employee and up and excluding month of given date in this year
     */
    public function getYearUpToMonthBeginTotalByEmployee(Employee $employee, DateTime $date = null)
    {
        if ($date === null) {
            $date = new DateTime();
        }
        $firstDayOfYear = $date->format('Y-01-01');
        $lastDayOfMonth = $date->format('Y-m-01');
        //var_dump($firstDayOfYear, $lastDayOfMonth);exit(0);
        
        $employeeEntryDate = $employee->getEntryDate();
        $employeeEntryDateString = $employeeEntryDate ? $employeeEntryDate->format('Y-m-d') : null;
        return $this->getTotalByEmployeeIdBetweendDates(
            $employee->getUid(),
            $firstDayOfYear,
            $lastDayOfMonth,
            false,
            $employeeEntryDateString
        );
    }
    public function getVacationTotalForYearByEmployeeId(int $employeeId, int $year = null): array
    {
        if ($year === null) {
            $year = (int)date('Y');
        }
        $firstDay = $year . '-01-01';
        $lastDay = $year . '-12-31';
        $queryBuilder = $this->getTotalQueryBuilder();
        $queryBuilder->where(
            $queryBuilder->expr()
->eq('employee_id', $queryBuilder->createNamedParameter($employeeId, Connection::PARAM_INT)),
            'DATE(start_date) >= ' . $queryBuilder->createNamedParameter($firstDay),
            'DATE(start_date) <= ' . $queryBuilder->createNamedParameter($lastDay),
            'internal_purpose = ' . PersonUtility::$activityInternalPurposeVacation,
            //'is_vacation = TRUE',
        );

        return $this->executeTotal($queryBuilder);
    }
    public function getVacationForYearByEmployeeId(int $employeeId, int $year = null): array
    {
        if ($year === null) {
            $year = (int)date('Y');
        }
        $firstDay = $year . '-01-01';
        $lastDay = $year . '-12-31';

        $search = [
            'employeeId' => $employeeId,
            'startDate' => $firstDay,
            'endDate' => $lastDay,
            'internalPurpose' => PersonUtility::$activityInternalPurposeVacation,
        ];
        
        $rows = $this->findBySearchExecute($search);
        return $rows;
    }



    public function getTotalQueryBuilder(): TimeQueryBuilder
    {
        $queryBuilder = $this->createTimeQueryBuilder();
        //$queryBuilder->addUserRestriction();
        $queryBuilder->selectLiteral(
            $queryBuilder->expr()
->sum('duration', 'timeInSeconds'),
            'SUM(duration * amount / 3600) AS amount'
        );
        $queryBuilder->from($this->tablename);
        $queryBuilder->joinActivity();
        return $queryBuilder;
    }

    public function executeTotal(TimeQueryBuilder $queryBuilder): array
    {
        $row = $queryBuilder->executeQuery()
->fetchAssociative();
        return [
            'timeInSeconds' => (int)$row['timeInSeconds'],
            'amount' => (float)$row['amount'],
        ];
    }
    /**
     * get total time and amount for given employee between given dates
     * if it is a reentry employeeStartDate ensures that only new times are counted
     */
    public function getTotalByEmployeeIdBetweendDates(
        int $employeeId,
        string $firstDay = null,
        string $lastDay = null,
        bool $includingLastDay = true,
        ?string $employeeStartDate = null
    ) {
        $queryBuilder = $this->getTotalQueryBuilder();

        $constraints = [
            $queryBuilder->expr()
->eq('employee_id', $queryBuilder->createNamedParameter($employeeId, Connection::PARAM_INT)),
        ];
        if ($firstDay) {
            $constraints[] = 'DATE(start_date) >= ' . $queryBuilder->createNamedParameter($firstDay);
        }
        if ($lastDay) {
            $constraints[] = 'DATE(start_date) ' . ($includingLastDay ? '<=' : '<') . ' ' . $queryBuilder->createNamedParameter(
                $lastDay
            );
        }
        // only after entry date of employee
        if ($employeeStartDate !== null && $employeeStartDate !== '') {
            $constraints[] = 'DATE(start_date) >=' . $queryBuilder->createNamedParameter($employeeStartDate);
        }
        $queryBuilder->where(...$constraints);
        return $this->executeTotal($queryBuilder);
    }
    public function getInternalForYearByEmployeeId(int $employeeId, int $year = null): array
    {
        if ($year === null) {
            $year = (int)date('Y');
        }
        $firstDay = $year . '-01-01';
        $lastDay = $year . '-12-31';

        // SELECT uid,name FROM tx_igcrmtimerecording_domain_model_activity WHERE active = TRUE AND is_internal = TRUE ORDER BY sorting;
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            $this->activityTablename
        );
        $res = $queryBuilder->select('uid', 'name', 'sorting')
                     ->from($this->activityTablename)
                     ->where('active = TRUE AND is_internal = TRUE')
                     ->orderBy('sorting')
                     ->executeQuery();
        $activitySelect = [];
        $activityConf = [];
        $emptyData = [
            'duration' => 0,
        ];
        while ($activity = $res->fetchAssociative()) {
            $activitySelect[] = 'SUM(CASE WHEN activity=' . $activity['uid'] . ' THEN duration ELSE 0 END) AS activity_' . $activity['uid'];
            $attribute = 'activity_' . $activity['uid'];
            $activityConf[] = [
                'uid' => $activity['uid'],
                'name' => $activity['name'],
                'attribute' => $attribute,
            ];
            $emptyData[] = [
                'attribute' => $attribute,
                'value' => 0,
                'percent' => 0,
                'isActivity' => true,
            ];
        }
        $queryBuilder = $this->createTimeQueryBuilder();
        $queryBuilder->selectLiteral($queryBuilder->expr() ->sum('duration', 'timeInSeconds'));
        $queryBuilder->from($this->tablename);

        //$queryBuilder->selectLiteral( queryBuilder->expr()->sum('duration', 'timeInSeconds'));

        $queryBuilder->addSelectLiteral(implode(',', $activitySelect));
        $queryBuilder->where(
            $queryBuilder->expr()
->eq('employee_id', $queryBuilder->createNamedParameter($employeeId, Connection::PARAM_INT)),
            'DATE(start_date) >= ' . $queryBuilder->createNamedParameter($firstDay),
            'DATE(start_date) <= ' . $queryBuilder->createNamedParameter($lastDay),
            //'is_internal = TRUE',
        );
        $queryBuilder->groupBy('employee_id');

        $cols = $queryBuilder->executeQuery()
->fetchAssociative();

        if ($cols === false) {
            $active = false;
            $data = $emptyData;
        } else {
            $active = true;
            $total = (float)$cols['timeInSeconds'];
            $data = [];
            foreach ($cols as $attribute => $colValue) {
                $data[] = [
                    'attribute' => $attribute,
                    'value' => $colValue,
                    'percent' => $colValue / $total * 100,
                    'isActivity' => str_starts_with($attribute, 'activity_'),

                ];
            }
        }
        return [
            'conf' => $activityConf,
            'cols' => $data,
            'active' => $active,
        ];

        /*
<table bgcolor="#000000">
<tr bgcolor="f0f0f0">
<td><strong>Jahr</strong></td>
<td><strong>Wer</strong></td>
<td colspan="2"><strong>Total</strong></td>
<%LOOPSQL ("SELECT * FROM taetigkeiten WHERE at_active = TRUE AND at_mode=1 ORDER BY at_was") {
<td align="right"><%= int(Ptotal/3600) %>h</td>
<td align="right" bgcolor="fff060"><%= number_format((Ptotal/3600/((Dj-Dj*0.03)/7*5))/8.24*100,2) %>%</td>
<%LOOP ($Xtaetigkeiten_loop, "," ) {
<%Xwert = int($PP($P)) %>
<td align="right"><%= int($Xwert/3600) %>h</td>
<td  align="right" bgcolor="fff060"><%= int($Xwert *100/int($PPtotal)) %>%</td>
<%X("total_"+$P)+= $Xwert %>
}%>
</tr>
}%>
        */
    }
    public function getTotalByObjects(array $times): array
    {
        $total = [
            'time' => 0,
            'amount' => 0,
            'activities' => [],
        ];
        $lastEmployeeId = 0;
        $lastEndDate = null;
        $lastStartDate = null;
        foreach ($times as $time) {
            $total['time'] += $time->getDuration();
            $activityId = $time->getActivityId();
            if (!isset($total['activities'][$activityId])) {
                $total['activities'][$activityId] = [
                    'name' => $time->getActivity()
                        ->getName(),
                    'rate' => $time->getActivity()
                        ->getAmount(),
                    'total' => 0,
                    'amount' => 0,
                ];
            }
            $amount = $time->getDuration() * $total['activities'][$activityId]['rate'] / 3600;
            $total['amount'] += $amount;
            $total['activities'][$activityId]['total'] += $time->getDuration();
            // Zeit ist ueberschnitten
            if ($lastEmployeeId === $time->getEmployeeId() && $lastEndDate !== null &&
                (
                    ($time->getStartDate() < $lastEndDate && $time->getStartDate() > $lastStartDate) || ($time->getEndDate() > $lastStartDate && $time->getEndDate() < $lastEndDate)
                )) {
                $time->setIsError(true);
            }
            $lastEmployeeId = $time->getEmployeeId();
            $lastStartDate = $time->getStartDate();
            $lastEndDate = $time->getEndDate();
        }
        return $total;
    }

    public function getTotalByArray(array &$times): array
    {
        $total = [
            'time' => 0,
            'amount' => 0,
            'activities' => [],
        ];
        $lastEmployeeId = 0;
        $lastEndDate = null;
        $lastStartDate = null;
        foreach ($times as &$time) {
            $total['time'] += $time['duration'];
            $activityId = $time['activity']['uid'];
            if (!isset($total['activities'][$activityId])) {
                $total['activities'][$activityId] = [
                    'name' => $time['activity']['name'],
                    'rate' => $time['activity']['amount'],
                    'total' => 0,
                    'amount' => 0,
                ];
            }
            $amount = $time['duration'] * $total['activities'][$activityId]['rate'] / 3600;
            $total['amount'] += $amount;
            $total['activities'][$activityId]['total'] += $time['duration'];
            // Zeit ist ueberschnitten
            if ($lastEmployeeId === $time['employeeId'] && $lastEndDate !== null &&
                (
                    ($time['startDate'] < $lastEndDate && $time['startDate'] > $lastStartDate) || ($time['endDate'] > $lastStartDate && $time['endDate'] < $lastEndDate)
                )) {
                $time['isError'] = true;
            } else {
                $time['isError'] = false;
            }
            $lastEmployeeId = $time['employeeId'];
            $lastStartDate = $time['startDate'];
            $lastEndDate = $time['endDate'];
        }
        return $total;
    }
}
