<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Domain\Repository;

use DateTime;
use IntlDateFormatter;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Repository;

/**
 * The repository for Holidays
 */
class HolidayRepository extends Repository
{
    public static $tablename = 'tx_igcrmtimerecording_domain_model_holiday';
    protected $holidays = [];
    protected ?IntlDateFormatter $formatter = null;

    public function findByYear(int $year)
    {
        $query = $this->createQuery();
        $query->matching(
            $query->logicalAnd(
                $query->greaterThanOrEqual('date', $year . '-01-01'),
                $query->lessThanOrEqual('date', $year . '-12-31'),
            )
        );
        return $query->execute();
        /*
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(self::$tablename);
        $res = $queryBuilder
            ->select('*')
            ->from(self::$tablename)
            ->where("`date` >= '" . $year . "-01-01' AND `date` <= '" . $year . "-12-31'" )
            ->executeQuery();
        return  $res->fetchAllAssociative();
        */
    }

    public function getHolidaysByYear(int $year)
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(self::$tablename);
        $res = $queryBuilder
            ->select('date')
            ->from(self::$tablename)
            ->where('YEAR(`date`)=' . $year)
             ->executeQuery();
        return $res->fetchFirstColumn();
        //SELECT date FROM holiday WHERE date_part('year',date) = $year;
    }
    public function getTargetTimesByYear(int $year, int $timePerDay)
    {
        $date = new DateTime($year . '-01-01');
        $totalTargetDays = 0;
        $data = [];
        while ((int)$date->format('Y') == $year) {
            $monthData = $this->getMonthData($date);
            $targetDays = $this->getTargetDaysForMonth($date);
            $monthData['targetDays'] = $targetDays;
            $monthData['targetTime'] = $targetDays * $timePerDay;
            $data[] = $monthData;
            $totalTargetDays += $targetDays;
        }
        return [
            'months' => $data,
            'total' => [
                'name' => 'Total',
                'targetDays' => $totalTargetDays,
                'targetTime' => $totalTargetDays * $timePerDay,
            ],
        ];
    }

    public function getTargetTimesByMonth(int $year, int $month, int $timePerDay)
    {
        $date = new DateTime();
        $date->setDate($year, $month, 1);
        $monthData = $this->getMonthData($date);
        $targetDays = $this->getTargetDaysForMonth($date);
        $monthData['targetDays'] = $targetDays;
        $monthData['targetTime'] = $targetDays * $timePerDay;
        return $monthData;
    }
    public function getTargetDaysUpToMonth(int $year, int $month, ?DateTime $employeeEntryDate = null)
    {
        if ($month === 0) {
            return 0;
        }
        $date = new DateTime();
        $date->setDate($year, 1, 1);
        if ($date < $employeeEntryDate) {
            $date = clone $employeeEntryDate;
        }
        
        $endDate = new DateTime();
        $endDate->setDate($year, $month, 1);
        $targetDays = 0;
        $targetTime = 0;
        while ($date <= $endDate) {
            $targetDays += $this->getTargetDaysForMonth($date);
        }
        return $targetDays;
    }
        
    public function getTargetTimesUpToMonth(
        int $year,
        int $month,
        int $timePerDay,
        ?DateTime $employeeEntryDate = null
    ) {
        $targetDays = $this->getTargetDaysUpToMonth($year, $month, $employeeEntryDate);
        return $targetDays * $timePerDay;
    }
    public function getTargetTimes(DateTime $date, int $timePerDay)
    {
        $targetDays = 0;
        $year = (int)$date->format('Y');
        $month = (int)$date->format('m');
        $this->init($year);
        $monthName = $this->formatter->format($date);
        $targetDaysLastMonth = 0;
        do {
            $currentDate = $date->format('Y-m-d');
            $weekday = $date->format('w');
        

            // only weekdays
            if ($weekday > 0 && $weekday < 6 && !in_array($currentDate, $this->holidays[$year])) {
                $targetDays++;
            }
            
            $date->modify('+1 day');
        } while ((int)$date->format('d') != 1);
        return [
            'number' => $month,
            'name' => $monthName,
            'year' => $year,
            'targetDays' => $targetDays - $targetDaysLastMonth,
            'targetTime' => ($targetDays - $targetDaysLastMonth) * $timePerDay,
        ];
    }

    /**
     * get target days for given month (from given date on)
     * increase date to 1. of next month
     */
    public function getTargetDaysForMonth(DateTime $date): int
    {
        $targetDays = 0;
        $year = (int)$date->format('Y');
        $this->init($year);
        do {
            $currentDate = $date->format('Y-m-d');
            $weekday = $date->format('w');
        

            // only weekdays
            if ($weekday > 0 && $weekday < 6 && !in_array($currentDate, $this->holidays[$year])) {
                $targetDays++;
            }
            
            $date->modify('+1 day');
        } while ((int)$date->format('d') != 1);
        return $targetDays;
    }

    public function getMonthData(DateTime $date): array
    {
        $year = (int)$date->format('Y');
        $this->init($year);
        $month = (int)$date->format('m');
        $monthName = $this->formatter->format($date);
        return [
            'date' => $date,
            'number' => $month,
            'name' => $monthName,
            'year' => $year,
        ];
    }
    public function getNaviArray(int $fromYear, int $toYear, int $toMonth): array
    {
        $date = new DateTime();
        $this->init();
        $yearMonths = [];
        for ($y = $fromYear; $y <= $toYear; $y++) {
            $yearMonths[$y] = [];
            for ($m = 1; $m <= 12 && ($y < $toYear || $m <= $toMonth); $m++) {
                $date->setDate($y, $m, 1);
                $yearMonths[$y][$m] = $this->formatter->format($date);
            }
        }
        return $yearMonths;
    }
    public static function isHoliday(DateTime $dateTime): bool
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(self::$tablename);
        $res = $queryBuilder
            ->selectLiteral('1')
            ->from(self::$tablename)
            ->where('`date` =' . $queryBuilder->createNamedParameter($dateTime->format('Y-m-d')))
             ->executeQuery();
        $row = $res->fetchFirstColumn();
        return (bool)($row[0] ?? false);
    }
    protected function init($year = null)
    {
        if ($year !== null && !isset($this->holidays[$year])) {
            $this->holidays[$year] = $this->getHolidaysByYear($year);
        }
        if ($this->formatter === null) {
            $this->formatter = IntlDateFormatter::create('de-DE', IntlDateFormatter::FULL, IntlDateFormatter::NONE);
            $this->formatter->setPattern('MMMM');
        }
    }
}
