<?php

declare(strict_types=1);

namespace Internetgalerie\IgCrmTimeRecording\Utility;

use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;

/**
 * import holidays from https://date.nager.at/Api
 */
class ImportHolidayUtility
{
    protected static array $countries = [
        [
            'countryCode' => 'CH',
            'name' => 'Schweiz',
        ],
    ];
    public static array $counties = [
        'CH-AG' => 'Aargau',
        'CH-AI' => 'Appenzell Innerrhoden',
        'CH-AR' => 'Appenzell Ausserrhoden',
        'CH-BE' => 'Bern',
        'CH-BL' => 'Basel-Landschaft',
        'CH-BS' => 'Basel-Stadt',
        'CH-FR' => 'Freiburg',
        'CH-GE' => 'Genf',
        'CH-GL' => 'Glarus',
        'CH-GR' => 'Graubünden',
        'CH-JU' => 'Jura',
        'CH-LU' => 'Luzern',
        'CH-NE' => 'Neuenburg',
        'CH-NW' => 'Nidwalden',
        'CH-OW' => 'Obwalden',
        'CH-SG' => 'St. Gallen',
        'CH-SH' => 'Schaffhausen',
        'CH-SO' => 'Solothurn',
        'CH-SZ' => 'Schwyz',
        'CH-TG' => 'Thurgau',
        'CH-TI' => 'Tessin',
        'CH-UR' => 'Uri',
        'CH-VD' => 'Waadt',
        'CH-VS' => 'Wallis',
        'CH-ZG' => 'Zug',
        'CH-ZH' => 'Zürich',
    ];

    public static function getHolidays(int $year = 0, string $country = 'CH', ?string $county = null, bool $excludeWeekends = true): array
    {
        if ($year == 0){
            $year = (int)date('Y');
        }
        if (strlen($country) != 2) {
            return [];
        }

        $apiUrl = 'https://date.nager.at/api/v3/PublicHolidays/' . $year . '/' . $country;
        $json = @file_get_contents($apiUrl);
        if ($json === false) {
            throw new \RuntimeException("Failed to fetch holidays from {$apiUrl}");
        }

        $allHolidays = json_decode($json, true);
        if (!is_array($allHolidays)) {
            throw new \RuntimeException("Invalid JSON from {$apiUrl}");
        }
        $holidays = [];
        foreach ($allHolidays as $holiday) {
            // Filter by canton
            $counties = $holiday['counties'] ?? null;
            $date = $holiday['date'];

            // Keep if:
            // - global holiday (no counties defined)
            // - or canton is explicitly listed
            if ($county !== null) {
                if ($counties && !in_array($county, $counties, true)) {
                    continue;
                }
            }
            // keep only public
            if (!in_array('Public', $holiday['types'])) {
                continue;
            }

            // Skip Saturdays and Sundays if desired
            if ($excludeWeekends) {
                $weekday = date('N', strtotime($date)); // 6 = Saturday, 7 = Sunday
                if ($weekday >= 6) {
                    continue;
                }
            }
            
            $holidays[] = $holiday;
        }


        return $holidays;
    }
    public static function updateHolidays(array $holidays, int $pid): int
    {
        $count = 0;
        $connection = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_igcrmtimerecording_domain_model_holiday');

        foreach ($holidays as $holiday) {
            $date = $holiday['date'];
            $name = $holiday['localName'];

            var_dump($date,$name);
            // Insert or update
            $existing = $connection->select(
                ['uid'],
                'tx_igcrmtimerecording_domain_model_holiday',
                ['date' => $date]
            )->fetchOne();

            if ($existing) {
                // $connection->update(
                //     'tx_igcrmtimerecording_domain_model_holiday',
                //     ['name' => $name],
                //     ['uid' => $existing]
                // );
            } else {
                $connection->insert(
                    'tx_igcrmtimerecording_domain_model_holiday',
                    [
                        'pid' => $pid,
                        'date' => $date,
                        'name' => $name,
                        'tstamp' => time(),
                        'crdate' => time(),
                    ]
                );
                $count++;
            }
        }
        return $count;
    }

    // API endpoint: /api/v3/AvailableCountries
    public static function getCountries(): array
    {
        return static::$countries;
    }

    // no API endpoint to get counties
    public static function getCounties(string $country): array
    {
        return static::$counties;
    }
}
