<?php

declare(strict_types=1);

namespace Internetgalerie\IgDoctrinePaginator\Pagination;

use GeorgRinger\NumberedPagination\NumberedPagination;
use InvalidArgumentException;
use TYPO3\CMS\Core\Database\Query\QueryBuilder;
use TYPO3\CMS\Core\Pagination\PaginatorInterface;

use TYPO3\CMS\Core\Pagination\SimplePagination;
use TYPO3\CMS\Core\Pagination\SlidingWindowPagination;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;

final class DoctrinePaginator implements PaginatorInterface
{
    private int $numberOfPages = 1;
    private int $keyOfFirstPaginatedItem = 0;
    private int $keyOfLastPaginatedItem = 0;
    private int $currentPageNumber = 1;
    private int $itemsPerPage = 10;
    
    /**
     * @var DataMapper
     **/
    private $dataMapper;

    private int $totalAmountOfItems = 0;
    private $modifyFunction;

    /**
     * @var array
     */
    private $paginatedQueryResult;

    public function __construct(
        private QueryBuilder $queryBuilder,
        int $currentPageNumber = 1,
        int $itemsPerPage = 10,
        private ?string $className = null,
        callable $modifyFunction = null
    ) {
        $this->modifyFunction = $modifyFunction;
        $this->dataMapper = GeneralUtility::makeInstance(DataMapper::class);
        $this->setCurrentPageNumber($currentPageNumber);
        $this->setItemsPerPage($itemsPerPage);
        $this->updateTotalAmountOfItems();

        $this->updateInternalState();
    }

    public function getPaginatedItems(): iterable
    {
        return $this->paginatedQueryResult;
    }

    public function setPaginatedItems(iterable $paginatedQueryResult)
    {
        return $this->paginatedQueryResult = $paginatedQueryResult;
    }

    public function getTotalAmountOfItems(): int
    {
        return $this->totalAmountOfItems;
    }

    public function getAmountOfItemsOnCurrentPage(): int
    {
        return count($this->paginatedQueryResult);
    }




    public function withItemsPerPage(int $itemsPerPage): PaginatorInterface
    {
        if ($itemsPerPage === $this->itemsPerPage) {
            return $this;
        }

        $new = clone $this;
        $new->setItemsPerPage($itemsPerPage);
        $new->updateInternalState();

        return $new;
    }

    public function withCurrentPageNumber(int $currentPageNumber): PaginatorInterface
    {
        if ($currentPageNumber === $this->currentPageNumber) {
            return $this;
        }

        $new = clone $this;
        $new->setCurrentPageNumber($currentPageNumber);
        $new->updateInternalState();

        return $new;
    }

    public function getNumberOfPages(): int
    {
        return $this->numberOfPages;
    }

    public function getCurrentPageNumber(): int
    {
        return $this->currentPageNumber;
    }

    public function getKeyOfFirstPaginatedItem(): int
    {
        return $this->keyOfFirstPaginatedItem;
    }

    public function getKeyOfLastPaginatedItem(): int
    {
        return $this->keyOfLastPaginatedItem;
    }
    /**
     * compatibility function to create SlidingWindowPagination (v12) or NumberedPagination (v11/v10)
     */
    public function createSlidingWindowPagination(int $maximumNumberOfLinks = 0)
    {
        if (class_exists(SlidingWindowPagination::class)) {
            return new SlidingWindowPagination($this, $maximumNumberOfLinks);
        }
        if (class_exists(NumberedPagination::class)) {
            return new NumberedPagination($this, $maximumNumberOfLinks);
        }
        return new SimplePagination($this);
    }

    private function updateTotalAmountOfItems(): void
    {
        $queryBuilderCount = clone $this->queryBuilder;
        $queryBuilderCount->resetOrderBy();
        $this->totalAmountOfItems = $queryBuilderCount->selectLiteral('count(1) AS totalAmountOfItems')
                                  ->executeQuery()
                                  ->fetchOne();
    }
    private function updatePaginatedItems(int $limit, int $offset): void
    {
        $this->paginatedQueryResult = $this->queryBuilder
                                    ->setMaxResults($limit)
                                    ->setFirstResult($offset)
                                    ->executeQuery()
                                    ->fetchAllAssociative();
        if ($this->className !== null) {
            $this->paginatedQueryResult = $this->dataMapper->map($this->className, $this->paginatedQueryResult);
        }
        if ($this->modifyFunction !== null) {
            $this->paginatedQueryResult = call_user_func($this->modifyFunction, $this->paginatedQueryResult);
        }
        //echo('update');
        //        var_dump($this->totalAmountOfItems, $this->paginatedQueryResult);        exit(0);
    }

    /**
     * States whether there are items on the current page
     */
    private function hasItemsOnCurrentPage(): bool
    {
        return $this->getAmountOfItemsOnCurrentPage() > 0;
    }

    /**
     * This method is the heart of the pagination. It updates all internal params and then calls the
     * {@see updatePaginatedItems} method which must update the set of paginated items.
     */
    private function updateInternalState(): void
    {
        $offset = (int)($this->itemsPerPage * ($this->currentPageNumber - 1));
        $totalAmountOfItems = $this->getTotalAmountOfItems();

        /*
         * If the total amount of items is zero, then the number of pages is mathematically zero as
         * well. As that looks strange in the frontend, the number of pages is forced to be at least
         * one.
         */
        $this->numberOfPages = max(1, (int)ceil($totalAmountOfItems / $this->itemsPerPage));

        /*
         * To prevent empty results in case the given current page number exceeds the maximum number
         * of pages, we set the current page number to the last page and update the internal state
         * with this value again. Such situation should in the first place be prevented by not allowing
         * those values to be passed, e.g. by using the "max" attribute in the view. However there are
         * valid cases. For example when a user deletes a record while the pagination is already visible
         * to another user with, until then, a valid "max" value. Passing invalid values unintentionally
         * should therefore just silently be resolved.
         */
        if ($this->currentPageNumber > $this->numberOfPages) {
            $this->currentPageNumber = $this->numberOfPages;
            $this->updateInternalState();
            return;
        }

        $this->updatePaginatedItems($this->itemsPerPage, $offset);

        if (!$this->hasItemsOnCurrentPage()) {
            $this->keyOfFirstPaginatedItem = 0;
            $this->keyOfLastPaginatedItem = 0;
            return;
        }

        $indexOfLastPaginatedItem = min($offset + $this->itemsPerPage, $totalAmountOfItems);

        $this->keyOfFirstPaginatedItem = $offset;
        $this->keyOfLastPaginatedItem = $indexOfLastPaginatedItem - 1;
    }

    private function setItemsPerPage(int $itemsPerPage): void
    {
        if ($itemsPerPage < 1) {
            throw new InvalidArgumentException('Argument $itemsPerPage must be greater than 0', 1573061766);
        }

        $this->itemsPerPage = $itemsPerPage;
    }

    private function setCurrentPageNumber(int $currentPageNumber): void
    {
        if ($currentPageNumber < 1) {
            throw new InvalidArgumentException('Argument $currentPageNumber must be greater than 0', 1573047338);
        }

        $this->currentPageNumber = $currentPageNumber;
    }
}
