<?php
namespace InternetGalerie\IgExport\Controller;

use InternetGalerie\Igshop2\Domain\Model\Order;
use InternetGalerie\Igshop2\Domain\Repository\CountryRepository;
use InternetGalerie\Igshop2\Domain\Repository\OrderRepository;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\Mime\Address;
use TYPO3\CMS\Backend\Template\ModuleTemplateFactory;
use TYPO3\CMS\Backend\Utility\BackendUtility;
use TYPO3\CMS\Core\Page\AssetCollector;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Http\PropagateResponseException;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3\CMS\Fluid\View\StandaloneView;

/**
 * BackendOrderController
 */
class BackendOrderController extends \TYPO3\CMS\Extbase\Mvc\Controller\ActionController
{
    /**
     * id
     * @var int
     */
    protected $id = 0;

    protected $pageInformation;

    public function __construct(
        private readonly ModuleTemplateFactory $moduleTemplateFactory,
        private readonly OrderRepository $orderRepository,
        private readonly CountryRepository $countryRepository,
        private readonly AssetCollector $assetCollector
    ) {
    }

    protected function initializeAction(): void
    {
        $this->id = (int)($this->request->getParsedBody()['id'] ?? $this->request->getQueryParams()['id'] ?? 0);
        $this->pageInformation = BackendUtility::readPageAccess($this->id, '');
    }

    public function listCartsAction(): ResponseInterface
    {
        $this->assetCollector->addStyleSheet('ig-ext-action', 'EXT:ig_export/Resources/Public/Css/Order/igShop_Orders.css');
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $arguments = $this->request->getArguments();
        $uids = $arguments['uids'] ?? '';
        $uids = str_replace(' ', '', $uids);
        if (strlen(preg_replace('/[0-9,]/', '', $uids)) > 0) {
            $this->view->assign('error', ['Nur Zahlen und Kommas eingeben']);
        }

        $searchKeys = $arguments['searchKeys'] ?? '';

        $vonObject = null;
        $bisObject = null;

        if ($arguments['vonDate'] ?? false) {
            $vonObject = new \DateTime($arguments['vonDate'] . ' 00:00:00');
        }

        if ($arguments['bisDate'] ?? false) {
            $bisObject = new \DateTime($arguments['bisDate'] . ' 23:59:59');
        }

        
        //echo($arguments['vonDate']. ' 00:00:00 == '. $vonObject->getTimestamp());

        $orders = [];
        $uidArray = [];
        if ($uids) {
            $uidArray = explode(',', $uids);
        } else {
            $uidArray = [];
        }

        if ((is_string($searchKeys) && strlen($searchKeys) > 0) || is_object($vonObject) || is_object($bisObject) || strlen($uids) > 0) {
            $orders = $this->orderRepository->findByOrderInfoLimited(100, $this->id, $searchKeys, $uidArray, $vonObject, $bisObject, true)->toArray();
        } else {
            $orders = $this->orderRepository->findBySubmittedLimited(false, $this->id, 100);
        }

        //print_r($orders,1);exit(0);
        //echo('a');var_dump($orders);exit(0);
        //$orders=array();
        $this->view->assign('uids', $arguments['uids'] ?? '');
        $this->view->assign('vonDate', $arguments['vonDate'] ?? '');
        $this->view->assign('bisDate', $arguments['bisDate'] ?? '');
        $this->view->assign('orders', $orders);
        $this->view->assign('action', 'listCarts');

        $moduleTemplate->setContent($this->view->render());
        return $this->htmlResponse($moduleTemplate->renderContent());
    }

    public function listAction(): ResponseInterface
    {
        $this->assetCollector->addStyleSheet('ig-ext-action', 'EXT:ig_export/Resources/Public/Css/Order/igShop_Orders.css');
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $arguments = $this->request->getArguments();
        $uids = $arguments['uids'] ?? '';
        $uids = str_replace(' ', '', $uids);
        if (strlen(preg_replace('/[0-9,]/', '', $uids)) > 0) {
            $this->view->assign('error', ['Nur Zahlen und Kommas eingeben']);
        }

        $searchKeys = $arguments['searchKeys'] ?? '';

        $vonObject = null;
        $bisObject = null;

        if ($arguments['vonDate'] ?? false) {
            $vonObject = new \DateTime($arguments['vonDate'] . ' 00:00:00');
        }

        if ($arguments['bisDate'] ?? false) {
            $bisObject = new \DateTime($arguments['bisDate'] . ' 23:59:59');
        }

        //echo($arguments['vonDate']. ' 00:00:00 == '. $vonObject->getTimestamp());

        $orders = [];
        $uidArray = null;
        if ($uids) {
            $uidArray = explode(',', $uids);
        } else {
            $uidArray = [];
        }

        if (strlen($searchKeys) > 0 || is_object($vonObject) || is_object($bisObject) || strlen($uids) > 0) {
            $orders = $this->orderRepository->findByOrderInfoLimited(100, $this->id, $searchKeys, $uidArray, $vonObject, $bisObject, false)->toArray();
        } else {
            $orders = $this->orderRepository->findBySubmittedLimited(true, $this->id, 100);
        }


        foreach($orders as $order) {
            $submitted = $order->getSubmitted();
            $order->setSubmitted(false);
            $order->getTotal();
            $order->setSubmitted($submitted);
        }

        //print_r($orders,1);exit(0);
        //echo('a');var_dump($orders);exit(0);
        //$orders=array();
        $this->view->assign('uids', $arguments['uids'] ?? '');
        $this->view->assign('vonDate', $arguments['vonDate'] ?? '');
        $this->view->assign('bisDate', $arguments['bisDate'] ?? '');
        $this->view->assign('orders', $orders);
        $this->view->assign('action', 'list');

        $moduleTemplate->setContent($this->view->render());
        return $this->htmlResponse($moduleTemplate->renderContent());
    }

    public function showAction(Order $order): ResponseInterface
    {
        $this->assetCollector->addStyleSheet('ig-ext-action', 'EXT:ig_export/Resources/Public/Css/Order/igShop_Orders.css');
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $submitted = $order->getSubmitted();
        $order->setSubmitted(false);
        $order->getTotal();
        $order->setSubmitted($submitted);
        $addressArray = $order->getOrderinfo();
        $this->view->assign('address', $this->getReviewAddressArray($addressArray));
        $this->view->assign('orderInfo', $order->getOrderinfo());
        if (isset($addressArray['isCalcAddress']) && $addressArray['isCalcAddress']) {
            $this->view->assign('calcAddress', $this->getReviewAddressArray($addressArray['calcAddress']));
        } elseif (isset($addressArray['isDeliveryAddress']) && $addressArray['isDeliveryAddress']) {
            $this->view->assign('calcAddress', $this->getReviewAddressArray($addressArray['calcAddress']));
        }
        $this->view->assign('order', $order);
        $this->view->assign('isHistory', 1);

        $moduleTemplate->setContent($this->view->render());
        return $this->htmlResponse($moduleTemplate->renderContent());
    }

    public function getReviewAddressArray($address)
    {
        $return = [];

        if (strlen($address['company']) > 0) {
            $return['company'] = $address['company'];
        }

        $return['name'] = (isset($address['salutation']) ? \TYPO3\CMS\Extbase\Utility\LocalizationUtility::translate('order.salutation.' . $address['salutation'], 'igshop2') . ' ' : '') . $address['lastName'] . ' ' . $address['firstName'];
        $return['street'] = $address['address'];
        $return['city'] = $address['zip'] . ' ' . $address['city'];

        if(is_numeric($address['country'])) {
            $c = $this->countryRepository->findByUid($address['country']);
            $return['country'] = $c->getName();
        } else if (is_object($address['country'])) {
            $return['country'] = $address['country']->getName();
        } else {
            $return['country'] = $address['country'];
        }
        if ($address['phone']) {
            $return['phone'] = $address['phone'];
        }

        if ($address['mail']) {
            $return['mail'] = $address['mail'];
        }

        return $return;
    }

    public function submitOrderAction(Order $order): ResponseInterface
    {
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $order->getTotal();
        $order->setSubmitted(true);

        $this->submitMailsByOrder($order);

        $this->orderRepository->update($order);
        GeneralUtility::makeInstance(PersistenceManager::class)->persistAll();

        // Send E-Mail

        return $this->redirect('show', null, null, ['order' => $order]);
    }

    private function submitMailsByOrder(Order $order): void
    {
        $address = $order->getOrderinfo();

        $adminMailTo = $this->settings['adminMailTo'];
        $adminMailFrom = $this->settings['adminMailFrom'];

        $fromMailCustomer = [];
        $fromMailCustomer['mail'] = $this->settings['customerMailFrom'];
        $fromMailCustomer['name'] = $this->settings['customerMailFromName'];
        $fromAddress = [];

        if ($fromMailCustomer['name'] && $fromMailCustomer['mail']) {
            $fromAddress = new Address($fromMailCustomer['mail'], $fromMailCustomer['name']);
        } else {
            $fromAddress = $adminMailTo;
        }

        // Customer Mail

        $customerMail = $address['mail'];

        $subject = $this->settings['customerMailSubject'];
        if (empty($subject)) {
            $subject = 'Ihre Bestellung';
        }

        $text = $this->getMailText($order);
        //echo($text);exit(0); // debug Mail
        if ($customerMail && \TYPO3\CMS\Core\Utility\GeneralUtility::validEmail($customerMail)) {
            $this->submitMail($order, $fromAddress, $customerMail, $subject, $text);
        }

        // Admin Mail

        $adminSubject = $this->settings['adminMailSubject'];
        if (empty($adminSubject)) {
            $adminSubject = 'Neue Bestellung';
        }

        $adminText = $this->getMailText($order, true);

        //admin mail
        if ($adminMailTo) {
            $this->submitMail($order, $adminMailFrom ? $adminMailFrom : $fromAddress, $adminMailTo, $adminSubject, $adminText);
        }
        // Save Order to File
        if ($this->settings['saveOrderToXmlEnable']) {
            $path = $this->settings['saveOrderToXmlPath'];
            //if($path[0]!='/') $path = realpath(PATH_site) .'/' .$path;
            //echo('mkdir:'. $path.'<br />');
            if (!is_dir($path)) {
                if (!mkdir($path)) {
                    $logger = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance('TYPO3\CMS\Core\Log\LogManager')->getLogger(__CLASS__);
                    $logger->log(\TYPO3\CMS\Core\Log\LogLevel::ERROR, 'Error: Create Folder ' . $path . '', error_get_last());
                }
            }
            if (is_dir($path)) {
                $params = [
                    'companyAddress' => $companyAddress,
                    'prepaidData' => $prepaidData,
                    'order' => $order,
                    'currency' => $order->getCurrency(),
                    'address' => $order->getOrderinfo(),
                    'baseUrl' => $baseUrl,
                    'hasPorto' => $order->hasPorto(),
                    'language' => $GLOBALS['TSFE']->config['config']['language'],
                ];

                $xmlContent = $this->renderFluid('Order/XmlExport.xml', $params);
                file_put_contents($path . '/order-' . $order->getUid() . '.xml', $xmlContent);
            }
        }

        //exit(0);
    }

    public function exportOrderAction(Order $order): ResponseInterface
    {
        $this->generateXlsForOrder($order);

        return $this->htmlResponse();
    }

    public function exportOrdersAction(): ResponseInterface
    {
        $arguments = $this->request->getArguments();
        $uids = $arguments['uids'];
        $uids = str_replace(' ', '', $uids);
        if (strlen(preg_replace('/[0-9,]/', '', $uids)) > 0) {
            $this->view->assign('error', ['Nur Zahlen und Kommas eingeben']);
        }

        $searchKeys = $arguments['searchKeys'] ?? '';

        $vonObject = null;
        $bisObject = null;

        if (isset($arguments['vonDate']) && $arguments['vonDate']) {
            $vonObject = new \DateTime($arguments['vonDate'] . ' 00:00:00');
        }

        if (isset($arguments['bisDate']) && $arguments['bisDate']) {
            $bisObject = new \DateTime($arguments['bisDate'] . ' 23:59:59');
        }

        $action = $arguments['sourceAction'] ?? 'list';

        // submitted orders or open carts
        $showOpenCarts = $action === 'listCarts' ? true : false;

        //echo($arguments['vonDate']. ' 00:00:00 == '. $vonObject->getTimestamp());

        $orders = [];
        $uidArray = [];
        if ($uids) {
            $uidArray = explode(',', $uids);
        } else {
            $uidArray = [];
        }

        if ((is_string($searchKeys) && strlen($searchKeys) > 0) || is_object($vonObject) || is_object($bisObject) || strlen($uids) > 0) {
            $orders = $this->orderRepository->findByOrderInfoLimited(null, $this->id, $searchKeys, $uidArray, $vonObject, $bisObject, false)->toArray();
        } else {
            $orders = $this->orderRepository->findBySubmittedLimited(true, $this->id,  null);
        }


        $ordersCart = null;
        if ((is_string($searchKeys) && strlen($searchKeys) > 0) || is_object($vonObject) || is_object($bisObject) || strlen($uids) > 0) {
            $ordersCart = $this->orderRepository->findByOrderInfoLimited(null, $this->id, $searchKeys, $uidArray, $vonObject, $bisObject, true)->toArray();
        } else {
            $ordersCart = $this->orderRepository->findBySubmittedLimited(false, $this->id, null);
        }

        // Create excel
        $this->generateXlsForOrders($orders, $ordersCart);

        return $this->htmlResponse();
    }

    public function generateXlsForOrders($orders, $ordersCart = null)
    {
        // Initialize XLS
        $spreadsheet = new Spreadsheet();

        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Bestellungen');


        // Generate header
        $headerArray = [
            'Bestellnummer',
            'Datum',
            'Anzahl Produkte',
            'Total ohne Porto',
            'Porto',
            'Total inkl. MwSt.',
            'Anrede',
            'Vorname',
            'Nachname',
            'Firma',
            'Adresse',
            'PLZ',
            'Ort',
            'Land',
            'Anderes Land',
            'Telefon',
            'E-Mail',
            'Zahlungsmethode',
            'Extra Rechnungsadresse',
            'Rechnung Anrede',
            'Rechnung Vorname',
            'Rechnung Nachname',
            'Rechnung Firma',
            'Rechnung Strasse',
            'Rechnung PLZ',
            'Rechnung Ort',
            'Rechnung Land',
            'Rechnung anderes Land'
        ]; /// TODO: Automatisches Ermitteln Felder, die ausgegeben werden sollen (Gestaltet sich als schwierig, weil Daten serialisiert gespeichert sind)

        $row = 1;
        $sheet->fromArray($headerArray, null, 'A' . $row++);
        $sheet->getStyle('A1:' . $sheet->getHighestDataColumn() . '1')->getFont()->setBold(true);

        foreach($orders as $order) {
            // Generate line
            $submitted = $order->getSubmitted();
            $order->setSubmitted(false);
            $order->getTotal();
            $order->setSubmitted($submitted);

            $paymentMethod = $order->getPaymentMethod();
            $country = $order->getCountry();

            $data = [];
            $orderInfo = $order->getOrderInfo();
            $orderInfo['salutation'] = $orderInfo['salutation'] == 1 ? 'Herr' : ($orderInfo['salutation'] == 2 ? 'Frau' : '');
            $orderInfo['payment'] = $paymentMethod ? $paymentMethod->getName() : '';
            $orderInfo['country'] = $country ? $country->getName() : '';
            $calcAddress = [];
            $isCalcAddress = $orderInfo['isCalcAddress'] || $orderInfo['extraBillingAddress'];
            unset($orderInfo['isCalcAddress']);
            $orderInfo['extraBillingAddress'] = $isCalcAddress ? 1 : 0;
            if($isCalcAddress) {
                $calcAddress = $orderInfo['calcAddress'];
                $calcAddress['salutation'] = $calcAddress['salutation'] == 1 ? 'Herr' : ($calcAddress['salutation'] == 2 ? 'Frau' : '');
                $calcAddress['country'] = is_object($order->getCountry()) ? $order->getCountry()->getName() : $order->getCountryText();
            }
            unset($orderInfo['calcAddress']);
            $data = array_merge(array_values($calcAddress), $data);
            $data = array_merge(array_values($orderInfo), $data);
            $data = array_merge([
                $order->getUid(),
                $order->getOrderdate()->format('d.m.Y H:i'),
                $this->orderRepository->getTotalProductsAmount($order->getUid()),
                $order->getCurrency()->getShort() . ' ' .  $order->getTotal(),
                $order->getCurrency()->getShort() . ' ' . $order->getPortoPrice(),
                $order->getCurrency()->getShort() . ' ' . ($order->getTotalWithPorto() > 0 ? $order->getTotalWithPorto() : $order->getTotal()),
            ], $data);

            $sheet->fromArray($data, null, 'A' . $row++);
        }

        foreach($sheet->getColumnIterator() as $column) {
            $sheet->getColumnDimension($column->getColumnIndex())->setAutoSize(true);
        }

        $baseFilename = 'orders.xlsx';

        if($ordersCart && (is_array($ordersCart) ? count($ordersCart) : $ordersCart->count())) {
            $sheet = new \PhpOffice\PhpSpreadsheet\Worksheet\Worksheet($spreadsheet, 'Offene Bestellungen');
            $spreadsheet->addSheet($sheet,1);

            $row = 1;
            $sheet->fromArray($headerArray, null, 'A' . $row++);
            $sheet->getStyle('A1:' . $sheet->getHighestDataColumn() . '1')->getFont()->setBold(true);

            foreach($ordersCart as $order) {
                // Generate line
                $submitted = $order->getSubmitted();
                $order->setSubmitted(false);
                $order->getTotal();
                $order->setSubmitted($submitted);

                $paymentMethod = $order->getPaymentMethod();
                $country = $order->getCountry();

                $data = [];
                $orderInfo = $order->getOrderInfo();
                $orderInfo['salutation'] = $orderInfo['salutation'] == 1 ? 'Herr' : ($orderInfo['salutation'] == 2 ? 'Frau' : '');
                $orderInfo['payment'] = $paymentMethod ? $paymentMethod->getName() : '';
                $orderInfo['country'] = $country ? $country->getName() : '';
                $calcAddress = [];
                $isCalcAddress = $orderInfo['isCalcAddress'] || $orderInfo['extraBillingAddress'];
                unset($orderInfo['isCalcAddress']);
                $orderInfo['extraBillingAddress'] = $isCalcAddress ? 1 : 0;
                if($isCalcAddress) {
                    $calcAddress = $orderInfo['calcAddress'];
                    $calcAddress['salutation'] = $calcAddress['salutation'] == 1 ? 'Herr' : ($calcAddress['salutation'] == 2 ? 'Frau' : '');
                    $calcAddress['country'] = is_object($order->getCountry()) ? $order->getCountry()->getName() : $order->getCountryText();
                }
                unset($orderInfo['calcAddress']);
                $data = array_merge(array_values($calcAddress), $data);
                $data = array_merge(array_values($orderInfo), $data);
                $data = array_merge([
                    $order->getUid(),
                    $order->getOrderdate()->format('d.m.Y H:i'),
                    $this->orderRepository->getTotalProductsAmount($order->getUid()),
                    $order->getCurrency()->getShort() . ' ' .  $order->getTotal(),
                    $order->getCurrency()->getShort() . ' ' . $order->getPortoPrice(),
                    $order->getCurrency()->getShort() . ' ' . ($order->getTotalWithPorto() > 0 ? $order->getTotalWithPorto() : $order->getTotal()),
                ], $data);

                $sheet->fromArray($data, null, 'A' . $row++);
            }

            $cellIterator = $sheet->getRowIterator()->current()->getCellIterator();
            $cellIterator->setIterateOnlyExistingCells(true);

            foreach($cellIterator as $cell) {
                $sheet->getColumnDimension($cell->getColumn())->setAutoSize(true);
            }
        }

        $spreadsheet->setActiveSheetIndex(0);

        $this->spreadsheetOut($spreadsheet, $baseFilename);
    }

    protected function spreadsheetOut(Spreadsheet $spreadsheet, string $filename): void
    {
        $writer = new Xlsx($spreadsheet);

        $tempFilename = tempnam('', 'ig-shop-export');
        $writer->save($tempFilename);
        $contentLength = filesize($tempFilename);
        $excelOutput = file_get_contents($tempFilename);
        unlink($tempFilename);
        /*
          ob_start();
          $writer->save('php://output');   
          $excelOutput = ob_get_contents();
          ob_end_clean();
          $contentLength = strlen(bin2hex($excelOutput))/2;
        */

        $response = $this->responseFactory->createResponse()
                                          ->withHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet')
                                          ->withHeader('Expires', '0')
                                          ->withHeader('Cache-Control', 'private')
                  //->withHeader('Cache-Control', 'must-revalidate, post-check=0, pre-check=0')                                  
                                          ->withHeader('Content-Disposition', 'attachment; filename="' . $filename . '"')
                                          ->withHeader('Content-Length', (string)$contentLength);
        $response->getBody()->write($excelOutput);
        throw new PropagateResponseException($response, 200);
    }
    
    
    public function generateXlsForOrder(Order $order)
    {
        // Initialize XLS
        $spreadsheet = new Spreadsheet();

        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Bestellung ' . $order->getUid());


        // Generate header
        $headerArray = [
            'Bestellnummer',
            'Datum',
            'Anrede',
            'Vorname',
            'Nachname',
            'Firma',
            'Adresse',
            'PLZ',
            'Ort',
            'Land',
            'Anderes Land',
            'Telefon',
            'E-Mail',
            'Zahlungsmethode',
            'Extra Rechnungsadresse',
            'Rechnung Anrede',
            'Rechnung Vorname',
            'Rechnung Nachname',
            'Rechnung Firma',
            'Rechnung Strasse',
            'Rechnung PLZ',
            'Rechnung Ort',
            'Rechnung Land',
            'Rechnung anderes Land'
        ]; /// TODO: Automatisches Ermitteln Felder, die ausgegeben werden sollen (Gestaltet sich als schwierig, weil Daten serialisiert gespeichert sind)

        $row = 1;

        $paymentMethod = $order->getPaymentMethod();
        $country = $order->getCountry();

        // Generate line
        $data = [];
        $orderInfo = $order->getOrderInfo();
        $orderInfo['salutation'] = $orderInfo['salutation'] == 1 ? 'Herr' : ($orderAddress['salutation'] == 2 ? 'Frau' : '');
        $orderInfo['payment'] = $paymentMethod ? $paymentMethod->getName() : '';
        $orderInfo['country'] = $country ? $country->getName() : '';
        $calcAddress = [];
        $isCalcAddress = $orderInfo['isCalcAddress'] || $orderInfo['extraBillingAddress'];
        unset($orderInfo['isCalcAddress']);
        $orderInfo['extraBillingAddress'] = $isCalcAddress ? 1 : 0;
        if($isCalcAddress) {
            $calcAddress = $orderInfo['calcAddress'];
            $calcAddress['salutation'] = $calcAddress['salutation'] == 1 ? 'Herr' : ($calcAddress['salutation'] == 2 ? 'Frau' : '');
            $calcAddress['country'] = $order->getCountry()->getName();
        }
        unset($orderInfo['calcAddress']);
        $data = array_merge(array_values($calcAddress), $data);
        $data = array_merge(array_values($orderInfo), $data);
        $data = array_merge([
            $order->getUid(),
            $order->getOrderdate()->format('d.m.Y H:i'),
        ], $data);

        $i = 0;
        foreach($data as $value) {
            $labelValuePair = [$headerArray[$i], $value];
            $sheet->getStyle('A' . $row)->getFont()->setBold(true);
            $sheet->fromArray($labelValuePair, null, 'A' . $row++);
            $i++;
        }

        $row++;

        $headerProductArray = ['Produkt', 'Anzahl', 'Preis'];
        $sheet->fromArray($headerProductArray, null, 'A' . $row);
        $sheet->getStyle('A' . $row . ':' . $sheet->getHighestDataColumn() . $row)->getFont()->setBold(true);
        $row++;

        foreach($order->getProducts() as $orderProduct) {

            $product = $orderProduct->getProduct();
            $amount = $orderProduct->getAmount();
            $orderProduct->calcPrice($order->getCurrency());
            $price = $orderProduct->getPrice();
            $data = [
                $product->getName(),
                $amount,
                $order->getCurrency()->getShort() . ' ' . $price
            ];

            $sheet->fromArray($data, null, 'A' . $row++);
        }

        $row++;

        if($order->getPorto()) {
            $data = [
                'Versandkosten',
                '',
                $order->getCurrency()->getShort() . ' ' . $order->getPorto()->getPrice()
            ];
            $sheet->fromArray($data, null, 'A' . $row);

            $sheet->getStyle('A' . $row)->getFont()->setBold(true);
            $row++;
        }
        $data = [
            'Total inkl. MwSt.',
            $this->orderRepository->getTotalProductsAmount($order->getUid()),
            $order->getCurrency()->getShort() . ' ' . ($order->getTotalWithPorto() > 0 ? $order->getTotalWithPorto() : $order->getTotal())
        ];
        $sheet->fromArray($data, null, 'A' . $row);

        $sheet->getStyle('A' . $row)->getFont()->setBold(true);

        foreach($sheet->getColumnIterator() as $column) {
            $sheet->getColumnDimension($column->getColumnIndex())->setAutoSize(true);
        }

        $row++;

        $spreadsheet->setActiveSheetIndex(0);


        $baseFilename = 'order-' . $order->getUid() . '.xlsx';
        $this->spreadsheetOut($spreadsheet, $baseFilename);
        
    }

    private function submitMail(Order $order, mixed $from, mixed $to, string $subject, string $mailContentHtml): bool
    {
        $message = GeneralUtility::makeInstance(\TYPO3\CMS\Core\Mail\MailMessage::class);
        $message->to($to)->from($from)->subject($subject);
        $message->html($mailContentHtml);

        if ($this->settings['orderConfirmationPdf']) {
            $filename = LocalizationUtility::translate('order.confirmation.filename', 'igshop2') . '_' . $order->getUid() . '.pdf';
            $tmpFilename = 'typo3temp/' . $filename;

            $fp = fopen($tmpFilename, 'w');
            fwrite($fp, $this->generateConfirmation($order));
            fclose($fp);

            $confirmationAttachment = \Swift_Attachment::fromPath($tmpFilename);
            $confirmationAttachment->setFilename($filename);
            $message->attach($confirmationAttachment);

            $message->send();
            unlink($tmpFilename);
        } else {
            $message->send();
        }
        return $message->isSent();
    }

    public function getMailText(Order $order, $adminMail = false): string
    {
        $companyAddress = $this->settings['customerMailAddress'] ? $this->settings['customerMailAddress']['value'] : '';
        $prepaidData = $this->settings['customerMailPrepaidData'] ? $this->settings['customerMailPrepaidData']['value'] : '';

        if (strlen($companyAddress) > 0) {
            $companyAddress = explode('\n', $companyAddress);
        }

        if (strlen($prepaidData) > 0) {
            $prepaidData = explode('\n', $prepaidData);
        }

        // get BaseUrl and set slash at end if not set - important for images!
        $baseUrl = $GLOBALS['TSFE']->tmpl->setup['config.']['baseURL'];
        if (substr($baseUrl, -1) != '/') {
            $baseUrl .= '/';
        }


        $params = [
            'companyAddress' => $companyAddress,
            'prepaidData' => $prepaidData,
            'order' => $order,
            'currency' => $order->getCurrency(),
            'address' => $order->getOrderinfo(),
            'logo' => $this->settings['mailLogo'],
            'baseUrl' => $baseUrl,
            'contactLink' => $this->settings['customerMailContactLink'],
            'isFirstOrder' => $this->settings['activateIsFirstOrder'],
            'knowFromWhere' => $this->settings['activateKnowFromWhere'],
        ];

        // Debug Mail
        // Mail wird nicht versendet, sondern auf der Webseite ausgegeben
        if ($this->settings['debugMail']) {
            echo $this->renderFluid('Mail/Admin.html', $params);
            exit;
        }

        if ($adminMail) {
            return $this->renderFluid('Mail/Admin.html', $params);
        } else {
            return $this->renderFluid('Mail/Index.html', $params);
        }
    }

    /**
     * Generate confirmation PDF
     */
    private function generateConfirmation(Order $order): string
    {
        $parameters = [
            'order' => $order,
        ];
        $confirmationHtml = $this->renderFluid('Mail/Confirmation.html', $parameters);

        $proc = proc_open(
            //'wkhtmltopdf -L 0 -R 0 -T 30 -B 20 --header-html ' . $tmpHeader . ' --footer-html ' . $tmpFooter . ' - -',
            'wkhtmltopdf --encoding \'utf-8\' -L 20 -R 20 -T 15 -B 10 - -',
            [
                0 => ['pipe', 'r'],
                1 => ['pipe', 'w'],
            ],
            $pipes
        );

        fwrite($pipes[0], $confirmationHtml);
        fclose($pipes[0]);
        $pdf = stream_get_contents($pipes[1]);
        fclose($pipes[1]);
        proc_close($proc);

        return $pdf;
    }

    /**
     * renders fluid template
     * @param string $template path to fluid-html-template, form: ext/Resources/Private/Templates/*$template*
     * @param array $parameter key=>value pairs, where key is fluid placeholder {key} in the template
     * @return string rendered (html-format) template
     */
    private function renderFluid(string $template, array $parameter): string
    {
        $fluidView = GeneralUtility::makeInstance(StandaloneView::class);
        $conf = $this->configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FRAMEWORK);
        $fluidView->setTemplate($template);
        $fluidView->setLayoutRootPaths($conf['view']['layoutRootPaths']);
        $fluidView->setTemplateRootPaths($conf['view']['templateRootPaths']);
        $fluidView->setPartialRootPaths($conf['view']['partialRootPaths']);
        $fluidView->assignMultiple($parameter);
        return $fluidView->render();
    }

}
