<?php

namespace Ig\IgFibu\Controller;

use DateTime;
use Exception;
use Ig\IgFibu\Domain\Model\Invoice;
use Ig\IgFibu\Domain\Model\InvoiceDate;
use Ig\IgFibu\Domain\Model\InvoiceInterface;
use Ig\IgFibu\Domain\Model\InvoiceItem;
use Ig\IgFibu\Domain\Model\InvoiceItemUnit;
use Ig\IgFibu\Domain\Model\Payment;
use Ig\IgFibu\Domain\Repository\CostCenterRepository;
use Ig\IgFibu\Domain\Repository\CreditRepository;
use Ig\IgFibu\Domain\Repository\InvoiceItemRepository;
use Ig\IgFibu\Domain\Repository\InvoicestatusRepository;
use Ig\IgFibu\Domain\Repository\LogMailRepository;
use Ig\IgFibu\Domain\Repository\PaymentRepository;
use Ig\IgFibu\Service\BookingService;
use Ig\IgFibu\Service\DebitorService;
use Ig\IgFibu\TaxBehavior;
use Ig\IgFibu\Utility\InvoiceUtility;
use Ig\IgFibu\Utility\PeriodUtility;
use Internetgalerie\IgCrmTemplate\Utility\TemplateUtility;
use Internetgalerie\IgDatapoolFe\Services\FormService;
use Internetgalerie\IgDoctrinePaginator\Pagination\DoctrinePaginator;
use Internetgalerie\IgExcel\Domain\Model\ImportConfig;
use Internetgalerie\IgExcel\Utility\ExcelUtility;
use Internetgalerie\IgExcel\Utility\ImportUtility;
use Internetgalerie\IgFrontendUser\Domain\Repository\FrontendUserRepository;
use Locale;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\Mime\Address as MimeAddress;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Http\PropagateResponseException;
use TYPO3\CMS\Core\Mail\FluidEmail;
use TYPO3\CMS\Core\Mail\Mailer;
use TYPO3\CMS\Core\Page\AssetCollector;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Resource\File as FileResource;
use TYPO3\CMS\Core\Resource\FileRepository;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Utility\VersionNumberUtility;
use TYPO3\CMS\Extbase\Annotation\IgnoreValidation;
use TYPO3\CMS\Extbase\Http\ForwardResponse;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3\CMS\Core\TypoScript\FrontendTypoScriptConfiguration;


/**
 * InvoiceController
 */
class AbstractInvoiceController extends AbstractFibuController
{
    protected $objectNames = ['invoice', 'newInvoice'];
    protected $dateObjects = [];
    protected $dateTimeObjects = [];
    protected $searchDefault = [
        //        'status' => 1,
    ];
    protected $debugMail = '';//da@summer.ch';//debug + Bcc bei mail an kunde nun in Conf.php gesetzt, bzw. auf FeUser
    protected $mailSend = 1; // 0 = debug, Mail nicht senden, 1= mail senden (an Kunde oder/und $debugMail)

    protected $maxMailsToSent = 100;//100; // maximal number of mails to send per cycle


    // max invoices for pdf
    protected $maxQrInvoice = 100;
    // max invoices for html format
    protected $maxQrInvoiceHtml = 1000;

    protected $optionPages = [];
    
    protected $sammelinvoices = [
        [
            'uid' => 1,
            'title' => 'Sammelrechnung',
        ],
        [
            'uid' => 0,
            'title' => 'Keine Sammelrechnung',
        ],
    ];

    protected $deliveryMethods = [
        'print' => 'drucken & Brief schicken',
        'mail' => 'E-Mail schicken',
    ];

    protected $invoicestatusRepository = null;
    /**
     * invoiceRepository
     */
    protected $invoiceRepository = null;

    /**
     * debitorService
     *
     * @var DebitorService
     */
    protected $debitorService = null;

    /**
     * frontendUserRepository
     *
     * @var FrontendUserRepository
     */
    protected $frontendUserRepository = null;

    
    protected ?InvoiceUtility $invoiceUtility = null;
    protected ?AssetCollector $assetCollector = null;
    
    
    public function injectDebitorService(DebitorService $debitorService): void
    {
        $this->debitorService = $debitorService;
    }

    
    public function injectFrontendUserRepository(FrontendUserRepository $frontendUserRepository): void
    {
        $this->frontendUserRepository = $frontendUserRepository;
    }
    
    public function injectInvoiceUtility(InvoiceUtility $invoiceUtility): void
    {
        $this->invoiceUtility = $invoiceUtility;
    }

    public function injectAssetCollector(AssetCollector $assetCollector): void
    {
        $this->assetCollector = $assetCollector;
    }
    
    public function initializeAction(): void
    {
        $formService =GeneralUtility::makeInstance(FormService::class);
        $request = $formService->setController($this);
        parent::initializeAction();
        $this->search = $this->getSearch();
        $this->redirectUrl = $this->request->hasArgument('redirectUrl') ? $this->request->getArgument(
            'redirectUrl'
        ) : '';
        if ($_GET['html'] ?? false) {
            $this->invoiceUtility->setPdfOutputMode(InvoiceUtility::OUTPUT_HTML_DIE);
        }
    }
    public function getPdfContent(
        $invoiceQueryBuilder,
        string $additionalFilename = '',
        ?string $letterDate = null
    ): string {
        $pages = $this->search['pages'] ?? 0;
        $limit = $this->maxQrInvoice;
        $offset = $pages * $this->maxQrInvoice;

        // get total entries
        $queryBuilderCount = clone $invoiceQueryBuilder;
        $totalAmountOfItems = $queryBuilderCount->selectLiteral('count(1) AS totalAmountOfItems')
                                                ->executeQuery()
                                                ->fetchOne();

            
        $invoices = $this->invoiceRepository->findBySearchExecute($invoiceQueryBuilder, $limit, $offset);
        if (VersionNumberUtility::convertVersionNumberToInteger(
            VersionNumberUtility::getNumericTypo3Version()
        ) >= 11000000) {
            $site = $this->request->getAttribute('site');
        } else {
            $site = $GLOBALS['TYPO3_REQUEST']->getAttribute('site', null);
        }
        // ugly set language uid in crmLanguage
        $languages = $site->getLanguages();
        $languageUid = 0;
        if (isset($this->search['language']) && $this->search['language'] == 3) {
            $languageUid = 2;
        }
        $language = $languages[$languageUid] ?? $languages[0];
            
        if ($offset > 0 || $totalAmountOfItems > $this->maxQrInvoice) {
            $digits = (int)ceil(log10($totalAmountOfItems));
            $start = ($offset + 1);
            $end = $offset + count($invoices);
            $additionalFilename .= '_' . str_pad($start, $digits, '0', STR_PAD_LEFT) . '-' . str_pad(
                $end,
                $digits,
                '0',
                STR_PAD_LEFT
            );
        }
        return $this->invoiceUtility->outInvoice(
            $this->view,
            $invoices,
            $language,
            $this->search,
            true,
            null,
            $additionalFilename,
            $letterDate
        );
    }

    public function getPrintPagesByCount(int $totalAmountOfItems)
    {
        $printPages = null;
        $printPages = [];
        $document = 0;
        for ($from = 1; $from <= $totalAmountOfItems; $from += $this->maxQrInvoice) {
            $document++;
            $to = $from + $this->maxQrInvoice - 1;
            if ($to > $totalAmountOfItems) {
                $to = $totalAmountOfItems;
            }
            $count = $to - $from + 1;
            $printPages[] = [
                'document' => $document,
                'from' => $from,
                'to' => $to,
                'count' => $count,
                'label' => $from . ($from != $to ? ' - ' . $to : ''),
            ];
        }
        return $printPages;
    }
    public function getPrintPagesByUids(array $uids, array $baseSearch = []): array
    {
        $pages = $this-> getPrintPagesByCount(count($uids));
        foreach ($pages as &$page) {
            $pageUids = array_slice($uids, $page['from'] - 1, $page['count']);
            $page['uids'] = $pageUids;
            $arguments = [
                'letterDate' => $baseSearch['letterDate'] ?? [],
            ];
            if ($baseSearch['sorting'] ?? false) {
                $arguments['sorting'] = $baseSearch['sorting'];
            }
            $arguments['uids'] = implode(',', $pageUids);
            $page['arguments'] = http_build_query([
                'search' => $arguments,
            ]);
            $search = [
                'uids' => implode(',', $pageUids),
            ];
            $page['search'] = $search;
        }
        return $pages;
    }


    /**
     * action list
     */
    public function initListAction(): ResponseInterface
    {
        $this->view->assign('redirectUrl', $this->redirectUrl);
        $this->view->assign('isAdmin', $this->debitorService->isAdmin());
        $this->view->assign('debitorConfig', $this->debitorService->getDebitorConfig());
        $acl = $this->debitorService->getAcl();
        $this->view->assign('acl', $acl);

        if (!$acl['igFibu']['showAll']) {
            $this->search['status'] = [1, 2, 16, 17, 32, 33, 256, 257];
        }
        $this->invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);
        $this->view->assign('config', $this->debitorService->getAllConfig());

        $this->view->assign('now', new DateTime());

        $this->view->assign('sammelinvoices', $this->sammelinvoices);
        $this->view->assign('deliveryMethods', $this->deliveryMethods);

        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/tableSelectToggle.js');
        //        $this->assetCollector->addJavaScript('tableSelectToggle','EXT:ig_fibu/Resources/Public/JavaScript/tableSelectToggle.js');

        $sorting = [];
        if (isset($this->search['tablenames'])) {
            $sorting = $this->debitorService->getSorting($this->search['tablenames']);
        }
        $this->view->assign('sorting', $sorting);
        $this->assignTotals();
        $this->assignFormfields();
        $this->assignPeriode();
        $this->view->assign('search', $this->search);
        $advancedBoxIsOpen = $this->hasAdvancedSearch(
            $this->search,
            ['status', 'deliveryMethod', 'keyword', 'output', 'verband', 'tenant']
        );
            
        $this->view->assign('advancedBoxIsOpen', $advancedBoxIsOpen);
        
        $this->assignList();
        if (!empty($this->search)) {
            $invoiceQueryBuilder = $this->invoiceRepository->findBySearch($this->search);
            $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument(
                'currentPage'
            ) : 1;

            $paginator = new DoctrinePaginator($invoiceQueryBuilder, $currentPage, 50, Invoice::class);
            $pagination = $paginator->createSlidingWindowPagination(10);
            
            //$totalAmountOfItems = $paginator->getTotalAmountOfItems();

            $this->view->assign('pagination', [
                'paginator' => $paginator,
                'pagination' => $pagination,
            ]);
        }
        return $this->htmlResponse($this->view->render());
    }
    public function exportAction(): ResponseInterface
    {
        $invoiceQueryBuilder = $this->invoiceRepository->findBySearch($this->search);
        $invoices = $this->invoiceRepository->findBySearchExecute($invoiceQueryBuilder);
        //$res = $invoiceQueryBuilder->executeQuery();
        $excelUtility = GeneralUtility::makeInstance(ExcelUtility::class);
        $colsFormatCodes = [];
        $excelDateFormat = 'dd.mm.yyyy';
        $excelDateTimeFormat = 'yyyy-mm-dd hh:mm:ss';
        $headers = [];
        // Datum
        $headers[] = LocalizationUtility::translate('invoice-number', 'ig_fibu');
        $headers[] = LocalizationUtility::translate('tx_igfibu_domain_model_invoice.status', 'ig_fibu');

        $headers[] = LocalizationUtility::translate('invoice-date', 'ig_fibu');
        $colsFormatCodes['C'] = $excelDateFormat;
        $headers[] = LocalizationUtility::translate('tx_igfibu_domain_model_invoice.paid_date', 'ig_fibu');
        $colsFormatCodes['D'] = $excelDateFormat;

        $headers[] = LocalizationUtility::translate('customer-number', 'ig_fibu');
        $headers[] = LocalizationUtility::translate('debitor', 'ig_fibu');
        $headers[] = LocalizationUtility::translate('periode', 'ig_fibu');
        $headers[] = LocalizationUtility::translate('total', 'ig_fibu');
        $headers[] = LocalizationUtility::translate('tx_igfibu_domain_model_credit', 'ig_fibu');
        $headers[] = LocalizationUtility::translate('tx_igfibu_domain_model_invoice.payments', 'ig_fibu');
        $headers[] = LocalizationUtility::translate('fibu-open', 'ig_fibu');
        $headers[] = LocalizationUtility::translate('igfibu.invoice.mwst', 'ig_fibu');
        //while ($invoice = $res->fetchAssociative()) {
        $rows = [];
        foreach ($invoices as $invoice) {
            $debitor = $invoice->getDebitor();
            $row = [];
            $row[] = $invoice->getUid();
            $row[] = $invoice->getStatus()->getTitle();
            $row[] = Date::PHPToExcel($invoice->getCreateDate());// $invoice->getCreateDate()->format('Y-m-d HH:ii:ss'); //
            $row[] = Date::PHPToExcel($invoice->getPaidDate());
            $row[] = $debitor ? (($this->settings['useOwnAddressNo'] ?? false) ? $debitor->getMeAddressid() : $debitor->getNumber()) : $invoice->getEntryUid();
            $row[] = $debitor ? $debitor->getName() : $invoice->getDebitor();
            $row[] = $invoice->getPeriode();
            $row[] = $invoice->getTotal();
            $row[] = $invoice->getInvoicePaymentsTotalCredit();
            $row[] = $invoice->getInvoicePaymentsTotal();
            $row[] = $invoice->getAmountOpen();
            $row[] = $invoice->getTaxRate();
            $rows[] = $row;
        }
        $excelUtility->export($headers, $rows, 'export-' . date('Y-m-d'), 'Rechnungen', $colsFormatCodes);
        var_dump($this->search);
        exit(0);
    }

    public function assignTotals(): void
    {
        $totals = $this->invoiceRepository->findTotal($this->search);
        $this->view->assign('totals', $totals);
    }
    public function assignPeriode(): void
    {
        $this->tenant = $this->debitorService->getMandantByUid($this->tenantId);
        if ($this->tenant) {
            $invoicePeriodeLengthInMonth = $this->tenant->getInvoicePeriodeLengthInMonth() ?? 12;
        } else {
            $invoicePeriodeLengthInMonth = 12;
        }
        $periodes = $this->invoiceRepository->findGroupByPeriodeForlabel(
            $invoicePeriodeLengthInMonth,
            $this->search,
            40
        );
        $this->view->assign('tenant', $this->tenant);

        $this->view->assign('periodes', $periodes);
        
        $config = $this->debitorService->getMandantConfig();
        $invoicePeriodeLengthInMonth = (int) ($config['default']['invoicePeriodeLengthInMonth'] ?? 12);
        $periodeLabel = LocalizationUtility::translate('periode-label-' . $invoicePeriodeLengthInMonth, 'ig_fibu');
        $this->view->assign('periodeLabel', $periodeLabel);
    }

    public function assignPages($totalAmountOfItems): void
    {
        $this->optionPages = [];
        if ($totalAmountOfItems > $this->maxQrInvoice) {
            for ($i = 1; $i < $totalAmountOfItems; $i += $this->maxQrInvoice) {
                $to = $i + $this->maxQrInvoice - 1;
                if ($to > $totalAmountOfItems) {
                    $to = $totalAmountOfItems;
                }
                $this->optionPages[] = 'Seite ' . $i . ' bis ' . $to;
            }
        }
        $this->view->assign('optionPages', $this->optionPages);
    }

    /**
     * action show
     */
    public function initShowAction(InvoiceInterface $invoice): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $acl = $this->debitorService->getAcl();
        $this->view->assign('acl', $acl);
        
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/ig_fibu.js');
        $this->assignFormfields();
        $this->view->assign('redirectUrl', $this->redirectUrl);
        $this->view->assign('invoice', $invoice);
        $this->invoiceUtility->setReceiver($invoice, false);
        $this->view->assign('receiver', $this->invoiceUtility->getReceiver());
        $this->view->assign('isAdmin', $this->debitorService->isAdmin());
        $activeTab = $this->request->hasArgument('activeTab') ? $this->request->getArgument('activeTab') : '';
        $this->view->assign('activeTab', $activeTab);
        $creditRepository = GeneralUtility::makeInstance(CreditRepository::class);

        $credits = $creditRepository->findOpenByDebitor($this->tenantId ?? 0, $invoice->getEntryUid());
        $this->view->assign('credits', $credits);
        $this->view->assign('lineBreak', "\n");
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action printPreviewAction
     */
    public function printPreviewAction(InvoiceDate $invoiceDate = null): ResponseInterface
    {
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $this->assetCollector->addJavaScript('pdf-lib', 'EXT:ig_fibu/Resources/Public/Libs/PdfLib/pdf-lib.min.js');
        $this->assetCollector->addJavaScript('toastr', 'EXT:ig_fibu/Resources/Public/Libs/Toastr/toastr.min.js');
        $this->assetCollector->addStyleSheet('toastr', 'EXT:ig_fibu/Resources/Public/Libs/Toastr/toastr.min.css');
        $this->assetCollector->addJavaScript(
            'ig-fibu-printPreview',
            'EXT:ig_fibu/Resources/Public/JavaScript/printPreview.js'
        );

        $this->view->assign('now', new DateTime());

        if (!isset($this->search['letterDate']['date']) || $this->search['letterDate']['date'] == '') {
            $this->search['letterDate'] = [
                'dateFormat' => 'Y-m-d',
                'date' => date('Y-m-d'),
            ];
        }
        //var_dump($this->search);exit();

        $invoiceQueryBuilder = $this->invoiceRepository->findBySearch($this->search);
        //$invoices = $this->invoiceRepository->findBySearchExecute($invoiceQueryBuilder);
        //$this->view->assign('invoices', $invoices);
        $invoiceCounts = $this->invoiceRepository->countBySearchExecute($invoiceQueryBuilder);
        $this->view->assign('invoiceCounts', $invoiceCounts);

        $invoiceUids = $this->invoiceRepository->uidBySearchExecute($invoiceQueryBuilder);

        $invoicePages = $this->getPrintPagesByUids($invoiceUids, $this->search);
        //var_dump( $invoicePages);exit(0);
        $this->view->assign('invoicePages', $invoicePages);
        $searchForGroupdByStatus = $this->search;
        $searchForGroupdByStatus['isFinished'] = 0;
        //$searchForGroupdByStatus['deliveryMethod'] = 'print';
        unset($searchForGroupdByStatus['status']);
        $invoicesGroupedByStatus = $this->invoiceRepository->findGroupByStatus($searchForGroupdByStatus);
        if (isset($this->search['status']) && is_array($this->search['status'])) {
            $searchStatus = $this->search['status'];
        } else {
            $searchStatus = isset($this->search['status']) ? GeneralUtility::intExplode(
                ',',
                $this->search['status'],
                true
            ) : [];
        }
        foreach ($invoicesGroupedByStatus as &$invoiceGroup) {
            $invoiceGroup['checked'] = in_array($invoiceGroup['uid'], $searchStatus);
            $search = $this->search;
            $search['status'] = (int)$invoiceGroup['uid'];
            $invoiceGroup['search'] = $search;
        }
        $this->view->assign('invoicesGroupedByStatus', $invoicesGroupedByStatus);
        
        $this->view->assign('redirectUrl', $this->redirectUrl);
        $this->invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);
        $this->view->assign('config', $this->debitorService->getAllConfig());
        $this->assignTotals();
        $this->assignFormfields();
        $this->assignPeriode();
        $this->view->assign('search', $this->search);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action printPdf
     */
    public function printPdfAction(): ResponseInterface
    {
        $invoiceQueryBuilder = $this->invoiceRepository->findBySearch($this->search);

        $additionalFilename = $this->getAdditionalFilename();
        return $this->getPdfContent($invoiceQueryBuilder, $additionalFilename);
        /*
        $response = $this->responseFactory->createResponse()
                                          ->withHeader('Content-Type', 'application/pdf')
                                          ->withHeader('Expires', '0')
                                          ->withHeader('Cache-Control', 'private')
                                          ->withHeader('Content-Disposition', 'attachment; filename="' . $filename . '"')
                                          ->withHeader('Content-Length', (string)$fileLength);
        $response->getBody()->write($xml);
        // The exception code is the HTTP status code to use (in this case 200).
        // DA: Status code nicht in V. 11 aber evtl. in PSR 17 în V. 12?
        throw new PropagateResponseException($response, 200);
        */
    }
    /**
     * action print Html
     */
    public function printHtmlAction(): ResponseInterface
    {
        $invoiceQueryBuilder = $this->invoiceRepository->findBySearch($this->search);
        $letterDate = $this->search['letterDate']['date'] ?? date('Y-m-d');

        $additionalFilename = $this->getAdditionalFilename();

        $invoiceUids = $this->invoiceRepository->uidBySearchExecute(clone $invoiceQueryBuilder);

        $arguments = [
            'letterDate' => $this->search['letterDate'] ?? [],
        ];
        if ($this->search['sorting'] ?? false) {
            $arguments['sorting'] = $this->search['sorting'];
        }
        $arguments['uids'] = implode(',', $invoiceUids);

        $invoicePage = [
            'uids' => $invoiceUids,
            'arguments' => http_build_query([
                'search' => $arguments,
            ]),
            'search' => $this->search,
        ];
        $this->view->assign('invoicePage', $invoicePage);

        $this->invoiceUtility->setPdfOutputMode(InvoiceUtility::OUTPUT_HTML_DIE);
        $this->maxQrInvoice = $this->maxQrInvoiceHtml;
        return $this->htmlResponse($this->getPdfContent($invoiceQueryBuilder, $additionalFilename, $letterDate));
    }
    /**
     * action ajaxPdf
     */
    public function ajaxPdfAction(): ResponseInterface
    {
        if (isset($_GET['search'])) {
            $this->search = $_GET['search'];
        }
        $page = (int)($_GET['page'] ?? 0);
        $additionalFilename = $_GET['additionalFilename'] ?? '';

        $letterDate = $this->search['letterDate']['date'] ?? date('Y-m-d');
        $invoiceQueryBuilder = $this->invoiceRepository->findBySearch($this->search);

        if (!$additionalFilename) {
            $additionalFilename = $this->getAdditionalFilename();
        }
        $this->invoiceUtility->setPdfOutputMode(InvoiceUtility::OUTPUT_PDF_RETURN);
        $pdfContent = $this->getPdfContent($invoiceQueryBuilder, $additionalFilename, $letterDate);

        $filename = $this->invoiceUtility->getPdfFilename($additionalFilename, $page);


        $debitorErrorCount = $this->invoiceUtility->getDebitorErrorCount();
        // return json with base64 pdf content
        $pdfBase64 = base64_encode($pdfContent);
        $json = json_encode([
            'pdf' => $pdfBase64,
            'filename' => $filename,
            'debitorErrorCount' => $debitorErrorCount,
        ]);
        $response = $this->responseFactory->createResponse()
                                          ->withHeader('Content-Type', 'application/json; charset=utf-8')
                                          ->withHeader('Expires', '0')
                                          ->withHeader('Cache-Control', 'private');
        $response->getBody()
            ->write($json);
        throw new PropagateResponseException($response, 200);

        /*
        // return pdf content only
        $response = $this->responseFactory->createResponse()
                                              ->withHeader('Content-Type', 'application/pdf')
                                              ->withHeader('Expires', '0')
                                              ->withHeader('Cache-Control', 'private')
                                              ->withHeader('Content-Disposition', 'attachment; filename="' . $filename . '"')
                                              ->withHeader('Content-Length', (string)$fileLength);
            $response->getBody()->write($pdfContent);
            throw new PropagateResponseException($response, 200);
        */
    }
    
    /**
     * action ajaxSetSend
     */
    public function ajaxSetSendAction(): ResponseInterface
    {
        if (isset($_GET['search'])) {
            $this->search = $_GET['search'];
        }
        $letterDate = $this->search['letterDate']['date'] ?? date('Y-m-d');
        $undo = $_GET['undo'] ?? false;

        $this->search['letterDate'] = $letterDate;

        $invoicesUids = GeneralUtility::intExplode(',', $this->search['uids'] ?? '', true);
        $notUpdatedInvoiceUids = [];
        
        if ($undo) {
            $notUpdatedInvoiceUids = $this->invoiceRepository->updateSetUnsend($invoicesUids);
        } else {
            $notUpdatedInvoiceUids = $this->invoiceRepository->updateSetSend(
                $invoicesUids,
                $letterDate,
                InvoiceDate::DELIVERY_METHOD_DOWNLOAD
            );
        }
        
        $data = [
            'error' => false,
            'action' => $undo ? 'undo' : 'setSend',
            'invoiceUids' => $invoicesUids,
            'notUpdatedInvoiceUids' => $notUpdatedInvoiceUids,
            'letterDate' => $letterDate,
        ];

        $json = json_encode($data);
        // Content-Type und Inhalte setzen
        //$this->jsonResponse($json);
        $response = $this->responseFactory->createResponse()
                                          ->withHeader('Content-Type', 'application/json; charset=utf-8')
                                          ->withHeader('Expires', '0')
                                          ->withHeader('Cache-Control', 'private');
        $response->getBody()
                 ->write($json);
        

        throw new PropagateResponseException($response, 200);
    }
    /**
     * action invoicePrintHtml
     */
    public function initInvoicePrintHtmlAction(
        InvoiceInterface $invoice,
        InvoiceDate $invoiceDate = null
    ): ResponseInterface {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $usePayments = $this->request->hasArgument('usePayments') ? $this->request->getArgument('usePayments') : true;
        $this->invoiceUtility->setPdfOutputMode(InvoiceUtility::OUTPUT_HTML_DIE);
        $language = $GLOBALS['TYPO3_REQUEST']->getAttribute('language', null);
        $this->invoiceUtility->outInvoice($this->view, [
            0 => $invoice,
        ], $language, $this->search, $usePayments, $invoiceDate);
        //$html = $this->view->render();
        //$pdf = GeneralUtility::makeInstance(PdfUtility::class);
        //$pdf->debug($html . $this->invoiceUtility->getJavaScriptPrintDialog());
    }

    /**
     * action invoicePrint
     */
    public function initInvoicePrintAction(
        InvoiceInterface $invoice,
        InvoiceDate $invoiceDate = null
    ): ResponseInterface {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $usePayments = $this->request->hasArgument('usePayments') ? $this->request->getArgument('usePayments') : true;

        $this->view->assign('type', 'pdf');

        $language = $GLOBALS['TYPO3_REQUEST']->getAttribute('language', null);

        $additionalFilename = $invoice->getUid();
        $this->invoiceUtility->outInvoice($this->view, [
            0 => $invoice,
        ], $language, $this->search, $usePayments, $invoiceDate, $additionalFilename);
        exit(0);
    }
    public function invoiceDateAction(
        InvoiceDate $invoiceDate = null
    ): ResponseInterface {
        $invoice = $invoiceDate->getInvoice();
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $fileSize = $invoiceDate->getPdfSize();
        if ($fileSize === null) {
            return $this->htmlResponse('no file found with uid=' . $invoiceDate->getUid());
        }
        $filename = 'invoice_' . $invoiceDate->getUid() . '.pdf';
        $response = $this->responseFactory->createResponse()
                  ->withHeader('Content-Type', 'application/pdf; charset=utf-8')
                  ->withHeader('Expires', '0')
                  ->withHeader('Cache-Control', 'private')
                  ->withHeader('Content-Disposition', 'attachment; filename="' . $filename . '"')
                  ->withHeader('Content-Length', (string)$fileSize);
        $response->getBody()->write($invoiceDate->getPdfContents());
        throw new PropagateResponseException($response, 200);
    }

    /**
     * action new
     */
    public function newAction(): ResponseInterface
    {
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/edit_invoice.js');
        $this->view->assign('redirectUrl', $this->redirectUrl);
        $this->invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);
        if ($this->request->hasArgument('data')) {
            $data = $this->request->getArgument('data');
            $invoice = GeneralUtility::makeInstance(Invoice::class);
            if ($data['verband'] ?? 0) {
                $invoice->setVerbandId((int) $data['verband']);
            }
            if ($data['debitorId'] ?? 0) {
                $invoice->setEntryId((int)$data['debitorId']);
            }
            $tenant = $invoice->getCreditor();
            if ($tenant instanceof ClientInterface) {
                $invoice->setTaxRate($tenant->getTaxRate());
            }
            $invoice->setPeriode(date('Y-m'));
        }
        $this->view->assign('newInvoice', $invoice);


        $this->assignFormfields();
        $this->assignEditFormfields();
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action create
     */
    public function initCreateAction(InvoiceInterface $newInvoice): ResponseInterface
    {
        $this->addFlashMessage('Rechnung wurde erstellt.', '', ContextualFeedbackSeverity::OK);
        $this->invoiceRepository->add($newInvoice);
        if ($this->redirectUrl) {
            return $this->redirectToUri($this->redirectUrl);
        }
        return $this->redirect('list');
    }

    /**
     * action edit
     */
    #[IgnoreValidation([])] // $invoice
    public function initEditAction(InvoiceInterface $invoice, array $search = []): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/edit_invoice.js');
        $this->view->assign('redirectUrl', $this->redirectUrl);
        $this->invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);

        $this->view->assign('config', $this->debitorService->getAllConfig());

        $this->view->assign('mandants', $this->debitorService->getMandants());
        //$this->view->assign('periodes', $this->debitorService->getPeriodes());

        
        $this->view->assign('currentConfig', $this->debitorService->getConfig($invoice->getTablenames()));

        $this->view->assign('invoice', $invoice);
        $this->view->assign('search', $search);
        $this->assignFormfields();
        $this->assignEditFormfields();
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action edit addresses
     */
    public function editAddressAction(Invoice $invoice, array $search = []): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/edit_invoice.js');
        $this->view->assign('redirectUrl', $this->redirectUrl);

        $this->view->assign('invoice', $invoice);
        $this->view->assign('search', $search);
        $this->assignFormfields();
        $this->assignEditFormfields();

        $invoiceAddress = trim(str_replace("\r", '', $invoice->getInvoiceAddress()));
        $postalAddress = trim(str_replace("\r", '', $invoice->getDebitor()->getPostalAddress()));

        $addressIsEqual = $invoiceAddress == $postalAddress;
        $this->view->assign('addressIsEqual', $addressIsEqual);
        $diff = $this->diff($invoiceAddress, $postalAddress);
        //echo($diff);exit(0);
        $this->view->assign('diff', $diff);
        //var_dump($invoiceAddress, $postalAddress,$addressIsEqual);exit();
        
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_datapool_fe/Resources/Public/JavaScript/dp-edit.js');
        return $this->htmlResponse($this->view->render());
    }



    public function diff(string $oldString, string $newString): string
    {
        $oldLines = explode("\n", $oldString);
        $newLines = explode("\n", $newString);
        $max = max(count($oldLines), count($newLines));
        $ret = '';
        for ($i = 0; $i < $max; $i++) {
            $old = $oldLines[$i] ?? '';
            $new = $newLines[$i] ?? '';
            if (trim($old) == trim($new)) {
                $ret .= $old . "\n";
                continue;
            }
            $from_start = strspn($old ^ $new, "\0");
            $from_end = strspn(strrev($old) ^ strrev($new), "\0");

            $old_end = strlen($old) - $from_end;
            $new_end = strlen($new) - $from_end;

            $start = substr($new, 0, $from_start);
            $end = substr($new, $new_end);
            $new_diff = substr($new, $from_start, $new_end - $from_start);
            $old_diff = substr($old, $from_start, $old_end - $from_start);
            $ret .= $start;
            if ($old_diff) {
                $ret .= '<del>' . $old_diff . '</del>';
            }
            if ($new_diff) {
                $ret .= '<ins>' . $new_diff . '</ins>';
            }
            $ret .= $end . "\n";
        }
        return $ret . '<style>del {background-color:#ffcccc;} ins {background-color:#ccffcc;};</style>';
        //$new = "$start<ins style='background-color:#ccffcc'>$new_diff</ins>$end";
        //$old = "$start<del style='background-color:#ffcccc'>$old_diff</del>$end";
        //return array("old"=>$old, "new"=>$new, 'diff' => $diff);
    }
    
    /**
     * action sorting
     */
    #[IgnoreValidation([])] // $invoice
    public function initSortingAction(
        InvoiceInterface $invoice,
        InvoiceItem $invoiceItem,
        array $search = []
    ): ResponseInterface {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $invoiceItems = $invoice->getInvoiceItems();
        $invoiceItemBefore = null;
        $invoiceItemAfter = null;
        $found = false;
        $last = null;
        foreach ($invoiceItems as $item) {
            if ($found) {
                $invoiceItemAfter = $item;
                break;
            }
            if ($item == $invoiceItem) {
                $found = true;
                $invoiceItemBefore = $last;
            }
            $last = $item;
        }
        $sortingAction = $this->request->hasArgument('sortingAction') ? $this->request->getArgument(
            'sortingAction'
        ) : '';
        $this->invoiceItemRepository = GeneralUtility::makeInstance(InvoiceItemRepository::class);
        if ($sortingAction === 'down' && $invoiceItemAfter instanceof InvoiceItem) {
            $sorting = $invoiceItemAfter->getSorting();
            $invoiceItemAfter->setSorting($invoiceItem->getSorting());
            $invoiceItem->setSorting($sorting);
            $this->invoiceItemRepository->update($invoiceItemAfter);
            $this->invoiceItemRepository->update($invoiceItem);
        } elseif ($sortingAction === 'up' && $invoiceItemBefore instanceof InvoiceItem) {
            $sorting = $invoiceItemBefore->getSorting();
            $invoiceItemBefore->setSorting($invoiceItem->getSorting());
            $invoiceItem->setSorting($sorting);
            $this->invoiceItemRepository->update($invoiceItemBefore);
            $this->invoiceItemRepository->update($invoiceItem);
        }
        $this->invoiceRepository->update($invoice);
        return $this->redirect('show', null, null, [
            'invoice' => $invoice->getUid(),
            'search' => $this->search,
        ]);
    }
    public function multipleEditAction(): ResponseInterface
    {
        $newStatus = $this->request->hasArgument('newStatus') ? $this->request->getArgument('newStatus') : [];
        //$this->search = $this->getSearch();
        if ($newStatus) {
            $invoices = $this->invoiceFindBySearchRaw();
            // make sure the status exists
            $this->invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);
            $invoicestatus = $this->invoicestatusRepository->findByUid(intval($newStatus));
            //$invoicestatusUid = $invoicestatus->getUid();

            $count = 0;

            $search = $this->request->getArgument('search');
            $letterDate = $search['letterDate']['date'] ?? strftime('%Y-%m-%d', time());
            if ($invoicestatus && !empty($invoices)) {
                foreach ($invoices as $invoice) {
                    //$invoice->setStatus($invoicestatus);
                    $count++;
                    //$this->invoiceRepository->update($invoice);
                    $this->invoiceRepository->updateStatus($invoice['uid'], $invoicestatus, $letterDate);
                }
            }
            $this->addFlashMessage(
                'Rechnungsstatus wurde aktualisiert (Anzahl: ' . $count . ').',
                '',
                ContextualFeedbackSeverity::OK
            );
        }
        return $this->redirect('list', null, null, [
            'search' => $this->search,
        ]);
    }

    /**
     * action update
     */
    public function initUpdateAction(
        InvoiceInterface $invoice,
        array $search = [],
        array $control = []
    ): ResponseInterface {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $this->addFlashMessage('Rechnung wurde aktualisiert.', '', ContextualFeedbackSeverity::OK);

        $calculate = $control['calculate'] ?? true;
        if ($calculate) {
            $this->invoiceRepository->updateAndCalculate($invoice);
        } else {
            $this->invoiceRepository->update($invoice);
        }

        if ($this->redirectUrl) {
            return $this->redirectToUri($this->redirectUrl);
        }
        if (!empty($search)) {
            return $this->redirect('list', null, null, [
                'search' => $search,
            ]);
        }
        $arguments = [
            'invoice' => $invoice->getUid(),
        ];
        if ($this->request->hasArgument('activeTab')) {
            $arguments['activeTab'] = $this->request->getArgument('activeTab');
        }
        return (new ForwardResponse('show'))->withArguments($arguments);
    }

    /**
     * action delete
     */
    public function initDeleteAction(InvoiceInterface $invoice, array $search = []): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $this->addFlashMessage('Rechnung wurde gelöscht.', '', ContextualFeedbackSeverity::OK);
        $this->invoiceRepository->remove($invoice);
        if ($this->redirectUrl) {
            return $this->redirectToUri($this->redirectUrl);
        }
        return $this->redirect('list', null, null, [
            'search' => $search,
        ]);
    }
    /**
     * action multipleDelete
     */
    public function initMultipleDeleteAction(array $multipleDelete): ResponseInterface
    {
        //$this->search = $this->request->hasArgument('search') ? $this->request->getArgument('search') : [];
        $this->addFlashMessage('Rechnungen wurden gelöscht.', '', ContextualFeedbackSeverity::OK);
        // update or delete? + how do i get search in here?
        $this->invoiceRepository->findByUidAndDelete($multipleDelete);
        return $this->redirect('list', null, null, [
            'search' => $this->search,
        ]);
    }
    /**
     * action book
     */
    #[IgnoreValidation([])] // $invoice
    public function initAddPaymentAction(InvoiceInterface $invoice, Payment $payment): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        if ($payment->getIsDone() || $payment->getIsOk()) {
            $message = 'Zahlung kann nicht mit der Rechnung Nr. ' . $invoice->getUid() . ' verknüpft werden. Zahlung ist nicht offen/schon verbucht.';
            $this->addFlashMessage($message, '', ContextualFeedbackSeverity::ERROR);
        } else {
            $oldInvoice = $payment->getInvoice();
            $message = 'Zahlung wurde mit der Rechnung Nr. ' . $invoice->getUid() . ' verknüpft.';
            if ($oldInvoice) {
                $message .= ' (Frühere Verknüpfung: Rechnung Nr. ' . $oldInvoice->getUid() . ')';
            }
            $this->addFlashMessage($message, '', ContextualFeedbackSeverity::OK);
            $payment->setInvoice($invoice);
            $invoiceDebitorId = $invoice->getDebitorId();
            if ($invoiceDebitorId > 0 && !$payment->getDebitorId()) {
                $payment->setDebitorId($invoiceDebitorId);
            }
            $this->paymentRepository = GeneralUtility::makeInstance(PaymentRepository::class);
            $this->paymentRepository->update($payment);
        }
        return $this->redirect('show', null, null, [
            'invoice' => $invoice->getUid(),
            'activeTab' => 'payments', // von Zahlungen verbuchen mit ..
        ]);
    }
    public function initCalculateAction(InvoiceInterface $invoice, string $calculate = null): ResponseInterface
    {
        if ($calculate) {
            if ($calculate == 'amountPaid') {
                $oldValue = $invoice->getAmountPaid();
                $this->invoiceUtility->calculateAmountPaid($invoice);
                $this->invoiceRepository->update($invoice);
                $newValue = $invoice->getAmountPaid();
                $this->addFlashMessage(
                    'Zahlungseingänge der Rechnung wurde mit vorhandenen Zahlungen abgeglichen. (Neu: ' . number_format(
                        $newValue,
                        2
                    ) . ' statt ' . number_format($oldValue, 2) . ')',
                    '',
                    ContextualFeedbackSeverity::OK
                );
            } elseif ($calculate == 'prePaymentCreditAmount') {
                $oldValue = $invoice->getPrePaymentCreditAmount();
                $this->invoiceUtility->calculatePrePaymentCreditAmount($invoice);
                $this->invoiceRepository->update($invoice);
                $newValue = $invoice->getPrePaymentCreditAmount();
                $this->addFlashMessage(
                    'Gutschriften der Rechnung wurde mit vorhandenen Gutschriften (vor Zahlungen) abgeglichen. (Neu: ' . number_format(
                        $newValue,
                        2
                    ) . ' statt ' . number_format($oldValue, 2) . ')',
                    '',
                    ContextualFeedbackSeverity::OK
                );
            } else {
                $this->addFlashMessage(
                    'ERROR: "' . $calculate . '" not found to recalculate.',
                    '',
                    ContextualFeedbackSeverity::ERROR
                );
            }
        }
        return $this->redirect('show', null, null, [
            'invoice' => $invoice->getUid(),
        ]);
    }

    public function initRecalculateAction(): ResponseInterface
    {
        $this->invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);
        $statusUid = (int)($this->search['status'] ?? 0);
        $invoicestatus = $this->invoicestatusRepository->findOneByUid($statusUid);
        if ($invoicestatus && $invoicestatus->getCanEdit()) {
            $invoiceQueryBuilder = $this->invoiceRepository->findBySearch($this->search);
            $invoices = $this->invoiceRepository->findBySearchExecute($invoiceQueryBuilder);
            foreach ($invoices as $invoice) {
                $this->invoiceRepository->updateAndCalculate($invoice);
            }
        } else {
            $msg = 'recalculate not allowed ';
            if ($invoicestatus) {
                $msg .='  status "' . $invoicestatus->getTitle() . '")';
            } else {
                $msg .=' without status';
            }
            $this->addFlashMessage(
                'ERROR: ' . $msg ,
                '',
                ContextualFeedbackSeverity::ERROR
            );

        }
        return $this->redirect('list', null, null, [
            'search' => $this->search,
        ]);
    }

    /**
     * action book
     */
    #[IgnoreValidation([])] // $invoice
    public function initBookAction(InvoiceInterface $invoice): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }
        $bookingService = GeneralUtility::makeInstance(BookingService::class);
        $bookingService->setSettings($this->settings);

        $bookingService->bookInvoice($invoice);

        $this->addFlashMessage('Zahlungen wurden mit der Rechnung verbucht.', '', ContextualFeedbackSeverity::OK);
        $arguments = [
            'invoice' => $invoice->getUid(),
        ];
        if ($this->request->hasArgument('activeTab')) {
            $arguments['activeTab'] = $this->request->getArgument('activeTab');
        }
        return $this->redirect('show', null, null, $arguments);
    }
    

    public function getInvoiceHeader()
    {
        return '';
        /*return '<html xml:lang="de" lang="de" xmlns="http://www.w3.org/1999/xhtml">
</head>
<body>';*/
    }
    public function getInvoiceFooter()
    {
        return '';
        //return '</body></html>';
    }
    public function getDebugMail()
    {
        return $this->debugMail;
        //return $this->conf->getDebugMail();
    }
    public function getTestMail($verband, $search)
    {
        $debugMail = $this->getDebugMail();
        $frontendUser = $this->getFrontendUser();
        if ($debugMail) {
            // defined is this class
            return $debugMail;
        } elseif ($frontendUser['email'] ?? false) {
            // frontend user has email
            return $frontendUser['email'];
        } elseif ($search['verband'] ?? false) {
            // search for verband
            return $verband->getInvoiceTestEmail();
        }
        // no testMail
        return '';
    }

    /*
     * @todo aktuell werden nur invoice.uid,invoice.verband.uid in die Tabelle tx_igfibu_domain_model_mailqueue geschrieben
     */

    public function initAddToSendmailListAction(): ResponseInterface
    {
        return $this->htmlResponse('initAddToSendmailListAction no yet supported');
        /*
          $emails = $this->getInvoiceEmails();
          $count = 0;
          foreach ($emails as $email) {
              $count++;
              $this->mailQueueRepository->add($email);
          }
          $this->view->assign('invoices', $invoices);
          $this->view->assign('countInvoices', $count);
        */
    }
    
    public function initShowSendmailListAction(): ResponseInterface
    {
        $search = $this->request->hasArgument('search') ? $this->request->getArgument('search') : [];
        //$letterDate = $search['letterDate']['date'] ?? strftime('%Y-%m-%d', time());
        $referrer = $this->request->hasArgument('referrer') ? $this->request->getArgument('referrer') : 'list';

        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/ig_fibu.js');


        $invoiceQueryBuilder = $this->invoiceRepository->findSendMailList($search);
        
        $invoices = $this->invoiceRepository->findBySearchExecute($invoiceQueryBuilder);
        if (!empty($invoices)) {
            $invoice = $invoices[0];
            $verband = $invoice->getCreditor();
            $debugMail = $this->getDebugMail();
            $testMail = $this->getTestMail($verband, $search);
            $this->view->assign('testMail', $testMail);
            $this->view->assign('mailSend', $this->mailSend == 0 ? ' (Kein Versand)' : '');
            $this->view->assign('verband', $verband);
            if (count($invoices) == 1 && !$invoice->getStatus()->getCanSend()) {
                $invoiceDateTime = $invoice->getInvoiceDate();
                if ($invoiceDateTime) {
                    $letterDate = $invoiceDateTime->format('Y-m-d');
                    $search['letterDate'] = $letterDate;
                    $this->search['letterDate'] = $letterDate;
                }
            }
        }

        $this->view->assign('now', new DateTime());

        $this->view->assign('search', $this->search);
        $this->view->assign('referrer', $referrer);
        $this->view->assign('invoices', $invoices);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * prepare email to send for invoice
     */
    public function getInvoiceEmails(array $search): array
    {
        $invoiceQueryBuilder = $this->invoiceRepository->findSendMailList($search);
        $limit = $this->maxQrInvoice;
        $invoices = $this->invoiceRepository->findBySearchExecute($invoiceQueryBuilder, $limit);
        $htmlHeader = $this->getInvoiceHeader();
        $htmlFooter = $this->getInvoiceFooter();

        $emails = [];
        $periodUtility = GeneralUtility::makeInstance(PeriodUtility::class);
        
        foreach ($invoices as $invoice) {
            //$invoice->set( $r );
            //$bet = $invoice->getBetrieb();
            $debitor = $invoice->getDebitor();
            //$countInvoices++;

            // Fuer Javascript
            //$this->invoiceIds[] = $invoice->getUid();
            // bet_ra_versandart  0: Betrieb, 1 Post, 2 E-Mail


            //$debitor = $invoice->getDebitor();
            $mailTo = $debitor->getInvoiceEmail();

            $oneInvoicePerMail = $debitor->getOneInvoicePerMail();
            //var_dump($oneInvoicePerMail, $debitor->getUid());
            //echo('Test: ' . $r['bet_ra_versandart'] .','. $r['bet_ra_tel2'] .', E-Mail: '.$mailTo.'<br />');

            // Fuer jede Invoice ein einzelnes Mail (true) oder Pro Mailadresse ein Mail mit allen Invoices (false)

            if (isset($emails[$mailTo]) && is_array($emails[$mailTo]) && !$oneInvoicePerMail) {
                $emails[$mailTo]['invoices'][] = $invoice;
                $emails[$mailTo]['invoice_uid'] .= ',' . $invoice->getUid();
                $emails[$mailTo]['betrag_ez'] .= ',' . $invoice->getTotalExkl();
                $emails[$mailTo]['entry_ids'][] = $debitor->getUid();
                $emails[$mailTo]['countInvoices']++;
            } else {
                //$periode_name = $invoice->getPeriodeName();
                //$pdf_filename = $invoice->getFilename();//$invoice->get_text('filename_anfang').'-'.$invoice->get_periode_name_for_filename().'.pdf';
                $templateUtility = GeneralUtility::makeInstance(TemplateUtility::class);
                $contact = $invoice->getDebitor();
                $crmLanguageUid = $this->invoiceUtility->getCrmLanguageUidByContact($contact);
                $languageLocale = $this->invoiceUtility->getCrmLanguageLocaleByContact($contact);
                $templateUtility->setLanguageUid($crmLanguageUid);
                $templateUtility->setTenant($invoice->getVerband());
                $template = $templateUtility->getTemplate('Mail', $invoice, $contact);
                if ($template === null) {
                    $this->invoiceUtility->noTemplateFoundError('Mail', $invoice, $contact);
                }
                //$modelClassName = $this->debitorService->getModelName($invoice->getTablenames());
                //$template = $invoice->getVerbandTemplateMail($modelClassName);
                setlocale(LC_ALL, $languageLocale);
                setlocale(LC_TIME, $languageLocale);
                Locale::setDefault($languageLocale);
                $tenant = $invoice->getCreditor();
                $periodService = $periodUtility->getPeriodServiceByDuration(
                    $tenant->getInvoicePeriodeLengthInMonth() ?? 12
                );

                $periodDate = $periodService->createByValue($invoice->getPeriode());
                $periode_name = $periodService->getLabel($periodDate);
                
                $subject = $debitor->getInvoiceMailSubject() ?: $template->getTitle();
                $invoicestatus = $invoice->getStatus();
                $variables = [
                    '{invoice.periode}' => $invoice->getPeriode(),
                    '{invoice.status.documentTitle}' => $invoicestatus->getDocumentTitle(),
                    '{invoice.status.documentTitleShort}' => $invoicestatus->getDocumentTitleShort(),
                    '{periode_name}' => $periode_name,
                    '###PERIODE###' => $periode_name,
                ];
                $subject = $this->replaceFluidArray($subject, $variables);


                $invoiceMail = $oneInvoicePerMail ? $invoice->getUid() : $mailTo;
                
                $emails[$invoiceMail] = [
                    'mailTo' => $mailTo,
                    'subject' => $subject,
                    'invoices' => [$invoice],
                    //'pdf_filename' => $pdf_filename ,
                    'periode_name' => $periode_name,
                    'invoice_uid' => (string) $invoice->getUid(),
                    'betrag_ez' => $invoice->getBetragEz(), //getTotalExkl(),
                    'debitor' => $debitor,
                    'entry_ids' => [$debitor->getUid()],
                    'verband_uid' => $invoice->getCreditorId(),
                    'verband' => $invoice->getCreditor(),
                    'template' => $template,
                    'languageLocale' => $languageLocale,
                    'countInvoices' => 1,
                    /*
                      'bet_k' => $bet->getBetK(),
                      'bet_b' => $bet->getBetB(),
                      'bet_a' => $bet->getBetA(),
                      'bet_id' => $bet->getBetId()
                    */
                ];
            }
        }

        return $emails;
    }
    
    public function initSendmailListAction(): ResponseInterface
    {
        $content = '';
        $countMailSend = 0;
        $countInvoices = 0;
        $notUpdatedInvoiceUids = [];
        
        $search = $this->request->hasArgument('search') ? $this->request->getArgument('search') : [];

        $emails = $this->getInvoiceEmails($search);

        $letterDate = $search['letterDate']['date'] ?? strftime('%Y-%m-%d', time());
        $logMailRepository = GeneralUtility::makeInstance(LogMailRepository::class);
        if ($this->request->hasArgument('sendEmailToCustomer')) {
            $sendEmailToCustomer = $this->request->getArgument('sendEmailToCustomer');
        } else {
            $sendEmailToCustomer = false;
        }

        // set setup and constant for spamProtectEmailAddresses and update caches (ugly)
        // used in EmailLinkBuilder call in e.g. f:format.html to prevent spam protection
        $frontendTypoScript = $this->request->getAttribute('frontend.typoscript');
        if (method_exists($frontendTypoScript, 'getConfigTree')) {
            // v13
            // typo3/cms-frontend/Classes/Typolink/EmailLinkBuilder.php uses getConfigArray so setup is not necessary
            $setupTree = $frontendTypoScript->getSetupTree();
            $configTree = $frontendTypoScript->getConfigTree();
            $setupChildNode = $setupTree->getChildByName('config')->getChildByName('spamProtectEmailAddresses');
            $configChildNode = $configTree->getChildByName('spamProtectEmailAddresses');
            $backupSpamProtectEmailAddresses = $setupChildNode->getValue();
            $setupChildNode->setValue('0');
            $configChildNode->setValue('0');
            $frontendTypoScript->setSetupTree($setupTree);
            $frontendTypoScript->setSetupArray($setupTree->toArray());
            $frontendTypoScript->setConfigTree($configTree);
            $frontendTypoScript->setConfigArray($configTree->toArray()); // this is the important one for EmailLinkBuilder
            // not needed, frontendTypoScript is there a reference
            //$request = $request->withAttribute('frontend.typoscript', $frontendTypoScript);
        } else {
            // v12
            $backupSpamProtectEmailAddresses = $GLOBALS['TSFE']->config['config']['spamProtectEmailAddresses'];
            $GLOBALS['TSFE']->config['config']['spamProtectEmailAddresses'] = 'ascii';
        }

        $this->mailer = GeneralUtility::makeInstance(Mailer::class);
        $count = 0;
        $periodUtility = GeneralUtility::makeInstance(PeriodUtility::class);
        foreach ($emails as $email => $conf) {
            $count++;
            if ($count > $this->maxMailsToSent) {
                break;
            }
            if ($count > 1) {
                usleep(100 * 1000);//0.10 Sekunden
            }
            /*
            $variables = [
                          'letterDate' => $letterDate,
                          'invoices' => $conf['invoices'],
                          'useQr' => true,
                          ];
            */
            $this->invoiceUtility->setPdfOutputMode(InvoiceUtility::OUTPUT_PDF_RETURN);
            $language = $GLOBALS['TYPO3_REQUEST']->getAttribute('language', null);
            $usePayments = true;
            $attachmentPdfData = $this->invoiceUtility->outInvoice(
                $this->view,
                $conf['invoices'],
                $language,
                $this->search,
                $usePayments,
                null,
                null,
                $letterDate
            );

            $mail = GeneralUtility::makeInstance(FluidEmail::class);
            $mailTo = $conf['mailTo'];

            $invoice = $conf['invoices'][0];
            $template = $conf['template'];

            //$msg = $invoice->getEmailBody(count($conf['invoices']));
            $languageCode = $invoice->getDebitor()
->getLanguageCode();
            $tenant = $invoice->getCreditor();
            $languageLocale = $conf['languageLocale'];
            setlocale(LC_ALL, $languageLocale);
            setlocale(LC_TIME, $languageLocale);
            Locale::setDefault($languageLocale);
            $periodService = $periodUtility->getPeriodServiceByDuration(
                $tenant->getInvoicePeriodeLengthInMonth() ?? 12
            );
            $periodDate = $periodService->createByValue($invoice->getPeriode());
            $periode = $periodService->getLabel($periodDate);
            //$periode = $invoice->getPeriodeName($languageCode);
            //var_dump($periode,$invoice->getUid(),$language);exit(0);
            $language = $invoice->getDebitor()
->getLanguage();
            $conf['crm_language'] = is_object($language) ? $language->getUid() : $language;
            $verband = $conf['verband'];
            $formMail = $verband->getInvoiceFromEmail();
            if (!$formMail && method_exists($verband, 'getEmail')) {
                $formMail = $verband->getEmail();
            }
            $formName = $verband->getInvoiceFromName() ?: $verband->getName();
            $mail->from(new MimeAddress($formMail, $formName));
            $mail->subject($conf['subject']);

            if ($template->getFilename()) {
                $periodFileNamePart = $periodDate ? $periodService->getFilenamePart($periodDate) : '';
                //$attachmentFilename = str_replace('{period.fileNamePart}', $periodFileNamePart, $template->getFilename());
                //if (count($conf['invoices']) == 1) {
                $statusFilename = $invoice->getStatus()
->getFilenamePart() ?: str_replace(' ', '-', $invoice->getStatus()->getTitle());

                $replaces = [
                    '{period.fileNamePart}' => $periodFileNamePart,
                    '{invoice.status.fileNamePart}' => $statusFilename,
                    '{period.filenamePart}' => $periodFileNamePart,
                    '{invoice.status.filenamePart}' => $statusFilename,

                ];
                $attachmentFilename = str_replace(
                    array_keys($replaces),
                    array_values($replaces),
                    $template->getFilename()
                );
                //var_dump($attachmentFilename, $periodService->getFilenamePart($periodDate), $periodDate);exit(0);
                //$attachmentFilename = str_replace('{period.fileName}', $invoice->getPeriodeFileName($languageCode), $template->getFilename());
            } else {
                $attachmentFilename = 'invoice.pdf';
            }
            //var_dump($template->getFilename(), $periode, $attachmentFilename);exit(0);

            $mail->attach($attachmentPdfData, $attachmentFilename, 'application/pdf');

            //$mail->setBody($body, 'text/html');
            //$mail->setRequest($this->request->withControllerName('Default'));
            $mail->setTemplate('Invoice')
                ->setRequest($this->request->withControllerName('Default'))
                  ->assignMultiple([
                      //'msg' => $msg,
                      'countInvoices' => count($conf['invoices']),
                      'periode' => $periode,
                      'verband' => $verband,
                      'template' => $template,
                      'debitor' => $conf['debitor'],
                      'invoices' => $conf['invoices'],
                      'invoice' => $invoice,
                  ]);
            //$mail->attach($attachment);
            $debugMail = $this->getDebugMail();
            $showInfo = false;

            if ($sendEmailToCustomer) {
                $bccMail = $debugMail ?: $verband->getInvoiceBccEmail();
                if ($bccMail) {
                    $mail->bcc($bccMail);
                    $showInfo = 'bcc';
                }
            } else {
                $testMail = $this->getTestMail($verband, $search);
                if ($testMail) {
                    //$mail->subject($conf['subject'] .' (' . $conf['bet_k'] . $conf['bet_b'] . $conf['bet_a'] .')');
                    $mail->subject($conf['subject'] . ' (Testlauf für ' . $mailTo . ')');
                    $mailTo = $testMail;
                    $showInfo = 'test';
                } else {
                    return $this->htmlResponse('Error: es ist keine Test E-Mail hinterlegt');
                }
            }
            if ($debugMail) {
                $mailTo = $debugMail;
            }
            $mailToEmails = GeneralUtility::trimExplode(',', str_replace([';', '/', ' '], ',', $mailTo), true);
            //$mailTos = GeneralUtility::trimExplode(',', $mailTo, true);
            $mailToIsFirst = true;
            $errorMails = [];
            foreach ($mailToEmails as $mailToEmail) {
                $mailToEmailAscii = idn_to_ascii($mailToEmail);
                if (filter_var($mailToEmailAscii, FILTER_VALIDATE_EMAIL)) {
                    if ($mailToIsFirst) {
                        $mailToIsFirst = false;
                        $mail->to($mailToEmail);
                    } else {
                        $mail->addTo($mailToEmail);
                    }
                } else {
                    $errorMails[] = $mailToEmail;
                }
            }
            $content .= '<li>E-Mail verschickt an: ' . htmlspecialchars((string) $mailTo);
            if ($showInfo) {
                //if ($mailTo!=$conf['mailTo']) {
                if ($showInfo == 'test') {
                    $content .= ' (würde dann an: ' . $conf['mailTo'] . ')';
                } elseif ($showInfo == 'bcc') {
                    $content .= ' ( ist auch BCC an ' . $bccMail . ')';
                } else {
                    $content .= ' (showInfo not known)';
                }
            }
            if (!empty($errorMails)) {
                $content .= ' <span class="message-error">(Ungültige E-Mail Adressen: ' . implode(
                    ',',
                    $errorMails
                ) . ')</span>';
            }
            //$content.= '<li>E-Mail verschickt an: '. htmlspecialchars($mailTo) . ( $debugMail ? ' (würde dann an: '. $conf['mailTo'] .')' : '' ) . '[File: '.$conf['pdf_filename'].'] '.$subject.'</li>';
            if ($this->mailSend && $mailToIsFirst === false) {
                // send_status existiert nicht mehr
                try {
                    $mail->setRequest($this->request->withControllerName('Default'));
                    $send_status = $this->mailer->send($mail);
                } finally {
                    // restore spamProtectEmailAddresses in setup and constant
                    if (method_exists($frontendTypoScript, 'getConfigTree')) {
                        $setupChildNode->setValue($backupSpamProtectEmailAddresses);
                        $frontendTypoScript->setSetupTree($setupTree);
                        $frontendTypoScript->setSetupArray($setupTree->toArray());
                        $configChildNode->setValue($backupSpamProtectEmailAddresses);
                        $frontendTypoScript->setConfigTree($configTree);
                        $frontendTypoScript->setConfigArray($configTree->toArray());
                    } else {
                        $GLOBALS['TSFE']->config['config']['spamProtectEmailAddresses'] = $backupSpamProtectEmailAddresses;
                    }                    
                }
                /*
                var_dump($mail->getTo());
                echo('<b>Subject: ' . $mail->getSubject() .'</b><br /><br />');
                echo('<pre><code>'. htmlspecialchars($mail->toString()) . '</code></pre>');
                */
                //echo('<textarea cols="200" rows="200">Wx' . $mail->getBody()->bodyToString() . '</textarea>');
                //die('mail-send DIE-3242423');
                if ($sendEmailToCustomer) { // && $send_status>=1) {
                    $invoicesUids = GeneralUtility::intExplode(',', $conf['invoice_uid'], true);
                    // erstes Argument ist string z.B. "1,2,3" d.h. koennte auch am Schluss gemacht werden - sinnvoll bei Absturz?
                    // Log message send
                    $logMailUid = $logMailRepository->insertInvoice($mailTo, $conf);
                    $notUpdatedInvoiceUids = $this->invoiceRepository->updateSetSend(
                        $invoicesUids,
                        $letterDate,
                        InvoiceDate::DELIVERY_METHOD_EMAIL,
                        $logMailUid,
                        $attachmentPdfData
                    );
                    if (!empty($notUpdatedInvoiceUids)) {
                        $content .= ' <span class="message-error">(Rechnungsstatus nicht geändert: ' . implode(
                            ',',
                            $notUpdatedInvoiceUids
                        ) . ')</span>';
                    }
                }
            }
            $content .= '</li>';
            $countMailSend++;
            $countInvoices += $conf['countInvoices'];
        }
        //  [MAIL][format] = both
        //die('mail NOT send DIE-43593459737');
        //tml = '<ul>'. $content .'</ul><p><b>Total Rechnungen verschickt: ' . $countInvoices .'<br />Total E-Mails verschickt: ' . $countMailSend .'</b></p>';
        $total = [
            'invoices' => $countInvoices,
            'mailSend' => $countMailSend,
        ];
        
        $this->view->assignMultiple([
            'content' => $content,
            'total' => $total,
            'search' => $search,
        ]);
        // template is set in outInvoice
        $this->view->setTemplate('SendmailList.html');

        return $this->htmlResponse($this->view->render());
    }

    /**
     * action import Form upload
     */
    public function importFormUploadAction(int $tenantUid): ResponseInterface
    {
        //if (!$this->securityUtility->aclCanRead($verband, ['fe_person_edit'])) {
        //    return $this->htmlResponse(self::ACCESS_DENIED);
        //}
        $this->view->assign('tenantUid', $tenantUid);
        return $this->htmlResponse($this->view->render());
    }
    /**
     * action import form config
     */
    public function importFormConfigAction(int $tenantUid, FileResource $importFile = null): ResponseInterface
    {
        //if (!$this->securityUtility->aclCanRead($verband, ['fe_person_edit'])) {
        //    return $this->htmlResponse(self::ACCESS_DENIED);
        //}
        if ($importFile === null) {
            $this->addFlashMessage('Kein File zum Importieren vorhanden.', '', ContextualFeedbackSeverity::ERROR);
            $args = [
                'tenantUid' => $tenantUid,
            ];
            return $this->redirectWithSearch('importFormUpload', '', null, $args);
            // debug add a file
            //$fileRepository = GeneralUtility::makeInstance(\TYPO3\CMS\Core\Resource\FileRepository::class);
            //$importFile = $fileRepository->findByUid(98);
        }
        $aliases = [
            'entry_id' => [
                'attributes' => ['Kundennummer', 'Nummer', 'Debitor', 'Kunde'],
            ],
            'invoiceitem.description' => [
                'attributes' => ['Beschreibung', 'Text', 'Rechnungstext'],
            ],
            'invoiceitem.quantity' => [
                'attributes' => ['Menge', 'Anzahl'],
            ],
            'invoiceitem.unit_price' => [
                'attributes' => ['Stückpreis'],
            ],
            'invoiceitem.amount' => [
                'attributes' => ['Rechnunsbetrag', 'Betrag'],
            ],
        ];
        $localPath = $importFile->getForLocalProcessing(false);
        $excelUtility = GeneralUtility::makeInstance(ExcelUtility::class);
        // attributes not shown for import selection
        $excludes = [
            'type',
            'verband_id',
            'subtotal',
            'subtotal_exkl',
            'total_tax',
            'total_discount',
            'total_exkl',
            'total',
            'invoiceitem',
            'tablenames',
            'position',
            'mode',
            'letter_date',
            'status',
            'art',
            'mitgliedschaft',
        ];
        $additionals = [
            'invoiceitem.description' => 'Rechnungseintrag: Beschreibung',
            'invoiceitem.quantity' => 'Rechnungseintrag: Menge',
            'invoiceitem.unit_price' => 'Rechnungseintrag: Stückpreis',
            'invoiceitem.amount' => 'Rechnungseintrag: Betrag',
        ];
        try {
            $importConfig = $excelUtility->config(
                'tx_igfibu_domain_model_invoice',
                $localPath,
                $aliases,
                $excludes,
                $additionals
            );
        } catch (Exception $e) {
            die('<h3>Exception ' . $e::class . '</h3> ' . $e->getMessage());
        }

        
        $tenant = $this->debitorService->getMandantByUid($tenantUid);
        $periodUtility = GeneralUtility::makeInstance(PeriodUtility::class);
        $invoicePeriodeLengthInMonth = $tenant->getInvoicePeriodeLengthInMonth() ?? 12;
        $periodService = $periodUtility->getPeriodServiceByDuration($invoicePeriodeLengthInMonth);
        $periodDates = $periodService->getRangeByCount(new DateTime(), 12, 0, true);
        $periods = $periodService->getValueLabel($periodDates);
        $this->view->assign('periods', $periods);


        $filters = [
            //'getKanton' => 'Kanton aus Abkürzung (z.B. BE)',
            //'getAnrede' => 'Anrede (Herr,Frau,Firma)',
            'getDebitorUidByOwnNumber' => 'Debitor mit eigener Nummer',
            'getBoolean' => 'Boolean Wert (j,y,x und Zahl>0 = true)',
            'getDate' => 'Datum (d.m.Y -> Y-m-d)',
            'getInt' => 'Integer Wert erzwingen (\' entfernen)',
            'getLanguage' => 'Sprache (d,f,e,i,D,F,E,I),',
        ];
        $this->view->assign('importConfig', $importConfig);
        $this->view->assign('updateModes', $this->updateModes);
        $this->view->assign('filters', $filters);
        $this->view->assign('tenantUid', $tenantUid);
        $this->view->assign('tenant', $tenant);
        $this->view->assign('importFile', $importFile);
        $this->view->assign('localPath', $localPath);
        //var_dump($localPath, $importFile);exit(0);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action import - Not working
     */
    public function importAction(int $tenantUid, int $fileUid, array $config, array $setting): ResponseInterface
    {
        $fileRepository = GeneralUtility::makeInstance(FileRepository::class);
        $importFile = $fileRepository->findByUid($fileUid);

        $debug = $this->request->hasArgument('test') ? (bool) $this->request->getArgument('test') : false;
        $invoiceCreateEmpty = $this->request->hasArgument('invoiceCreateEmpty') ? (bool) $this->request->getArgument(
            'invoiceCreateEmpty'
        ) : false;
        $tenant = $this->debitorService->getMandantByUid($tenantUid);
        $defaultTaxBehavior = $tenant->getTaxBehavior();
        $defaultTaxRate = $tenant->getTaxRate();
        /*
        $formInvoiceDate = $setting['invoiceDate'];
        $invoiceDateString = $formInvoiceDate['date'] ?? null;
        $invoiceDate =  new \DateTime($invoiceDateString);
        */
        $period = $this->request->hasArgument('period') ? (bool) $this->request->getArgument(
            'period'
        ) : $setting['period'];

        $localPath = $importFile->getForLocalProcessing(false);
        $this->importUtility = GeneralUtility::makeInstance(ImportUtility::class);
        $importConfig = GeneralUtility::makeInstance(ImportConfig::class);
        $importConfig->setTablename('tx_igfibu_domain_model_invoice');

        $meRemark = 'IMPORT' . strftime('%Y%m%d');
        $importSource = strftime('%Y%m%d %H:%M');
        

        $importConfig->setDefaultValues(
            [
                'pid' => 2006,
                // 'source' => $importSource,
            ]
        );
        $importConfig->setCols($config);


        $this->importUtility->open($localPath, $importConfig);
        $updates = [];

        $totals = [
            'total' => 0,
            'invoices' => 0,
            'amount' => 0,
            'missingDebitor' => 0,
        ];
        $lastDebitorUid = 0;
        $lastInvoiceObject = null;
        while ($line = $this->importUtility->getLine()) {
            $data = $line['tx_igfibu_domain_model_invoice'];
            
            $debitorUid = (int)($data['entry_id'] ?? 0);
            $debitor = $debitorUid > 0 ? $this->debitorService->getDebitorById($debitorUid) : null;
            if ($debitor) {
                // add entries to last invoice
                $lastInvoiceTotal = 0;
                if ($lastDebitorUid === $debitorUid) {
                    $invoice = $lastInvoiceObject;
                    $lastInvoiceTotal = $invoice->getTotal();
                } else {
                    $invoice = GeneralUtility::makeInstance(\Internetgalerie\IgsCrm\Domain\Model\Invoice::class);
                    $taxBehavior = $defaultTaxBehavior;
                    $taxRate = $defaultTaxRate;
                    //$debitor = findByUid();
                    //$invoice->setEntryId($debitorUid);
                    $invoice->setDebitor('Person', $debitorUid);
                    $invoice->setTenantId(1);
                    $invoice->setStatusDefault();


                    $lastInvoiceObject = $invoice;
                    $lastDebitorUid = $debitorUid;
                    //$invoice->setInvoiceDate($invoiceDate->getInvoiceDate());
                    //var_dump($data);
                }
                if ($data['invoiceitem.amount'] ?? false || $data['invoiceitem.description'] ?? false) {
                    $quantity = (int)($data['invoiceitem.quantity'] ?? 0);
                    $unitPrice = (float)($data['invoiceitem.unit_price'] ?? 0.0);

                    if ($quantity > 0 && $unitPrice > 0.0) {
                        $invoiceItem = GeneralUtility::makeInstance(InvoiceItemUnit::class);
                    } else {
                        $invoiceItem = GeneralUtility::makeInstance(InvoiceItem::class);
                    }
                    $invoiceItem->setDescription($data['invoiceitem.description']);
                    if ($data['invoiceitem.quantity'] ?? false) {
                        $invoiceItem->setQuantity($quantity);
                    }
                    if ($data['invoiceitem.unit_price'] ?? false) {
                        $invoiceItem->setUnitPrice($unitPrice);
                    }
                    if ($period) {
                        $invoiceItem->setPeriod($period);
                    }

                    $amount = (float)($data['invoiceitem.amount'] ?? $quantity * $unitPrice);
                    //$invoiceItem->setAmount($amount);
                    
                    //if ($client->getTaxBehaviorIsExkl()) {
                    if (isset($setting['taxRate'])) {
                        $taxRate = (float) $setting['taxRate'];
                    } else {
                        $taxRate = null;
                    }
                    if (!isset($setting['taxBehavior']) || $setting['taxBehavior'] === null || $setting['taxBehavior'] === '') {
                        $taxBehavior = null;
                        $invoiceItem->setAmount($amount);
                        $invoiceItem->setTaxRate(0);
                        $invoiceItem->setTaxBehavior($taxBehavior);
                    }
                    if ($setting['taxBehavior'] == TaxBehavior::EXEMPT) {
                        $invoiceItem->setTotalExempt($amount);
                    } elseif ($setting['taxBehavior'] == TaxBehavior::INCLUSIVE) {
                        $taxBehavior = TaxBehavior::INCLUSIVE;
                        $invoiceItem->setTotalInkl($amount, $taxRate);// ?? $client->getTaxRate());
                    } else {
                        $taxBehavior = TaxBehavior::EXCLUSIVE;
                        $invoiceItem->setTotalExkl($amount, $taxRate);// ?? $client->getTaxRate());
                    }
                    $invoice->addInvoiceItem($invoiceItem);
                }
                $invoice->setTaxRate($taxRate);

                // todo remove?
                if ($period) {
                    $invoice->setPeriode($period);
                }

                $this->invoiceRepository->calculate($invoice);
                if ($invoice->getTotal() > 0.00 || $invoiceCreateEmpty) {
                    if (!$debug) {
                        $this->invoiceRepository->add($invoice);
                    }
                    if ($lastInvoiceTotal == 0.0) {
                        $totals['invoices']++;
                    }
                    $totals['amount'] += $invoice->getTotal() - $lastInvoiceTotal;
                }
            } else {
                $totals['missingDebitor']++;
                $invoice = null;
            }
            //var_dump($invoice);
            $totals['total']++;
            $update = [
                'isNew' => $invoice ? true : false,
                'invoice' => $invoice,
                'debitorUid' => $debitorUid,
            ];
            $updates[] = $update;
        }

        $this->importUtility->close();

        if ($debug) {
            $this->addFlashMessage(
                'Testlauf - es wurden KEINE Daten importiert.',
                '',
                ContextualFeedbackSeverity::WARNING
            );
        } else {
            $this->addFlashMessage('Daten wurden importiert.', '', ContextualFeedbackSeverity::OK);
        }
        
        
        $this->view->assign('debug', $debug);
        $this->view->assign('setting', $setting);
        $this->view->assign('tenant', $tenant);
        $this->view->assign('localPath', $localPath);
        $this->view->assign('importFile', $importFile);
        $this->view->assign('updates', $updates);
        $this->view->assign('totals', $totals);
        
        return $this->htmlResponse($this->view->render());
    }

    
    /**
     * assign for edit/new form
     */
    public function assignEditFormfields(): void
    {
        $costCenterRepository = GeneralUtility::makeInstance(CostCenterRepository::class);
        $costCenters = $costCenterRepository->findForVerbandGrouped($this->search['verband'] ?? null);
        $this->view->assign('costCenters', $costCenters);

        $this->view->assign('debitorSettings', $this->debitorService->getConfig(null));
        
        // assign products if available
        // @todo should be done with rest api/ajax in fluid (should be ok for less than 100 products)
        $productRepositoryClass = $this->settings['productTable']['repository'] ?? '';
        if (class_exists($productRepositoryClass)) {
            $productRepository = GeneralUtility::makeInstance($productRepositoryClass);
            $products = $productRepository->findAll();//ForVerband($this->search['verband'] ?? null);
            $this->view->assign('products', $products);
        }
    }

    /**
     * assign for all actions
     */
    public function assignFormfields(): void
    {
        $items = $GLOBALS['TCA']['tx_igfibu_domain_model_invoice']['columns']['type']['config']['items'];
        $optionsType = [];
        foreach ($items as $item) {
            $label = $item['label'] ?? $item[0] ?? '';
            $value = $item['value'] ?? $item[1] ?? '';
            if ($label ?? false) {
                $optionsType[$value] = LocalizationUtility::translate($label, 'ig_fibu') ?: $label;
            }
        }
        //var_dump($optionsType);exit(0);
        $this->view->assign('optionsType', $optionsType);

        $invoiceitemTypes = $GLOBALS['TCA']['tx_igfibu_domain_model_invoiceitem']['columns']['type']['config']['items'];
        
        $optionsInvoiceitemTypes = [];
        foreach ($invoiceitemTypes as $item) {
            $label = $item['label'] ?? $item[0] ?? '';
            $value = $item['value'] ?? $item[1] ?? '';
            if ($label ?? false) {
                $optionsInvoiceitemTypes[$value] = LocalizationUtility::translate($label, 'ig_fibu') ?: $label;
            }
        }
        $this->view->assign('invoiceitemTypes', $optionsInvoiceitemTypes);
        
        if ($this->invoicestatusRepository) {
            $invoicestatus = $this->invoicestatusRepository->findAll();
            $this->view->assign('invoicestatus', $invoicestatus);
        }
        $costCenterRepository = GeneralUtility::makeInstance(CostCenterRepository::class);

        
        if ($this->search['verband'] ?? 0) {
            $this->tenantId = (int)$this->search['verband'];
        } else {
            $this->tenantId = null;
        }
        
        $this->view->assign('searchFormCostCenters', $costCenterRepository->findForVerband($this->tenantId));
    }

    public function getAdditionalFilename(): string
    {
        return '';
    }
    public function assignList(): void
    {
        // allow to add assign to list in own controller
    }

    public function getFrontendUser()
    {
        $frontendUserAuthentication = $this->request->getAttribute('frontend.user');
        return $frontendUserAuthentication->user;
    }

    protected function getSearch()
    {
        $search = [];
        if ($this->request->hasArgument('@search')) {
            $search = $this->request->getArgument('@search');
        } elseif ($this->request->hasArgument('search')) {
            $search = $this->request->getArgument('search');
        } else {
            $search = $this->searchDefault;
        }

        return $search;
    }


    protected function invoiceFindBySearchRaw()
    {
        $queryBuilder = $this->invoiceRepository->findBySearch($this->search);
        return $queryBuilder->executeQuery()
            ->fetchAllAssociative();
    }

    protected function replaceFluidArray(string $subject, array $variables)
    {
        if (!empty($variables)) {
            foreach ($variables as $search => $replace) {
                $subject = str_replace($search, $replace, $subject);
            }
        }
        return $subject;
    }
}
