<?php

namespace Ig\IgFibu\Controller;

use DateTime;
use Ig\IgFibu\Domain\Repository\CreditRepository;
use Ig\IgFibu\Domain\Repository\InvoiceRepository;
use Ig\IgFibu\Domain\Repository\LogMailRepository;
use Ig\IgFibu\Domain\Repository\PaymentImportRepository;
use Ig\IgFibu\Domain\Repository\PaymentRepository;
use Ig\IgFibu\Service\DebitorService;
use Ig\IgFibu\Utility\ConfUtility;
use Ig\IgFibu\Utility\PeriodUtility;
use IntlDateFormatter;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;

class FibuStatistikController extends ActionController
{
    /**
     * creditRepository
     *
     * @var CreditRepository
     */
    protected $creditRepository = null;

    /**
     * invoiceRepository
     *
     * @var InvoiceRepository
     */
    protected $invoiceRepository = null;

    /**
     * paymentRepository
     *
     * @var PaymentRepository
     */
    protected $paymentRepository = null;

    /**
     * paymentImportRepository
     *
     * @var PaymentImportRepository
     */
    protected $paymentImportRepository = null;

    /**
     * logMailRepository
     *
     * @var LogMailRepository
     */
    protected $logMailRepository = null;


    /**
     * debitorService
     *
     * @var DebitorService
     */
    protected $debitorService = null;

    
    public function injectCreditRepository(CreditRepository $creditRepository): void
    {
        $this->creditRepository = $creditRepository;
    }

    
    public function injectInvoiceRepository(InvoiceRepository $invoiceRepository): void
    {
        $this->invoiceRepository = $invoiceRepository;
    }
    
    
    public function injectPaymentRepository(PaymentRepository $paymentRepository): void
    {
        $this->paymentRepository = $paymentRepository;
    }

    
    public function injectPaymentImportRepository(PaymentImportRepository $paymentImportRepository): void
    {
        $this->paymentImportRepository = $paymentImportRepository;
    }

    
    public function injectLogMailRepository(LogMailRepository $logMailRepository): void
    {
        $this->logMailRepository = $logMailRepository;
    }

    
    public function injectDebitorService(DebitorService $debitorService): void
    {
        $this->debitorService = $debitorService;
    }

    public function assignSearchForm($searchVerbandsUids): void
    {
        $this->view->assign('paymentsStatus', ConfUtility::$paymentsStatus);
        $this->view->assign('config', $this->invoiceRepository->getDebitorConfig());
        $this->view->assign('types', $this->invoiceRepository->findStatistikType($searchVerbandsUids));
        //var_dump($this->invoiceRepository->findStatistikType($searchVerbandsUids), $this->invoiceRepository->getDebitorConfig());exit(0);
        $this->view->assign('search', $this->search);
       
        $this->view->assign('years', $this->paymentRepository->getYears($searchVerbandsUids));
    }
    
    /**
     * action dashboard
     */
    public function dashboardAction(): ResponseInterface
    {
        if (!$this->debitorService->isAdmin()) {
            $invoicePid = (int)$this->settings['invoicePid'];
            $this->uriBuilder
                ->reset()
                ->setRequest($this->request)
                ->setTargetPageUid($invoicePid);
            $search = [];
            $uri = $this->uriBuilder->uriFor('list', [
                'search' => $search,
            ]);
            return $this->redirectToUri($uri, null, 303);
        }
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/ig_fibu.js');

        $this->search = $this->getSearch();
        $tenantUid = $this->assignTenantsGetCurrent();
        if ($tenantUid === null) {
            return $this->htmlResponse('');
        }
        $searchVerbandsUids = [$tenantUid];
        $this->assignSearchForm($searchVerbandsUids);
        $allMainStatus = $this->invoiceRepository->findStatistikMainStatus($searchVerbandsUids, $this->search);
        $statistikInvoiceStatus = array_filter($allMainStatus, fn($m) => !($m['ident'] ?? false));
        $this->view->assign(
            'statistikInvoiceStatus', [
                'list' => $allMainStatus, // including sub items -> for list
                'chart' => $statistikInvoiceStatus, // only main items -> for chart
            ]
        );

        
        $currentYear = date('Y');
        //$this->search['year'] = $year;
        //$this->search['period'] = $year;
        $this->view->assign('year', $currentYear);
        //$this->view->assign('searchCreateDateFrom', ['date' => $currentYear . '-01-01']);
        //$this->view->assign('searchCreateDateTo', ['date' => $currentYear . '-12-31']);


        $statistikInvoiceTopCustomers = [];
        for ($year = $currentYear; $year >= $currentYear - 1; $year--) {
            $data = $this->invoiceRepository->findStatistikInvoiceTopCustomers(
                $searchVerbandsUids,
                array_merge($this->search, [
                    'year' => $year,
                ])
            );
            if (!empty($data)) {
                $statistikInvoiceTopCustomers[] = [
                    'title' => 'Top-Kunden: ' . $year,
                    'data' => $data,
                    'search' => [
                        'createDateFrom' => [
                            'date' => $year . '-01-01',
                        ],
                        'createDateTo' => [
                            'date' => $year . '-12-31',
                        ],
                    ],
                ];
            }
        }
        $this->view->assign('statistikInvoiceTopCustomers', $statistikInvoiceTopCustomers);

        
        $statistikInvoiceItemCostCenter = [];
        for ($year = $currentYear; $year >= $currentYear - 1; $year--) {
            $data = $this->invoiceRepository->findStatistikInvoiceItemCostCenter(
                $searchVerbandsUids,
                array_merge($this->search, [
                    'year' => $year,
                ])
            );
            if (!empty($data)) {
                $total = $this->invoiceRepository->getStatistikTotal($data);
                $statistikInvoiceItemCostCenter[] = [
                    'title' => 'Erstellt: ' . $year,
                    'data' => $data,
                    'total' => $total,
                    'search' => [
                        'createDateFrom' => [
                            'date' => $year . '-01-01',
                        ],
                        'createDateTo' => [
                            'date' => $year . '-12-31',
                        ],
                    ],
                ];
            }
        }
        $periodUtility = GeneralUtility::makeInstance(PeriodUtility::class);
        $invoicePeriodeLengthInMonth = $this->tenant->getInvoicePeriodeLengthInMonth() ?? 12;
        $periodService = $periodUtility->getPeriodServiceByDuration($invoicePeriodeLengthInMonth);
        $periodDates = $periodService->getRangeByCount(new DateTime(), 2, 0, true);

        $formatter = IntlDateFormatter::create('de-DE', IntlDateFormatter::FULL, IntlDateFormatter::NONE);
        foreach ($periodDates as $periodDate) {
            //$datetime = new \DateTime($year . '-' . $period->getFromDate());
            //$formatter->setPattern($period->getValue());
            //$periodValue = $formatter->format($periodDate);
            $periodValue = $periodService->getValue($periodDate);
            $data = $this->invoiceRepository->findStatistikInvoiceItemCostCenter(
                $searchVerbandsUids,
                array_merge($this->search, [
                    'period' => $periodValue,
                ])
            );
            if (!empty($data)) {
                $total = $this->invoiceRepository->getStatistikTotal($data);
                $periodLabel = $periodService->getLabel($periodDate);
                $title = ($invoicePeriodeLengthInMonth == 12 ? 'Periode: ' : '') . $periodLabel;
                $statistikInvoiceItemCostCenter[] = [
                    'title' => $title,
                    'data' => $data,
                    'total' => $total,
                    'search' => [
                        'periode' => $periodValue,
                    ],
                ];
            }
        }

        //var_dump($this->tenant->getInvoicePeriodeLengthInMonth(), count($statistikInvoiceItemCostCenter));exit(0);
        $this->view->assign('statistikInvoiceItemCostCenter', $statistikInvoiceItemCostCenter);

        
        // payments
        $totalOpen = $this->paymentRepository->findTotalOpen($this->search);
        // payment import
        $lastImport = $this->paymentImportRepository->findLast($this->search);
        $this->view->assign('tabPayment', [
            'totalOpen' => $totalOpen,
            'lastImport' => $lastImport,
        ]);

        // open credits
        $this->view->assign(
            'statistikCreditOpen',
            $this->creditRepository->findStatistikOpen($searchVerbandsUids, $this->search)
        );


        return $this->htmlResponse($this->view->render());
    }


    /**
     * action overview: Statistik der Rechnunen und Payments
     */
    public function overviewAction(): ResponseInterface
    {
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/ig_fibu.js');

        $this->search = $this->getSearch();
        $tenantUid = $this->assignTenantsGetCurrent();
        if ($tenantUid === null) {
            return $this->htmlResponse('');
        }
        $searchVerbandsUids = [$tenantUid];
        $this->assignSearchForm($searchVerbandsUids);
        
        $this->view->assign(
            'statistikPeriodeJahr',
            $this->invoiceRepository->findStatistikPeriodeJahr($searchVerbandsUids, $this->search)
        );
        $this->view->assign(
            'statistikLogMail',
            $this->logMailRepository->findStatistikYearMonth($searchVerbandsUids, $this->search)
        );


        
        $this->view->assign(
            'statistikInvoiceStatus',
            $this->invoiceRepository->findStatistikStatus($searchVerbandsUids, $this->search)
        );
        $this->view->assign(
            'statistikInvoiceCreatedate',
            $this->invoiceRepository->findStatistikCreatedate($searchVerbandsUids, $this->search)
        );
        $this->view->assign(
            'statistikInvoicePaiddateJahr',
            $this->invoiceRepository->findStatistikPaiddateJahr($searchVerbandsUids, $this->search)
        );
        $this->view->assign(
            'statistikInvoicePaiddate',
            $this->invoiceRepository->findStatistikPaiddate($searchVerbandsUids, $this->search)
        );


        // invoices with costCenter (open)
        $statistikInvoiceItems = $this->invoiceRepository->findStatistikItems($searchVerbandsUids, $this->search);
        $this->view->assign('statistikInvoiceItemsYearMonth', $statistikInvoiceItems['yearMonth']);
        $this->view->assign('statistikInvoiceItemsYear', $statistikInvoiceItems['year']);

        // invoices with costCenter (all)
        $statistikInvoiceAllItems = $this->invoiceRepository->findStatistikItems(
            $searchVerbandsUids,
            $this->search,
            []
        );
        $this->view->assign('statistikInvoiceAllItemsYearMonth', $statistikInvoiceAllItems['yearMonth']);
        $this->view->assign('statistikInvoiceAllItemsYear', $statistikInvoiceAllItems['year']);

        // invoice combined with payments and memberships
        $useMembership = (bool)$this->settings['mandant']['active']; // use membership if mandant is active
        $statistikInvoiceGroupWithPayment = $this->invoiceRepository->findStatistikGroupWithPayment(
            $searchVerbandsUids,
            $useMembership,
            $this->search,
        );
        $this->view->assign('statistikInvoiceGroupWithPayment', $statistikInvoiceGroupWithPayment);
        $this->view->assign('useMembership', $useMembership);
        if ($useMembership) {
            $statistikInvoiceGroupWithPaymentTitle = 'Bezahlte Rechnungen/Zahlungen nach Mitgliedschaft (Jahr)';
        } else {
            $statistikInvoiceGroupWithPaymentTitle = 'Bezahlte Rechnungen/Zahlungen (Jahr)';
        }
        $this->view->assign('statistikInvoiceGroupWithPaymentTitle', $statistikInvoiceGroupWithPaymentTitle);

        
        $this->view->assign(
            'statistikPaymentStatus',
            $this->paymentRepository->findStatistikStatus($searchVerbandsUids, $this->search)
        );
        //$this->view->assign('statistikPaymentValutadateJahr', $this->paymentRepository->findStatistikValutadateJahr($searchVerbandsUids, $this->search));
        //$this->view->assign('statistikPaymentValutadate', $this->paymentRepository->findStatistikValutadate($searchVerbandsUids, $this->search));
        $this->view->assign(
            'statistikPaymentPaiddateJahr',
            $this->paymentRepository->findStatistikPaiddateJahr($searchVerbandsUids, $this->search)
        );
        $this->view->assign(
            'statistikPaymentPaiddate',
            $this->paymentRepository->findStatistikPaiddate($searchVerbandsUids, $this->search)
        );
        return $this->htmlResponse($this->view->render());
    }

    protected function getSearch()
    {
        if ($this->request->hasArgument('search')) {
            return $this->request->getArgument('search');
        }
        return [];
    }

    protected function assignTenantsGetCurrent()
    {
        $verbandsUids = [];
        if ($this->settings['mandant']['active']) {
            $verbands = $this->debitorService->getMandants(true);
            foreach ($verbands as $verband) {
                $verbandsUids[] = (int) $verband['uid'];
            }
            // no tenant
            if (empty($verbandsUids)) {
                return null;
            }
        } else {
            // set default
            $verbands = [];
        }

        $this->view->assign('verbands', $verbands);
        if (isset($this->search['verband']) && in_array((int)$this->search['verband'], $verbandsUids)) {
            $tenantUid = (int) $this->search['verband'];
        } else {
            // no search - take first
            $this->search['verband'] = $verbandsUids[0] ?? '';
            $tenantUid = $verbandsUids[0] ?? null;
        }
        $this->tenant = $this->debitorService->getMandantByUid($tenantUid);
        $tenantUid = $this->tenant->getUid() ?? 0;
        return $tenantUid;
    }
}
