<?php

namespace Ig\IgFibu\Domain\Model;

use DateTime;
use Ig\IgFibu\Domain\Repository\InvoiceDateRepository;
use Ig\IgFibu\Domain\Repository\InvoicePaymentRepository;
use Ig\IgFibu\Domain\Repository\InvoicestatusRepository;
use Ig\IgFibu\Domain\Repository\PaymentRepository;
use Ig\IgFibu\Service\DebitorService;
use Ig\IgFibu\TaxBehavior;
use Ig\IgFibu\Utility\ConfUtility;
use Ig\IgFibu\Utility\HelperUtility;
use Ig\IgFibu\Utility\InvoiceUtility;
use Internetgalerie\IgCrmTemplate\Domain\Model\TemplateObjectInterface;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\DomainObject\AbstractEntity;
use TYPO3\CMS\Extbase\Persistence\ObjectStorage;

class Invoice extends AbstractEntity implements InvoiceInterface, TemplateObjectInterface, FibuAclInterface
{
    /**
     * entryId
     *
     * @var int
     */
    protected $entryId = '';

    /**
     * type
     *
     * @var string
     */
    protected $type = '';

    /**
     * tablenames
     *
     * @var string
     */
    protected $tablenames = '';

    /**
     * position
     *
     * @var string
     */
    protected $position = '';

    /**
     * periode
     *
     * @var string
     */
    protected $periode = '';

    /**
     * mode
     *
     * @var int
     */
    protected $mode = 0;

    /**
     * createDate
     *
     * @var DateTime
     */
    protected $createDate = null;

    /**
     * invoiceDate
     *
     * @var DateTime
     */
    protected $invoiceDate = null;

    /**
     * paidDate
     *
     * @var DateTime
     */
    protected $paidDate = null;

    /**
     * subtotal
     *
     * @var float
     */
    protected $subtotal = 0.0;

    /**
     * subtotalExkl
     *
     * @var float
     */
    protected $subtotalExkl = 0.0;

    /**
     * discountPercentage
     *
     * @var float
     */
    protected $discountPercentage = 0.0;

    /**
     * discountAmount
     *
     * @var float
     */
    protected $discountAmount = 0.0;

    /**
     * totalDiscount
     *
     * @var float
     */
    protected $totalDiscount = 0.0;

    /**
     * tax rate (e.x. 7,7)
     *
     * @var float
     */
    protected $taxRate = 0.0;
    /**
     * rounding step e.g. 0.05 or 0.01
     */
    protected float $roundingStep = 0.05;

    /**
     * totalTax
     *
     * @var float
     */
    protected $totalTax = 0.0;

    /**
     * totalExkl
     *
     * @var float
     */
    protected $totalExkl = 0.0;

    /**
     * total
     *
     * @var float
     */
    protected $total = 0.0;

    /**
     * amountPaid
     *
     * @var float
     */
    protected $amountPaid = 0.0;

    /**
     * prePaymentCreditAmount
     *
     * @var float
     */
    protected $prePaymentCreditAmount = 0.0;

    /**
     * postPaymentCreditAmount
     *
     * @var float
     */
    protected $postPaymentCreditAmount = 0.0;
    
    /**
     * letterDate
     *
     * @var DateTime
     */
    protected $letterDate = null;

    /**
     * invoiceitem
     *
     * @var ObjectStorage<InvoiceItem>
     */
    #[TYPO3\CMS\Extbase\Annotation\ORM\Lazy]
    #[TYPO3\CMS\Extbase\Annotation\ORM\Cascade([
        'value' => 'remove',
    ])]
    protected $invoiceitem = null;

    /**
     * status
     *
     * @var Invoicestatus
     */
    protected $status = null;

    /**
     * art
     *
     * @var Invoicescheme
     */
    #[TYPO3\CMS\Extbase\Annotation\ORM\Lazy]
    protected $art = null;

    /**
     * verbandId
     *
     * @var int
     */
    protected $verbandId = 0;

    /**
     * debitor
     *
     * @var object
     */
    protected $debitor = false;

    /**
     * debitorLink
     *
     * @var array
     */
    protected $debitorLink = null;

    /**
     * debitorSettings
     *
     * @var array
     */
    protected $debitorSettings = null;


    /**
     * debitorService
     *
     * @var DebitorService
     */
    protected $debitorService = null;

    /**
     * initializeInvoicestatusUid
     *
     * @var int
     */
    protected $initializeInvoicestatusUid = 1;

    /**
     * verband
     *
     * @var object
     */
    protected $verband = null;

    /**
     * payments
     *
     * @var array
     */
    protected $payments = null;

    /**
     * payment totoal
     *
     * @var float
     */
    protected $paymentTotal = null;

    /**
     * invoice payment total
     *
     * @var float
     */
    protected $invoicePaymentTotal = null;

    /**
     * customerReference
     *
     * @var string
     */
    protected $customerReference = '';

    /**
     * description
     *
     * @var string
     */
    protected $description = '';

    /**
     * invoiceAddress
     *
     * @var string
     */
    protected $invoiceAddress = '';

    /**
     * deliveryAddress
     *
     * @var string
     */
    protected $deliveryAddress = '';

    /**
     * @var array
     */
    protected $invoicePayments = null;
    /**
     * @var array
     */
    protected $openPayments = null;

    /**
     * @var array
     */
    protected $invoicePaymentsCreditAll = null;
    /**
     * @var array
     */
    protected $invoicePaymentsCreditPre = null;
    /**
     * @var array
     */
    protected $invoicePaymentsCreditPost = null;

    /**
     * @var array
     */
    protected $invoicePaymentTotalPayment = null;

    protected ?int $colspan = null;
    protected ?InvoicePaymentRepository $invoicePaymentRepository = null;
    /**
     * __construct
     */
    public function __construct()
    {
        //Do not remove the next line: It would break the functionality
        $this->initStorageObjects();
        $this->createDate = new DateTime();
    }

    /**
     * @return DebitorService
     */
    public function getDebitorService()
    {
        if ($this->debitorService === null) {
            $this->debitorService = GeneralUtility::makeInstance(DebitorService::class);
        }
        return $this->debitorService;
    }

    /**
     * Returns the type
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * Sets the type
     */
    public function setType(string $type): void
    {
        $this->type = $type;
    }


    /**
     * Returns the position
     *
     * @return string
     */
    public function getPosition()
    {
        return $this->position;
    }

    /**
     * Sets the position
     *
     * @param  string $position
     */
    public function setPosition($position): void
    {
        $this->position = $position;
    }

    /**
     * Returns the periode
     *
     * @return string
     */
    public function getPeriode()
    {
        return $this->periode;
    }

    // used in Pedos Content.html
    public function getPeriodMonth()
    {
        return substr($this->periode, 5);
    }
    /*
    public function getPeriodeName($languageKey = null)
    {
        return $this->getDebitorService()->getPeriodeLabel($this->periode, $languageKey);
    }
    */
    public function getPeriodeFileName($languageKey = null)
    {
        $date = $this->periode;
        $year = substr($this->periode, 0, 4);
        $month = (int)$this->getPeriodMonth();
        if ($month >= 1 && $month <= 3) {
            return $year . '-' . ConfUtility::$quartale[1]['filename'][$languageKey];
        }
        if ($month >= 4 && $month <= 6) {
            return $year . '-' . ConfUtility::$quartale[2]['filename'][$languageKey];
        }
        if ($month >= 7 && $month <= 9) {
            return $year . '-' . ConfUtility::$quartale[3]['filename'][$languageKey];
        }
        if ($month >= 10 && $month <= 12) {
            return $year . '-' . ConfUtility::$quartale[4]['filename'][$languageKey];
        }

        return $date;
    }

    public function getLastPeriode()
    {
        return (int) $this->periode - 1;
    }

    /**
     * Sets the periode
     *
     * @param  string $periode
     */
    public function setPeriode($periode): void
    {
        $this->periode = $periode;
    }

    /**
     * Returns the mode
     *
     * @return int
     */
    public function getMode()
    {
        return $this->mode;
    }

    /**
     * Sets the mode
     *
     * @param  int $mode
     */
    public function setMode($mode): void
    {
        $this->mode = $mode;
    }

    /**
     * Returns the createDate
     *
     * @return DateTime
     */
    public function getCreateDate()
    {
        return $this->createDate;
    }

    /**
     * Sets the createDate
     */
    public function setCreateDate(DateTime $createDate = null): void
    {
        $this->createDate = $createDate;
    }

    /**
     * Returns the invoiceDate
     *
     * @return DateTime
     */
    public function getInvoiceDate()
    {
        return $this->invoiceDate;
    }
    public function getFirstInvoiceDate()
    {
        $history = $this->getHistory();
        if (empty($history)) {
            return $this->getInvoiceDate();
        }
        $invoiceDate = null;
        // newest entry with: Status = 2 ("Verschickt / Offen"),  with actgion delivery (mail or print) and invoiceDate
        foreach ($history as $log) {
            if ($log['status'] == 2 && $log['action'] == InvoiceDate::ACTION_DELIVERY && $log['invoiceDate'] !== null) {
                $invoiceDate = $log['invoiceDate'];
                break;
            }
        }
        // if no real delivery was found, take oldest with status = 2
        if ($invoiceDate === null) {
            foreach ($history as $log) {
                if ($log['status'] == 2 && $log['createDate'] !== null) {
                    $invoiceDate = $log['invoiceDate'] ?? $log['createDate'];
                    break;
                }
            }
        }
        // nothing found, take oldes entry
        if ($invoiceDate === null) {
            $log = end($history);
            $invoiceDate = $log['invoiceDate'];
        }

        return $invoiceDate;
    }

    /**
     * Sets the invoiceDate
     */
    public function setInvoiceDate(DateTime $invoiceDate = null): void
    {
        $this->invoiceDate = $invoiceDate;
    }

    
    /**
     * Returns the paidDate
     *
     * @return DateTime
     */
    public function getPaidDate()
    {
        return $this->paidDate;
    }

    /**
     * Sets the paidDate
     */
    public function setPaidDate(DateTime $paidDate = null): void
    {
        $this->paidDate = $paidDate;
    }

    /**
     * Returns the subtotal
     *
     * @return float
     */
    public function getSubtotal()
    {
        return $this->subtotal;
    }

    /**
     * Sets the subtotal
     *
     * @param  float $subtotal
     */
    public function setSubtotal($subtotal): void
    {
        $this->subtotal = $subtotal;
    }

    /**
     * Returns the subtotalExkl
     *
     * @return float
     */
    public function getSubtotalExkl()
    {
        return $this->subtotalExkl;
    }

    /**
     * Sets the subtotalExkl
     *
     * @param  float $subtotalExkl
     */
    public function setSubtotalExkl($subtotalExkl): void
    {
        $this->subtotalExkl = $subtotalExkl;
    }

    /**
     * Returns the discountPercentage
     *
     * @return float
     */
    public function getDiscountPercentage()
    {
        return $this->discountPercentage;
    }

    /**
     * Sets the discountPercentage
     *
     * @param  float $discountPercentage
     */
    public function setDiscountPercentage($discountPercentage): void
    {
        $this->discountPercentage = $discountPercentage;
    }
    
    /**
     * Returns the discountAmount
     *
     * @return float
     */
    public function getDiscountAmount()
    {
        return $this->discountAmount;
    }

    /**
     * Sets the discountAmount
     *
     * @param  float $discountAmount
     */
    public function setDiscountAmount($discountAmount): void
    {
        $this->discountAmount = $discountAmount;
    }

    /**
     * Returns the totalDiscount
     *
     * @return float
     */
    public function getTotalDiscount()
    {
        return $this->totalDiscount;
    }

    /**
     * Sets the totalDiscount
     *
     * @param  float $totalDiscount
     */
    public function setTotalDiscount($totalDiscount): void
    {
        $this->totalDiscount = $totalDiscount;
    }


    /**
     * Returns the mwst as float
     */
    public function getTaxRate(): float
    {
        return $this->taxRate;
    }
    public function getMwst()
    {
        return $this->taxRate;
    }

    public function setTaxRate(float $taxRate): void
    {
        $this->taxRate = $taxRate;
    }


    public function getRoundingStep(): float
    {
        return $this->roundingStep;
    }
    public function setRoundingStep(float $roundingStep): void
    {
        $this->roundingStep = $roundingStep;
    }

    
    /**
     * Returns the total MWST
     *
     * @return float
     */
    public function getTotalMwst()
    {
        return $this->totalTax;
        //return $this->totalExkl *  $this->taxRate / 100;
        //return $this->total - $this->totalExkl;
    }
    
    /**
     * Returns the totalTax
     *
     * @return float
     */
    public function getTotalTax()
    {
        return $this->totalTax;
    }

    /**
     * Sets the totalTax
     *
     * @param  float $totalTax
     */
    public function setTotalTax($totalTax): void
    {
        $this->totalTax = $totalTax;
    }
    
    /**
     * Returns the totalExkl
     *
     * @return float
     */
    public function getTotalExkl()
    {
        return $this->totalExkl;
    }
    /**
     * Returns the totalInkl
     *
     * @return float
     */
    public function getTotalInkl()
    {
        return $this->total;
    }

    /**
     * Sets the totalExkl
     *
     * @param  float $totalExkl
     */
    public function setTotalExkl($totalExkl): void
    {
        $this->totalExkl = $totalExkl;
    }

    /**
     * Returns the total
     *
     * @return float
     */
    public function getTotal()
    {
        return $this->total;
    }
    public function getBetragEz()
    {
        return $this->total;
    }
    public function getAmountOpenWithPostPaymentCreditAmount()
    {
        if ($this->total === 0.0) {
            return 0.0;
        }
        return $this->getAmountRemaining() - $this->getPostPaymentCreditAmount();
        //return $this->total - $this->getPaymentsTotal();
    }
    public function getAmountOpen()
    {
        if ($this->total === 0.0) {
            return 0.0;
        }
        return $this->getAmountRemaining();
        //return $this->total - $this->getPaymentsTotal();
    }
    public function getAmountOpenMinus()
    {
        return -$this->getAmountOpen();
    }
    public function getAmountToPay()
    {
        if ($this->total === 0.0) {
            return 0.0;
        }
        return $this->total - $this->getInvoicePaymentsTotalCredit();
        //return $this->total - $this->getPaymentsTotal();
    }

    /**
     * Sets the total
     *
     * @param  float $total
     */
    public function setTotal($total): void
    {
        $this->total = $total;
    }

    public function getPrePaymentCreditAmount(): float
    {
        return $this->prePaymentCreditAmount;
    }

    public function setPrePaymentCreditAmount(float $prePaymentCreditAmount): void
    {
        $this->prePaymentCreditAmount = $prePaymentCreditAmount;
    }

    public function addPrePaymentCreditAmount($prePaymentCreditAmount): void
    {
        $this->prePaymentCreditAmount += $prePaymentCreditAmount;
    }

    public function subPrePaymentCreditAmount($prePaymentCreditAmount): void
    {
        $this->prePaymentCreditAmount -= $prePaymentCreditAmount;
    }


    public function getPostPaymentCreditAmount(): float
    {
        return $this->postPaymentCreditAmount;
    }

    public function setPostPaymentCreditAmount(float $postPaymentCreditAmount): void
    {
        $this->postPaymentCreditAmount = $postPaymentCreditAmount;
    }

    public function addPostPaymentCreditAmount($postPaymentCreditAmount): void
    {
        $this->postPaymentCreditAmount += $postPaymentCreditAmount;
    }
    
    public function subPostPaymentCreditAmount($postPaymentCreditAmount): void
    {
        $this->postPaymentCreditAmount -= $postPaymentCreditAmount;
    }
    

    
    public function getAmountPaid(): float
    {
        return $this->amountPaid;
    }

    public function setAmountPaid(float $amountPaid): void
    {
        $this->amountPaid = $amountPaid;
    }

    public function addAmountPaid($amountPaid): void
    {
        $this->amountPaid += $amountPaid;
    }

    public function subAmountPaid($amountPaid): void
    {
        $this->amountPaid -= $amountPaid;
    }
    public function addPayment(Payment $payment): ?float
    {
        if (!$payment->getIsDone()) {
            $paymentAmount = $payment->getAmountOpen();
            // the invoice is an open invoice so use full payment amount
            if ($this->total == 0.0) {
                $this->amountPaid = $paymentAmount;
                return $paymentAmount;
            }
            $invoiceAmountOpen = $this->getAmountOpen();
            if ($invoiceAmountOpen >= $paymentAmount) {
                $this->addAmountPaid($paymentAmount);
                return $paymentAmount;
            }
            //$paymentOpenAmount = $paymentAmount - $invoiceAmountOpen;
            $this->amountPaid = $invoiceAmountOpen;
            return $invoiceAmountOpen;
        }
        return null;
    }

    
    /**
     * Returns the amountRemaining
     * The difference between amountDue and amountPaid
     *
     * @return float $amountRemaining = $amountDue - $amountPaid
     */
    public function getAmountRemaining()
    {
        return InvoiceUtility::subtractWithEpsilon($this->getAmountDue(), $this->amountPaid);
        //return $this->getAmountDue() - $this->amountPaid;
    }

    /**
     * Returns the amountDue
     * Final amount due at this time for this invoice.
     * applied credits (pre Payment) to this invoice are taken into account
     *
     * @return float $amountDue = $total - $prePaymentCreditAmount
     */
    public function getAmountDue()
    {
        return $this->total - $this->prePaymentCreditAmount;
    }

    /**
     * @todo remove
     * @deprecated
     */
    public function setBetragEz($total): void
    {
        $this->total = $total;
    }

    /**
     * Returns the letterDate
     *
     * @return DateTime
     */
    public function getLetterDate()
    {
        return $this->letterDate;
    }

    /**
     * Sets the letterDate
     */
    public function setLetterDate(DateTime $letterDate = null): void
    {
        $this->letterDate = $letterDate;
    }

    /**
     * Returns the status
     *
     * @return Invoicestatus
     */
    public function getStatus()
    {
        return $this->status;
    }
    public function getIsPaid()
    {
        if ($this->status === null) {
            return false;
        }
        // offener Betrag = 0 und status auf bezahlt
        return ($this->getAmountOpen() == 0 || $this->getTotal() == 0) && $this->status->getIsPaid() === true;
    }

    // for IgCrmTemplate
    public function getTemplateSubTag(): ?int
    {
        $status = $this->getStatus();
        return $status ? $status->getUid() : null;
    }

    // for IgCrmTemplate
    public function getTemplateMainTag(): ?int
    {
        return null;
    }
    
    /**
     * Sets the status
     */
    public function setStatus(Invoicestatus $status): void
    {
        $this->status = $status;
    }


    public function setStatusDefault(): void
    {
        $invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);
        $this->status = $invoicestatusRepository->findOneBy([
            'uid' => $this->initializeInvoicestatusUid,
        ]);
    }

    
    /**
     * Returns the art
     *
     * @return Invoicescheme
     */
    public function getArt()
    {
        return $this->art;
    }

    /**
     * Sets the art
     */
    public function setArt(Invoicescheme $art): void
    {
        $this->art = $art;
    }

    /**
     * Adds a InvoiceItem
     */
    public function addInvoiceItem(InvoiceItem $invoiceitem): void
    {
        $this->invoiceitem->attach($invoiceitem);
    }

    /**
     * Removes a InvoiceItem
     *
     * @param InvoiceItem $invoiceitemToRemove The InvoiceItem to be removed
     */
    public function removeInvoiceItem(InvoiceItem $invoiceitemToRemove): void
    {
        $this->invoiceitem->detach($invoiceitemToRemove);
    }

    /**
     * Returns the invoiceitem
     *
     * @return ObjectStorage<InvoiceItem> invoiceitem
     */
    public function getInvoiceItem()
    {
        return $this->invoiceitem;
    }
    /**
     * Returns the invoiceitems
     *
     * @return ObjectStorage<InvoiceItem> invoiceitem
     */
    public function getInvoiceItems()
    {
        return $this->invoiceitem;
    }
    public function getInvoiceItemsMaxSorting()
    {
        $sorting = 0;
        if (!empty($this->invoiceitem)) {
            foreach ($this->invoiceitem as $item) {
                $itemSorting = $item->getSorting();
                if ($itemSorting > $sorting) {
                    $sorting = $itemSorting;
                }
            }
        }
        return $sorting;
    }

    public function getInvoiceItemsTaxBehavior()
    {
        $taxBehavior = null;
        if (!empty($this->invoiceitem)) {
            foreach ($this->invoiceitem as $item) {
                $itemTaxBehavior = $item->getTaxBehavior();
                if ($itemTaxBehavior === TaxBehavior::INCLUSIVE->value) {
                    return $itemTaxBehavior;
                }
                if ($taxBehavior === null) {
                    $taxBehavior = $itemTaxBehavior;
                } elseif ($taxBehavior !== $itemTaxBehavior && $itemTaxBehavior === TaxBehavior::EXCLUSIVE->value) {
                    $taxBehavior = $itemTaxBehavior;
                }
            }
        }
        return $taxBehavior;
    }

    /**
     * Sets the invoiceitem
     *
     * @param ObjectStorage<InvoiceItem> $invoiceitem
     */
    public function setInvoiceItem(ObjectStorage $invoiceitem): void
    {
        $this->invoiceitem = $invoiceitem;
    }
    public function clearInvoiceItem(): void
    {
        $this->initStorageObjects();
    }
    /**
     * Returns the entryId
     */
    public function getRawEntryId(): int
    {
        return $this->entryId;
    }

    /**
     * Returns the entryId
     *
     * @return int entryId
     */
    public function getEntryId()
    {
        $entry = $this->getDebitor();
        return is_object($entry) ? $entry : $this->entryId;
    }
    public function getEntryUid()
    {
        $entry = $this->getEntryId();
        return is_object($entry) ? $entry->getUid() : $entry;
    }
    public function getDebitorId()
    {
        $entry = $this->getEntryId();
        return is_object($entry) ? $entry->getUid() : (int)$entry;
    }

    /**
     * Sets the entryId
     *
     * @param  string $entryId
     */
    public function setEntryId($entryId): void
    {
        $this->entryId = $entryId;
    }

    /**
     * Returns the tablenames
     *
     * @return string
     */
    public function getTablenames()
    {
        return $this->tablenames;
    }

    /**
     * Sets the tablenames
     *
     * @param  string $tablenames
     */
    public function setTablenames($tablenames): void
    {
        $this->tablenames = $tablenames;
    }

    /**
     * Sets the tablenames
     *
     * @param  string $tablenames
     * @param  string $entryId
     */
    public function setDebitor($tablenames, $entryId): void
    {
        $this->tablenames = $tablenames;
        $this->entryId = $entryId;
    }
    /**
     * Get Debitor Object
     */
    public function getDebitor()
    {
        if ($this->debitor === false) {
            $modelClassName = $this->getDebitorService()
->getModelClassName($this->tablenames);
            if ($modelClassName) {
                $repositoryClassName = $this->getDebitorService()
->getRepositoryClassName($this->tablenames);
                if ($repositoryClassName) {
                    $repository = GeneralUtility::makeInstance($repositoryClassName);
                    $this->debitor = $repository->findByUid($this->entryId);
                    //var_dump($this->debitor);                    die ($modelClassName . '=' . $repositoryClassName .' with ' . $this->entryId);
                } else {
                    $this->debitor = null;
                }
            } else {
                $this->debitor = null;
            }
        }
        return $this->debitor;
    }
    /**
     * Get Link Uid
     */
    /*
    public function getDebitorLink()
    {
        if ($this->debitorLink === null) {
            $this->getDebitor();
            $type = $this->debitor ? $this->debitor->getType() : '';
            $this->debitorLink = $this->getDebitorService()->getLink($this->tablenames, $this->entryId, $this->verbandId, $type);
        }
        return $this->debitorLink;
    }
    public function getDebitorSettings()
    {
        if ($this->debitorSettings===null) {
            $this->debitorSettings=$this->getDebitorService()->getConfig($this->tablenames);
            if ($this->debitorSettings===false) {
                die('No configuration for table="'.$this->tablenames .'"');
            }
        }
        return $this->debitorSettings;
    }
    */


    /**
     * Returns the verbandId
     *
     * @return int
     */
    public function getVerbandId()
    {
        return $this->verbandId;
    }
    public function getTenantId(): ?int
    {
        return $this->verbandId;
    }
    public function getVerband()
    {
        if ($this->verband === null) {
            $conf = $this->getDebitorService()
->getMandantConfig();
            if ($conf['repository'] ?? false) {
                if ($this->verbandId) {
                    $verbandRepository = GeneralUtility::makeInstance($conf['repository']);
                    $this->verband = $verbandRepository->findOneByUid($this->verbandId);
                } else {
                    die('repository: ' . $conf['repository'] . ' is set, but creditorId in invoice is not set');
                }
            } else {
                $creditor = GeneralUtility::makeInstance(Creditor::class);
                $creditor->setArray($conf['default']);
                $creditor->setUid(0); //bugfix for use of verband in links
                $this->verband = $creditor;
            }
            //var_dump($conf['repository'],$this->verband);die('a');
        }
        return $this->verband;
    }
    public function getTenant()
    {
        return $this->getVerband();
    }
    public function getCreditor()
    {
        return $this->getVerband();
    }
    public function getCreditorId()
    {
        return $this->getVerbandId();
    }

    /*
    public function getVerbandTemplate(string $objectType)
    {
        $this->getVerband();
        if ($this->verband === null) {
            die('no verband set for template');
        }
        $this->getDebitor();
        //return  $this->verband->getTemplateLetter('Invoice', $this->debitor, $this);
    }
    public function getVerbandTemplateMail(string $objectType)
    {
        $this->getVerband();
        $this->getDebitor();
        //return  $this->verband->getTemplateLetter('Mail', $this->debitor, null);
    }
     */
    /**
     * Sets the verbandId
     */
    public function setVerbandId($verbandId): void
    {
        $this->verbandId = $verbandId;
    }
    public function setTenantId(int $verbandId): void
    {
        $this->verbandId = $verbandId;
    }




    /**
     * Add Amount
     *
     * @param float $amountInkl
     * @param float $amountExkl
     * @param float $taxRate        Mwst
     * @param text  $title
     */
    public function addAmount($amountInkl, $amountExkl, $taxRate, $title, $description = null): void
    {
        $this->totalExkl = $amountExkl;
        $this->taxRate = $taxRate;
        $this->total = $amountInkl;
        $this->position = $title;
        $this->createDate = new DateTime();
        $invoiceitem = GeneralUtility::makeInstance(InvoiceItem::class);
        $invoiceitem->setTitle($title);
        if ($description !== null) {
            $invoiceitem->setDescription($description);
        }
        $invoiceitem->setQuantity(1);
        $invoiceitem->setTotalExkl($amountExkl);
        $invoiceitem->setTaxRate($taxRate);
        $invoiceitem->setTotalInkl($amountInkl);
        $this->addInvoiceItem($invoiceitem);
    }

    /**
     * Add Amount
     *
     * @param float $amountExkl
     * @param float $taxRate        Mwst
     * @param text  $title
     */
    public function addExkl($amountExkl, $taxRate, $title, $description = null): void
    {
        $amountInkl = round($amountExkl * (100 + $taxRate)) / 100;
        $this->addAmount($amountInkl, $amountExkl, $taxRate, $title, $description);
    }

    /**
     * Add Amount
     *
     * @param float $amountInkl
     * @param float $taxRate        Mwst
     * @param text  $title
     */
    public function addInkl($amountInkl, $taxRate, $title, $description = null): void
    {
        $amountExkl = round($amountInkl * 100 / (100 + $taxRate) * 100) / 100;
        $this->addAmount($amountInkl, $amountExkl, $taxRate, $title, $description);
        $this->setStatusDefault();
    }

    /**
     * Set Amount Inkl.
     *
     * @param float $amountInkl
     * @param float $taxRate        Mwst
     * @param text  $title
     */
    public function setInkl($amountInkl, $taxRate, $title, $description = null): void
    {
        $amountExkl = round($amountInkl * 100 / (100 + $taxRate) * 100) / 100;
        $this->totalExkl = $amountExkl;
        $this->taxRate = $taxRate;
        $this->total = $amountInkl;
        $this->position = $title;
        $this->createDate = new DateTime();
        $this->setStatusDefault();
    }


    /**
     * Returns the customerReference
     *
     * @return string
     */
    public function getCustomerReference()
    {
        return $this->customerReference;
    }

    /**
     * Sets the customerReference
     *
     * @param  string $customerReference
     */
    public function setCustomerReference($customerReference): void
    {
        $this->customerReference = $customerReference;
    }

    /**
     * Returns the description
     *
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Sets the description
     *
     * @param  string $description
     */
    public function setDescription($description): void
    {
        $this->description = $description;
    }



    /**
     * Returns the invoiceAddress
     *
     * @return string
     */
    public function getInvoiceAddress()
    {
        return $this->invoiceAddress;
    }

    /**
     * Sets the invoiceAddress
     *
     * @param  string $invoiceAddress
     */
    public function setInvoiceAddress($invoiceAddress): void
    {
        $this->invoiceAddress = $invoiceAddress;
    }


    /**
     * Returns the deliveryAddress
     *
     * @return string
     */
    public function getDeliveryAddress()
    {
        return $this->deliveryAddress;
    }

    /**
     * Sets the deliveryAddress
     *
     * @param  string $deliveryAddress
     */
    public function setDeliveryAddress($deliveryAddress): void
    {
        $this->deliveryAddress = $deliveryAddress;
    }


    public function getInvoicePaymentsCredit()
    {
        return $this->getInvoicePaymentsCreditPre();
    }
    
    public function getInvoicePaymentsCreditAll()
    {
        if ($this->invoicePaymentsCreditAll === null) {
            $invoicePaymentRepository = GeneralUtility::makeInstance(InvoicePaymentRepository::class);
            $this->invoicePaymentsCreditAll = $invoicePaymentRepository->findCreditByInvoice($this->uid);
            //var_dump($this->uid, $this->invoicePayments->count());exit(0);
        }
        return $this->invoicePaymentsCreditAll;
    }

    public function getInvoicePaymentsCreditPre()
    {
        if ($this->invoicePaymentsCreditPre === null) {
            $invoicePaymentRepository = GeneralUtility::makeInstance(InvoicePaymentRepository::class);
            $this->invoicePaymentsCreditPre = $invoicePaymentRepository->findCreditByInvoice($this->uid, false);
            //var_dump($this->uid, $this->invoicePayments->count());exit(0);
        }
        return $this->invoicePaymentsCreditPre;
    }
    /**
     * pre credit amount from items (use only for checks)
     */
    public function getInvoicePaymentsCreditPreAmount()
    {
        $amount = 0;
        $invoicePaymentsCredit = $this->getInvoicePaymentsCreditPre();
        foreach ($invoicePaymentsCredit as $invoicePayment) {
            $amount += $invoicePayment->getAmountApplied();
        }
        return $amount;
    }
    
    public function getInvoicePaymentsCreditPost()
    {
        if ($this->invoicePaymentsCreditPost === null) {
            $invoicePaymentRepository = GeneralUtility::makeInstance(InvoicePaymentRepository::class);
            $this->invoicePaymentsCreditPost = $invoicePaymentRepository->findCreditByInvoice($this->uid, true);
            //var_dump($this->uid, $this->invoicePayments->count());exit(0);
        }
        return $this->invoicePaymentsCreditPost;
    }

    /**
     * post credit amount from items (use only for checks)
     */
    public function getInvoicePaymentsCreditPostAmount()
    {
        $amount = 0;
        $invoicePaymentsCredit = $this->getInvoicePaymentsCreditPost();
        foreach ($invoicePaymentsCredit as $invoicePayment) {
            $amount += $invoicePayment->getAmountApplied();
        }
        return $amount;
    }
    
    public function getInvoicePaymentsPayment()
    {
        if ($this->invoicePaymentsPayment === null) {
            $invoicePaymentRepository = GeneralUtility::makeInstance(InvoicePaymentRepository::class);
            $this->invoicePaymentsPayment = $invoicePaymentRepository->findPaymentByInvoice($this->uid);
            //var_dump($this->uid, $this->invoicePayments->count());exit(0);
        }
        return $this->invoicePaymentsPayment;
    }

    public function getInvoicePayments()
    {
        if ($this->invoicePayments === null) {
            $invoicePaymentRepository = GeneralUtility::makeInstance(InvoicePaymentRepository::class);
            $this->invoicePayments = $invoicePaymentRepository->findBy([
                'invoice' => $this->uid,
            ]);
            //var_dump($this->uid, $this->invoicePayments->count());exit(0);
        }
        return $this->invoicePayments;
    }

    public function getInvoicePaymentsTotal()
    {
        if ($this->invoicePaymentTotalPayment === null) {
            $this->getInvoicePayments();
            $this->invoicePaymentTotalPayment = 0;
            $this->invoicePaymentTotalCredit = 0;
            foreach ($this->invoicePayments as $invoicePayment) {
                if ($invoicePayment->getCredit()) {
                    $this->invoicePaymentTotalCredit += $invoicePayment->getAmountApplied();
                } else {
                    $this->invoicePaymentTotalPayment += $invoicePayment->getAmountApplied();
                }
            }
            $this->invoicePaymentTotal = $this->invoicePaymentTotalCredit + $this->invoicePaymentTotalPayment;
        }
        return $this->invoicePaymentTotal;
    }
    
    public function getInvoicePaymentsTotalPayment()
    {
        $this->getInvoicePaymentsTotal();
        return $this->invoicePaymentTotalPayment;
    }
    public function getInvoicePaymentsTotalCredit()
    {
        $this->getInvoicePaymentsTotal();
        return $this->invoicePaymentTotalCredit;
    }
    public function getPayments()
    {
        if ($this->payments === null) {
            $paymentRepository = GeneralUtility::makeInstance(PaymentRepository::class);
            $this->payments = $paymentRepository->findBy([
                'invoice' => $this->uid,
            ]);
        }
        //foreach($this->payments as $z) {var_dump($z);};echo('R='.$this->uid);exit(0);
        return $this->payments;
    }

    public function getOpenPayments()
    {
        if ($this->openPayments === null) {
            $paymentRepository = GeneralUtility::makeInstance(PaymentRepository::class);
            $this->openPayments = $paymentRepository->findOpenByInvoice($this->uid);
            if (empty($this->openPayments) && $this->getCanPay()) {
                $this->openPayments = $paymentRepository->findOpenByDebitor($this->getDebitorId());
            }
        }
        //foreach($this->payments as $z) {var_dump($z);};echo('R='.$this->uid);exit(0);
        return $this->openPayments;
    }

    public function getPaymentsTotal()
    {
        if ($this->paymentTotal === null) {
            $this->getPayments();
            $this->paymentTotal = 0;
            foreach ($this->payments as $payment) {
                $this->paymentTotal += $payment->getBetrag();
            }
        }
        return $this->paymentTotal;
    }
    
    public function getCanDelete()
    {
        if ($this->status) {
            return $this->status->getCanDelete();
        }
        return true;
    }
    public function getCanEdit()
    {
        if ($this->status) {
            return $this->status->getCanEdit();
        }
        return true;
    }
    public function getCanPay()
    {
        if ($this->status) {
            return $this->status->getCanPay() || $this->status->getIsSuspended();
        }
        return true;
    }
    public function getCanAutoPay()
    {
        if ($this->status) {
            return $this->status->getCanPay();
        }
        return true;
    }

    public function getSettingOneInvoicePerMail()
    {
        // $betrieb = $this->getBetrieb();
        // return $betrieb->getEineInvoiceProMail();
        return true;
    }
    public function getSettingMailTo()
    {
        //$betrieb = $this->getBetrieb();
        $debitor = $this->getDebitor();
        return $debitor->getEmail();
    }
    public function getColspan()
    {
        if ($this->colspan === null) {
            $this->colspan = 1;
            if (!empty($this->invoiceitem)) {
                foreach ($this->invoiceitem as $invoiceitem) {
                    if ($invoiceitem instanceof InvoiceItemUnit) {
                        $this->colspan = 4;
                        break;
                    }
                }
            }
        }
        return $this->colspan;
    }

    public function getHistory()
    {
        return InvoiceDateRepository::getByInvoice($this->uid);
    }

    public function getFilename()
    {
        // @todo delete or add translation
        return 'invoice.pdf';
        //$language = $this->getDebitor()->getLanguage();
        //$languageCode = in_array($language, ['de', 'fr']) ? $language : 'de';
        //return 'Invoice-' . $this->getPeriodeFileName($languageCode) . '.pdf';
    }

    public function getMailLog(bool $raw = false)
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            'tx_igfibu_domain_model_log_mail'
        );
        $result = $queryBuilder->select('*')
                               ->from('tx_igfibu_domain_model_log_mail')
                               ->where('FIND_IN_SET(' . intval($this->uid) . ',lm_invoice_re_id)')
                               ->orderBy('create_date', 'DESC')
                               ->executeQuery();
        $rows = $result->fetchAllAssociative();
        if ($raw) {
            return $rows;
        }
        return HelperUtility::underscoredToLowerCamelCaseArrays($rows);
    }
    public function getMailLogTagAttributes()
    {
        $logs = $this->getMailLog(true);
        if (!empty($logs)) {
            $log = $logs[0];
            // Wurde verschickt per Mail
            $logTitle = 'An: ' . $log['lm_mail_to'] . "\nBetreff: " . $log['lm_subject'] . "\nDatum: " . $log['create_date'] . "\nAnzahl Rechnungen: " . $log['lm_invoice_count'];
            $logClass = 'pedos-done';
        } elseif ($this->mode == 0) {
            // muss noch verschickt werden per Mail
            $logTitle = $this->getDebitor()
->getVersandartEmail() . ' - nicht gesendet';
            $logClass = 'pedos-open';
        } else {
            // wurde nicht verschickt per Mail
            $logTitle = $this->getDebitor()
->getVersandartEmail();
            $logClass = 'pedos-unknown';
        }
        return 'class="' . $logClass . '" title="' . htmlspecialchars((string) $logTitle) . '"';
    }

    /**
     * Initializes all ObjectStorage properties
     * Do not modify this method!
     * It will be rewritten on each save in the extension builder
     * You may modify the constructor of this class instead
     */
    protected function initStorageObjects()
    {
        $this->invoiceitem = new ObjectStorage();
    }
}
