<?php

namespace Ig\IgFibu\Domain\Repository;

use Ig\IgFibu\Service\DebitorService;
use Ig\IgFibu\Utility\InvoiceUtility;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\QueryBuilder;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use TYPO3\CMS\Extbase\Persistence\Repository;

/**
 * The repository with base functions for ig_fibu
 */
class BaseRepository extends Repository
{
    protected static $tablename;
    protected static $queryBuilderClass;
    protected static $localDebitorAttribute;
    protected static $localClientAttribute;
    protected static $localTypeAttribute = 'type';
    protected static $llocalTablenamesAttribute = 'tablenames';

    protected $entryTablename = null;
    protected $entryPrimaryKey = null;
    protected $entryLanguageAttribute = null;
    protected $clientTablename = null; // Verband/Mandant table name
    protected $entryKeywordSearchAttributes = [];
    protected $clientPrimaryKey = 'uid'; // Verband/Mandant Primary key

    protected $defaultOrderings = [
        'uid' => QueryInterface::ORDER_DESCENDING,
        //'uid' => QueryInterface::ORDER_ASCENDING,
    ];

    
    /**
     * debitorService
     *
     * @var DebitorService
     */
    protected $debitorService = null;

    /**
     * invoiceUtility
     *
     * @var InvoiceUtility
     */
    protected $invoiceUtility = null;

    /**
     * dataMapper
     *
     * @var DataMapper
     */
    protected $dataMapper = null;


    public function injectDebitorService(DebitorService $debitorService): void
    {
        $this->debitorService = $debitorService;
        $this->connectionPool = GeneralUtility::makeInstance(ConnectionPool::class);
        $this->debitorService->setKeywordSearchAttributes($this->entryKeywordSearchAttributes);
        $this->debitorService->setLanguageAttribute($this->entryLanguageAttribute);
        if ($this->entryTablename === null) {
            $this->entryTablename = $this->debitorService->getEntryTable();
        }
        if ($this->entryPrimaryKey === null) {
            $this->entryPrimaryKey = $this->debitorService->getEntryPrimaryKey();
        }
    }

    public function injectInvoiceUtility(InvoiceUtility $invoiceUtility): void
    {
        $this->invoiceUtility = $invoiceUtility;
    }

    public function injectDataMapper(DataMapper $dataMapper): void
    {
        $this->dataMapper = $dataMapper;
    }

    
    /**
     * return a queryBuilder for credits
     *
     * @return CreditQueryBuilder
     */
    public function createQueryBuilder()
    {
        $conn = $this->connectionPool->getConnectionForTable(static::$tablename);
        $queryBuilder = GeneralUtility::makeInstance(static::$queryBuilderClass, $conn);
        $queryBuilder->injectDebitorService($this->debitorService);
        return $queryBuilder;
    }


    public function createQueryBuilderLeftJoinEntries()
    {
        $queryBuilder = $this->createQueryBuilder();
        $queryBuilder->from(static::$tablename);
        if ($this->entryTablename !== null) {
            $queryBuilder->setEntryTablename($this->entryTablename);
            $queryBuilder->setEntryPrimaryKey($this->entryPrimaryKey);
            $queryBuilder->leftJoin(
                static::$tablename,
                $this->entryTablename,
                $this->entryTablename,
                $queryBuilder->expr()
->eq(
    static::$localDebitorAttribute,
    $queryBuilder->quoteIdentifier($this->entryTablename . '.' . $this->entryPrimaryKey)
),
            );
        }
        if ($this->clientTablename !== null) {
            $queryBuilder->leftJoin(
                static::$tablename,
                $this->clientTablename,
                $this->clientTablename,
                $queryBuilder->expr()
->eq(
    static::$localClientAttribute,
    $queryBuilder->quoteIdentifier($this->clientTablename . '.' . $this->clientPrimaryKey)
),
            );
        }
        return $queryBuilder;
    }

    public static function andVerband(array $verbandsUids, array $search = [], $sqlPrefix = ' AND ')
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(static::$tablename);
        $constraints = [
            //static::$tablename . '.deleted=0'
        ];
        if (!empty($verbandsUids)) {
            $constraints[] = static::$tablename . '.' . static::$localClientAttribute . ' IN (' . implode(
                ',',
                $verbandsUids
            ) . ')';
        }
        if (static::$llocalTablenamesAttribute !== null && isset($search['tablenames']) && $search['tablenames'] !== '') {
            $constraints[] = static::$llocalTablenamesAttribute . '=' . $conn->quote($search['tablenames']);
        }
        if (static::$localTypeAttribute !== null && isset($search['type']) && $search['type'] !== '') {
            $constraints[] = static::$localTypeAttribute . '=' . $conn->quote($search['type']);
        }
        if (isset($search['year']) && $search['year'] !== '') {
            $constraints[] = 'YEAR(create_date)=' . (int)$search['year'];
        }
        if (isset($search['period']) && $search['period'] !== '') {
            $constraints[] = 'SUBSTR(tx_igfibu_domain_model_invoiceitem.period,1,4)=' . $conn->quote($search['period']);
        }
        return empty($constraints) ? '' : ' ' . $sqlPrefix . ' (' . implode(' AND ', $constraints) . ')';
    }



    /**
     * find by search
     */
    public function findBySearch(array $search = [])
    {
        if (static::$localDebitorAttribute === null) {
            $queryBuilder = $this->createQueryBuilder();
            $queryBuilder->from(static::$tablename);
        } else {
            $queryBuilder = $this->createQueryBuilderLeftJoinEntries();
        }

        $queryBuilder->select(static::$tablename . '.*');
        $queryBuilder->addSearch($search);
        $queryBuilder->addUserRestriction();
        if ($search['sorting'] ?? false) {
            $sortingAttributes = $this->debitorService->getSortingAttributesById(
                static::$tablename,
                $search['sorting']
            );
            if ($sortingAttributes === null) {
                $queryBuilder = $this->queryBuilderAddSorting($queryBuilder, $search);
            }
            if (!empty($sortingAttributes)) {
                foreach ($sortingAttributes as $attribute) {
                    $orderByArgs = GeneralUtility::trimExplode(' ', $attribute, true);
                    $sort = $orderByArgs[0];
                    $order = $orderByArgs[1] ?? 'ASC';
                    $queryBuilder->addOrderBy($sort, $order);
                    //echo($sort . ' ' . $order .'<br />');
                }
            }
            //var_dump($search['sorting'], $sortingAttributes);exit();
        } else {
            $this->queryBuilderAddDefaulSorting($queryBuilder);
        }

        return $queryBuilder;
    }

    public function findBySearchExecute(QueryBuilder $queryBuilder, int $limit = null, int $offset = null)
    {
        if ($limit !== null) {
            $queryBuilder->setMaxResults($limit);
        }
        if ($offset !== null) {
            $queryBuilder->setFirstResult($offset);
        }
        $results = $queryBuilder->executeQuery()
                 ->fetchAllAssociative();
        return $this->dataMapper->map(static::$modelClass, $results);
    }
    
    public function countBySearchExecute(QueryBuilder $queryBuilder, int $limit = null, int $offset = null)
    {
        return $queryBuilder->count(static::$tablename . '.uid')->executeQuery()->fetchOne();
    }
    public function uidBySearchExecute(QueryBuilder $queryBuilder, int $limit = null, int $offset = null)
    {
        $res = $queryBuilder->select(static::$tablename . '.uid')->executeQuery();
        $uids = [];
        while ($uid = $res->fetchOne()) {
            $uids[] = $uid;
        }
        //$res->fetchAllAssociative();
        return $uids;
    }

    

    public function sqlFetchAll(string $tablename, string $sql)
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable($tablename);
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        return $res->fetchAllAssociative();
    }

    /**
     * add default sorting for entry/debitor table
     */
    protected function queryBuilderAddDefaulSorting(QueryBuilder $queryBuilder): QueryBuilder
    {
        foreach ($this->defaultOrderings as $attribute => $order) {
            // add table name for joins
            $queryBuilder->addOrderBy(static::$tablename . '.' . $attribute, $order);
        }
        return $queryBuilder;
    }
    
    /**
     * add sorting for entry/debitor table
     */
    protected function queryBuilderAddUserSorting(QueryBuilder $queryBuilder, $sortingPropertyWithTable, $sortingOrder)
    {
        return false;
    }
    protected function queryBuilderAddSorting(QueryBuilder $queryBuilder, array $search): QueryBuilder
    {
        $sorting = GeneralUtility::trimExplode(',', $search['sorting'], true);
        if (!empty($sorting)) {
            foreach ($sorting as $sortingPart) {
                [$sortingPropertyWithTable, $sortingOrder] = GeneralUtility::trimExplode(' ', $sortingPart);
                if (!$this->queryBuilderAddUserSorting($queryBuilder, $sortingPropertyWithTable, $sortingOrder)) {
                    $sortingPropertyWithTableArray = GeneralUtility::trimExplode('.', $sortingPropertyWithTable);
                    if (count($sortingPropertyWithTableArray) == 1) {
                        $sortingPropertyTable = static::$tablename;
                        $sortingProperty = $sortingPropertyWithTableArray[0];
                    } else {
                        $sortingPropertyTable = $sortingPropertyWithTableArray[0] === 'contact' ? $this->entryTablename : static::$tablename;
                        $sortingProperty = $sortingPropertyWithTableArray[1];
                    }
                    $sortingAttribute = $sortingPropertyTable . '.' . GeneralUtility::camelCaseToLowerCaseUnderscored(
                        $sortingProperty
                    );
                    $queryBuilder->addOrderBy($sortingAttribute, $sortingOrder == 'desc' ? 'DESC' : 'ASC');
                }
            }
        }
        return $queryBuilder;
    }
}
