<?php

namespace Ig\IgFibu\Domain\Repository;

use DateTime;
use Ig\IgFibu\Domain\Model\InvoiceDate;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Repository;

/**
 * The repository for InvoiceDateRepository
 */
class InvoiceDateRepository extends Repository
{
    protected static $tablename = 'tx_igfibu_domain_model_invoicedate';
    protected static ?bool $joinFrontendUsers = null;

    public static function log(
        int $invoiceUid,
        int $statusUid,
        ?string $letterDate = null,
        ?int $action = null,
        ?int $deliveryMethod = null,
        ?int $logMail = null,
        ?string $pdfData = null
    ) {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            static::$tablename
        );
        $context = GeneralUtility::makeInstance(Context::class);
        $frontendUserUid = $context->getPropertyFromAspect('frontend.user', 'id');
        $values = [
            'invoice' => $invoiceUid,
            'status' => $statusUid,
            'invoice_date' => $letterDate,
            'action' => $action,
            'delivery_method' => $deliveryMethod,
            'frontend_user' => $frontendUserUid,
            'log_mail' => $logMail,
        ];
        $affectedRows = $queryBuilder->insert(static::$tablename)->values($values)->executeStatement();
        $connection = $queryBuilder->getConnection();
        $recordUid = (int)$connection->lastInsertId(static::$tablename);
        if ($pdfData) {
            $filePath = InvoiceDate::getPdfPathByUid($recordUid);
            GeneralUtility::mkdir_deep(dirname($filePath));

            // don't overwrite accidentally -> should never happen
            if (file_exists($filePath)) {
                throw new \RuntimeException('Invoice PDF already exists: ' . $filePath);
            }

            file_put_contents($filePath, $pdfData);
         }
         return $recordUid;
    }
    
    public static function getByInvoice(int $invoiceUid)
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            static::$tablename
        );
        if (static::$joinFrontendUsers === null) {
            
            $conn = $queryBuilder->getConnection();
            if (method_exists($conn, 'createSchemaManager')) {
                $schemaManager = $conn->createSchemaManager();// TYPO3 v13 (Doctrine DBAL 3.x)
            } else {
                $schemaManager = $conn->getSchemaManager();// TYPO3 v12 (Doctrine DBAL 2.x)
                
            }
            static::$joinFrontendUsers = $schemaManager->tablesExist(['fe_users']);
        }
        
        $queryBuilder
            ->select(static::$tablename . '.*', 'document_title', 's.title')//, 'username')
            ->from(static::$tablename)
            ->where('invoice = ' . (int) $invoiceUid)
            ->join(
                static::$tablename,
                'tx_igfibu_domain_model_invoicestatus',
                's',
                $queryBuilder->expr()
->eq('status', $queryBuilder->quoteIdentifier('s.uid'))
            );
        if (static::$joinFrontendUsers) {
            $queryBuilder
                ->leftJoin(
                    static::$tablename,
                    'fe_users',
                    'fe_users',
                    $queryBuilder->expr()
->eq('frontend_user', $queryBuilder->quoteIdentifier('fe_users.uid'))
                )
                ->addSelect('username');
        }
        $result = $queryBuilder->orderBy('create_date', 'DESC')
                               ->addOrderBy('invoice_date', 'DESC')
                               ->executeQuery();
        $rows = [];
        while ($row = $result->fetchAssociative()) {
            $pdfPath = InvoiceDate::getPdfPathByUid($row['uid']);
            $rows[] = [
                'uid' => $row['uid'],
                'invoice' => $row['invoice'],
                'status' => $row['status'],
                'title' => $row['title'],
                'documentTitle' => $row['document_title'],
                'invoiceDate' => $row['invoice_date'] ? new DateTime($row['invoice_date']) : null,
                'createDate' => new DateTime($row['create_date']),
                'action' => $row['action'],
                'deliveryMethod' => $row['delivery_method'],
                'logMail' => $row['log_mail'],
                'frontendUser' => [
                    'uid' => $row['frontend_user'],
                    'username' => $row['username'] ?? null,
                ],
                'pdfExists' => is_file($pdfPath),
            ];
        }
        return $rows;
    }
}
