<?php

namespace Ig\IgFibu\Service;

use DateTime;
use Ig\IgFibu\Domain\Model\Credit;
use Ig\IgFibu\Domain\Model\Invoice;
use Ig\IgFibu\Domain\Model\InvoicePayment;
use Ig\IgFibu\Domain\Model\Payment;
use Ig\IgFibu\Domain\Repository\CreditRepository;
use Ig\IgFibu\Domain\Repository\InvoicePaymentRepository;
use Ig\IgFibu\Domain\Repository\InvoiceRepository;
use Ig\IgFibu\Domain\Repository\InvoicestatusRepository;
use Ig\IgFibu\Domain\Repository\PaymentRepository;
use Internetgalerie\IgCrmTemplate\Domain\Model\AddressInterface;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;

class BookingService implements SingletonInterface
{
    protected bool $testOnly = false; // falls true wird nichts verbucht (gespeichert)

    protected float $floatPrecision = 0.001; // allowed difference in float compparsion
    
    protected string $useBookRule = '';
    
    protected int $bookedByInvoiceNumber = 0;
    protected int $countPartialPayment = 0;
    protected int $countOverpaidPayment = 0;
    protected int $countCouldBookedByOverpaid = 0;
    protected int $countBookedByDebitorIdAmount = 0;
    protected int $countCouldBookedByDebitorIdAmount = 0;
    protected array $invoiceUidsCouldBookedByDebitorIdAmount = [];
    protected array $paymentUidsCouldBookedByDebitorIdAmount = [];

    protected int $countCouldBookedBy = 0;
    protected array $invoiceUidsCouldBookedByOverpaid = [];
    protected array $paymentUidsCouldBookedByOverpaid = [];

    protected int $couldCreateCredit = 0;
    protected array $invoiceUidsCouldCreateCredit = [];
    protected array $paymentUidsCouldCreateCredit = [];

    protected int $countBookedByDebitorAddressAmount = 0;
    protected int $countCouldBookedByDebitorAddressAmount = 0;
    protected array $invoiceUidsCouldBookedByDebitorAddressAmount = [];
    protected array $paymentUidsCouldBookedByDebitorAddressAmount = [];

    protected int $countErrorBetrag = 0;
    protected int $countErrorNotFound = 0;
    protected int $countErrorStatusTotal = 0;
    protected array $countErrorStatus = [];

    protected array $invoicesDebitor = [];

    
    public function __construct(
        protected readonly CreditRepository $creditRepository,
        protected readonly InvoiceRepository $invoiceRepository,
        protected readonly InvoicePaymentRepository $invoicePaymentRepository,
        protected readonly InvoicestatusRepository $invoicestatusRepository,
        protected readonly PaymentRepository $paymentRepository
    ) { }
    
    public function setSettings($settings): void
    {
        $this->settings = $settings;
        $uid = (int)$this->settings['invoicesStatus']['paid'];
        $this->invoicestatusPaid = $this->invoicestatusRepository->findByUid($uid);
    }

    public function setUseBookRule(string $useBookRule): void
    {
        $this->useBookRule = $useBookRule;
    }

    public function getUseBookRule(): string
    {
        return $this->useBookRule;
    }

    // public function setUseBookByDebitor(bool $useBookByDebitor): void
    // {
    //     $this->useBookByDebitor = $useBookByDebitor;
    // }

    // public function getUseBookByDebitor(): bool
    // {
    //     return $this->useBookByDebitor;
    // }

    // public function setUseBookOverpaid(bool $useBookOverpaid): void
    // {
    //     $this->useBookOverpaid = $useBookOverpaid;
    // }

    // public function getUseBookOverpaid(): bool
    // {
    //     return $this->useBookOverpaid;
    // }
  
    // public function setUseCreateCredit(bool $useCreateCredit): void
    // {
    //     $this->useCreateCredit = $useCreateCredit;
    // }

    // public function getUseCreateCredit(): bool
    // {
    //     return $this->useCreateCredit;
    // }

    /**
     * @param Payment $payment the payment to book
     * @param Invoice $invoice if set the invoice is set in the payment
     * @return bool is booking successful
     */
    public function bookPaymentToInvoice(Payment $payment, Invoice $invoice = null): bool
    {
        if ($invoice !== null) {
            $payment->setInvoice($invoice);
        } else {
            $invoice = $payment->getInvoice();
        }
        $paymentAmountApplied = $invoice->addPayment($payment);
        // is invoice now paid
        if (abs($invoice->getAmountOpen()) < $this->floatPrecision || $invoice->getTotal() == 0) {
            $invoice->setStatus($this->invoicestatusPaid);
            $dateValuta = $payment->getDateValuta();
            $invoice->setPaidDate($dateValuta instanceof DateTime ? $dateValuta : new DateTime());
        }
        // payment was used
        if ($paymentAmountApplied > 0) {
            $invoicePayment = new InvoicePayment();
            $invoicePayment->setAmountApplied($paymentAmountApplied);
            $invoicePayment->setBookingDate(new DateTime());
            $invoicePayment->setInvoice($invoice);
            $invoicePayment->setPayment($payment);
            $invoicePayment->setTenantId($invoice->getVerbandId());
            $invoicePayment->setDebitorId($invoice->getDebitorId());
            $payment->addAmountApplied($paymentAmountApplied);
        } else {
            $invoicePayment = null;
        }
        if (abs($payment->getAmountOpen()) < $this->floatPrecision) {
            $payment->setIsOk(true);
            $payment->setIsDone(true);
        }
        
        if (!$this->testOnly) {
            // update invoice + Zahlung
            $this->invoiceRepository->update($invoice);
            $this->paymentRepository->update($payment);
            if ($invoicePayment !== null) {
                $this->invoicePaymentRepository->add($invoicePayment);
            }
        }
        // not booked: payment could not booked or booked amount = 0
        if ($paymentAmountApplied === null || $paymentAmountApplied === 0.0) {
            return false;
        }
        return true;
    }
    
    /**
     * book payments belonging to this invoice
     */
    public function bookInvoice(Invoice $invoice): bool
    {
        $dateValuta = null;
        foreach ($invoice->getPayments() as $payment) {
            if ($dateValuta === null || $dateValuta < $payment->getDateValuta()) {
                $dateValuta = $payment->getDateValuta();
            }
            $this->bookPaymentToInvoice($payment, $invoice);
        }
        return $invoice->getIsPaid();
    }

    /**
     * @return ?Invoice $invoice null or the invoice used to book the payment (open invoice found by tenant,debitor uid,amount)
     */
    public function bookByDebitorId(Payment $payment): ?Invoice
    {
        $debitor = $payment->getDebitor();
        
        if ($debitor instanceof AddressInterface) {
            // best match
            $verbandId = $payment->getVerbandId();
            $invoice = $this->invoiceRepository->findOpenByDebitorAndAmount(
                $debitor->getUid(),
                $payment->getBetrag(),
                $verbandId,
            );
            if ($invoice instanceof Invoice) {
                if ($this->useBookRule !== 'BookByDebitor') {
                    return $invoice;
                }
                if (!in_array($invoice->getUid(), $this->invoicesDebitor)) {
                    $this->invoicesDebitor[] = $invoice->getUid();
                    $this->bookPaymentToInvoice($payment, $invoice);
                    return $invoice;
                }
            }
        }
        return null;
    }


    /**
     * @return ?Invoice $invoice null or the invoice used to book the payment (open invoice found by tenant,debitor uid,amount)
     */
    public function bookByDebitorAddress(Payment $payment): ?Invoice
    {
        $name = $payment->getDebitorOrganisation();
        $debitorAddress = $payment->getDebitorAddress();

        // Check if the input contains " CH " and break the line if found
        if (str_contains($debitorAddress, ' CH ')) {
            $debitorAddress = str_replace(' CH ', "\nCH-", $debitorAddress);
        }
        $lines = explode("\n", $debitorAddress);
        $address = trim($lines[0] ?? '');
        $secondLine = trim($lines[1] ?? '');
        
        
        if (preg_match('/^([A-Z]{2,3})[- ]*(\d+)\s+(.+)$/', $secondLine, $matches)) {
            $country = $matches[1] ?? 'CH'; // Country code (e.g., CH, USA)
            $zip = trim($matches[2] ?? ''); // ZIP code (e.g., 3013)
            $city = trim($matches[3] ?? ''); // City name (e.g., Bern)
        } else {
            // If no country code match, try to match only the ZIP and city
            if (preg_match('/^(\d+)[- ]*(.+)$/', $secondLine, $matches)) {
                $country = 'CH'; // No country code
                $zip = trim($matches[1] ?? ''); // ZIP code (e.g., 3013)
                $city = trim($matches[2] ?? ''); // City name (e.g., Bern)
            } else {
                $zip = '';
                $city = '';
            }
        }
        if ($zip === '' || $city === '' || $name === '') {
            return null;
        }
        $debitorRepository = $payment->getDebitorService()
                           ->getDebitorRepository('');
        $tenantId = $payment->getTenantId();

        if (method_exists($debitorRepository, 'findByNameAndAddress')) {
            $debitor = $debitorRepository->findByNameAndAddress($name, $address, $zip, $city, $country, $tenantId);
            /*
            if ($payment->getUid() === 5) {
                echo($tenantId .'<br />' .$name .'<br />' . $address  .'<br />' .$country .'-' . $zip . ' ' . $city .'<br />');
                echo($secondLine .'<br />');
                var_dump($debitor);
                die('uid=128  found debitor' );
            }
            */
            
            if ($debitor instanceof AddressInterface) {
                // best match
                $verbandId = $payment->getVerbandId();
                $invoice = $this->invoiceRepository->findOpenByDebitorAndAmount(
                    $debitor->getUid(),
                    $payment->getBetrag(),
                    $verbandId,
                );
                if ($invoice instanceof Invoice) {
                    if ($this->useBookRule !== 'BookByDebitor') {
                        return $invoice;
                    }
                    if (!in_array($invoice->getUid(), $this->invoicesDebitor)) {
                        $this->invoicesDebitor[] = $invoice->getUid();
                        $this->bookPaymentToInvoice($payment, $invoice);
                        return $invoice;
                    }
                }
            }
        }
        return null;
    }


    /**
     * book credit with given invoice
     *
     * @param Credit $credit the credit to book
     * @param Invoice $invoice if set the invoice is set in the payment
     * @return bool is booking successful
     */
    public function bookCreditToInvoice(
        Credit $credit,
        Invoice $invoice = null,
        bool $invoiceMustBeGreater = true
    ): bool {
        $amount = $credit->getAmount();
        if (($invoiceMustBeGreater && $amount > $invoice->getAmountOpen()) || $credit->getIsDone()) {
            return false;
        }
                
        $invoice->addPrePaymentCreditAmount($amount);
        
        // is invoice now paid
        if (abs($invoice->getAmountOpen()) < $this->floatPrecision || $invoice->getTotal() == 0) {
            $invoice->setStatus($this->invoicestatusPaid);
            $dateValuta = new DateTime();
            $invoice->setPaidDate($dateValuta instanceof DateTime ? $dateValuta : new DateTime());
        }
        if ($amount !== 0) {
            $invoicePayment = new InvoicePayment();
            $invoicePayment->setAmountApplied($amount);
            $invoicePayment->setBookingDate(new DateTime());
            $invoicePayment->setCredit($credit);
            $invoicePayment->setInvoice($invoice);
            $invoicePayment->setCredit($credit);
            $invoicePayment->setTenantId($invoice->getVerbandId());
            $invoicePayment->setDebitorId($invoice->getDebitorId());
        }

        $credit->setIsDone(1);
        
        if (!$this->testOnly) {
            // update invoice + Zahlung
            $this->invoiceRepository->update($invoice);
            $this->creditRepository->update($credit);
            $this->invoicePaymentRepository->add($invoicePayment);
        }
        return true;
    }

    /**
     * book credit with invoices
     */
    public function bookCreditToDraftInvoices(Credit $credit, bool $doBooking = true): ?Invoice
    {
        $invoiceMustBeGreater = false;
        $invoice = $this->invoiceRepository->findDraftByCredit($credit, $invoiceMustBeGreater);
        if ($invoice) {
            if ($doBooking) {
                $ok = $this->bookCreditToInvoice($credit, $invoice, $invoiceMustBeGreater);
                return $ok ? $invoice : null;
            }
            return $invoice;
        }
        return null;
    }

    /**
     * create credit with negative open amount of invoice or return null if open amount is 0
     */
    public function createCreditWithOpenAmountOfInvoice(Invoice $invoice, ?string $title = null): ?Credit
    {
        $amount = $invoice->getAmountOpen();
        if ($amount === 0.0) {
            return null;
        }
        $credit = GeneralUtility::makeInstance(Credit::class);
        $credit->setAmount(-$amount);
        $credit->setTenantId($invoice->getVerbandId());
        $credit->setDebitorId($invoice->getDebitorId());
        if ($title !== null) {
            $credit->setTitle($title);
        }
        $this->creditRepository->add($credit);
        $this->bookCreditForInvoice($invoice, $credit);
        $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
        $persistenceManager->persistAll();
        return $credit;
    }
    

    /**
     * create credit with open amount of payment or return null if open amount is 0
     */
    public function createCreditWithOpenAmountOfPayment(Payment $payment, ?string $title = null): ?Credit
    {
        $amount = $payment->getAmountOpen();
        if ($amount === 0.0) {
            return null;
        }
        $debitorId = $payment->getDebitorId();
        if ($debitorId === null) {
            return null;
        }
        $credit = GeneralUtility::makeInstance(Credit::class);
        $credit->setAmount($amount);
        $credit->setTenantId($payment->getVerbandId());
        $credit->setDebitorId($debitorId);
        if ($title !== null) {
            $credit->setTitle($title);
        }
        $this->creditRepository->add($credit);
        $this->bookCreditForPayment($payment, $credit);
        $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
        $persistenceManager->persistAll();
        return $credit;
    }
    

    
    public function book(Payment $payment): void
    {
        $invoice = $payment->getInvoice();
        if ($invoice) {
            $status = $invoice->getStatus();
            if ($status->getCanPay()) {
                if ($invoice->getAmountOpen() + $this->floatPrecision >= $payment->getBetrag() || $invoice->getTotal() == 0) {
                    if ($this->bookPaymentToInvoice($payment, $invoice)) {
                        if ($payment->getInvoice()->getIsPaid()) {
                            $this->bookedByInvoiceNumber++;
                        } else {
                            $this->countPartialPayment++;
                        }
                    }
                } else {
                    if ($this->useBookRule === 'BookOverpaid') {
                        if ($this->bookPaymentToInvoice($payment, $invoice)) {
                            if ($payment->getInvoice()->getIsPaid()) {
                                $this->bookedByInvoiceNumber++;
                            } else {
                                $this->countOverpaidPayment++;
                            }
                        }
                    } else {
                        $this->countCouldBookedByOverpaid++;
                        $this->invoiceUidsCouldBookedByOverpaid[] = $invoice->getUid();
                        $this->paymentUidsCouldBookedByOverpaid[] = $payment->getUid();
                        //$this->countErrorBetrag++;
                    }
                }
            } else {
                // @todo $status->getFinished() or $status->getIsPaid()
                $debitorInvoice = $status->getIsPaid() ? $this->bookByDebitorId($payment) : null;
                if ($debitorInvoice !== null) {
                    if ($this->useBookRule !== 'BookByDebitor') {
                        $this->countCouldBookedByDebitorIdAmount++;
                        $this->invoiceUidsCouldBookedByDebitorIdAmount[] = $debitorInvoice->getUid();
                        $this->paymentUidsCouldBookedByDebitorIdAmount[] = $payment->getUid();
                    } elseif ($debitorInvoice->getIsPaid()) {
                        $this->countBookedByDebitorIdAmount++;
                    } else {
                        $this->countPartialPayment++;
                    }
                } else {
                    if ($invoice->getIsPaid()) {
                        if ($this->useBookRule === 'CreateCredit') {
                            $this->createCreditWithOpenAmountOfPayment($payment, 'Überzahlung Rechnung ' . $invoice->getUid());
                        } else {
                            $this->couldCreateCredit++;
                            $this->invoiceUidsCouldCreateCredit[] = $invoice->getUid();
                            $this->paymentUidsCouldCreateCredit[] = $payment->getUid();
                        }
                    }
                    $statusUid = $status->getUid();
                    if (!is_array($this->countErrorStatus[$statusUid] ?? false)) {
                        $this->countErrorStatus[$statusUid] = [
                            'status' => $status,
                            'count' => 0,
                        ];
                    }
                    $this->countErrorStatus[$statusUid]['count']++;
                    
                    
                    $this->countErrorStatusTotal++;
                }
            }
        } else {
            // no invoice found
            $debitorInvoice = $this->bookByDebitorId($payment);
            if ($debitorInvoice !== null) {
                if ($this->useBookRule !== 'BookByDebitor') {
                    $this->countCouldBookedByDebitorIdAmount++;
                    $this->invoiceUidsCouldBookedByDebitorIdAmount[] = $debitorInvoice->getUid();
                    $this->paymentUidsCouldBookedByDebitorIdAmount[] = $payment->getUid();
                } else {
                    $this->countBookedByDebitorIdAmount++;
                }
            } else {
                $debitorInvoice = $this->bookByDebitorAddress($payment);
                if ($debitorInvoice !== null) {
                    if ($this->useBookRule !== 'BookByDebitor') {
                        $this->countCouldBookedByDebitorAddressAmount++;
                        $this->invoiceUidsCouldBookedByDebitorAddressAmount[] = $debitorInvoice->getUid();
                        $this->paymentUidsCouldBookedByDebitorAddressAmount[] = $payment->getUid();
                    } else {
                        $this->countBookedByDebitorAddressAmount++;
                    }
                } else {
                    $this->countErrorNotFound++;
                }
            }
        }
    }

    public function unbook(InvoicePayment $invoicePayment): bool
    {
        $unbookDone = false;
        
        $invoice = $invoicePayment->getInvoice();
        $credit = $invoicePayment->getCredit();
        $payment = $invoicePayment->getPayment();
        
        $amount = $invoicePayment->getAmountApplied();
        if ($invoice !== null) {
            if ($credit) {
                if ($invoicePayment->getPostPayment()) {
                    $invoice->subPostPaymentCreditAmount($amount);
                } else {
                    $invoice->subPrePaymentCreditAmount($amount);
                }
            } else {
                $invoice->subAmountPaid($amount);
            }
            if ($invoice->getStatus() == $this->invoicestatusPaid) {
                $invoicestatusDraft = $this->invoicestatusRepository->findByUid(1);
                $invoice->setStatus($invoicestatusDraft);
            }
            if (!$this->testOnly) {
                $this->invoiceRepository->update($invoice);
                if ($credit) {
                    if ($credit->getIsDone()) {
                        $credit->setIsDone(0);
                        $this->creditRepository->update($credit);
                    } elseif (count($credit->getInvoicePayments()) == 1) {
                        // only this invoicePayment is on this credit
                        $this->creditRepository->remove($credit);
                    }
                }
                $unbookDone = true;
            }
        }
        
        if ($payment) {
            $payment->subAmountApplied($amount);
            $payment->setIsDone(0);
            $payment->setIsOk(0);
            if (!$this->testOnly) {
                $this->paymentRepository->update($payment);
                $unbookDone = true;
                if ($credit) {
                    if ($invoicePayment->getUndoDeleteCredit()) {
                        $this->creditRepository->remove($credit);
                    }
                }
            }
        }
        if (!$this->testOnly) {
            $this->invoicePaymentRepository->remove($invoicePayment);
        }

        return $unbookDone;
    }

    public function getStatistic(): array
    {
        return [
            'bookedByInvoiceNumber' => $this->bookedByInvoiceNumber,
            'bookedByDebitorIdAmount' => $this->countBookedByDebitorIdAmount,
            'couldBookedByDebitorIdAmount' => $this->countCouldBookedByDebitorIdAmount,
            'invoiceUidsCouldBookedByDebitorIdAmount' => implode(',', $this->invoiceUidsCouldBookedByDebitorIdAmount),
            'paymentUidsCouldBookedByDebitorIdAmount' => implode(',', $this->paymentUidsCouldBookedByDebitorIdAmount),
            'bookedByDebitorAddressAmount' => $this->countBookedByDebitorAddressAmount,
            'couldBookedByDebitorAddressAmount' => $this->countCouldBookedByDebitorAddressAmount,
            'invoiceUidsCouldBookedByDebitorAddressAmount' => implode(
                ',',
                $this->invoiceUidsCouldBookedByDebitorAddressAmount
            ),
            'paymentUidsCouldBookedByDebitorAddressAmount' => implode(
                ',',
                $this->paymentUidsCouldBookedByDebitorAddressAmount
            ),
            'couldBookedByOverpaid' => $this->countCouldBookedByOverpaid,
            'invoiceUidsCouldBookedByOverpaid' => implode(',', $this->invoiceUidsCouldBookedByOverpaid),
            'paymentUidsCouldBookedByOverpaid' => implode(',', $this->paymentUidsCouldBookedByOverpaid),
            'couldCreateCredit' => $this->couldCreateCredit,
            'invoiceUidsCouldCreateCredit' => implode(',', $this->invoiceUidsCouldCreateCredit),
            'paymentUidsCouldCreateCredit' => implode(',', $this->paymentUidsCouldCreateCredit),
            'partialPayment' => $this->countPartialPayment,
            'overpaidPayment' => $this->countOverpaidPayment,
            'errorBetrag' => $this->countErrorBetrag,
            'errorStatusTotal' => $this->countErrorStatusTotal,
            'errorStatus' => $this->countErrorStatus,
            'errorNotFound' => $this->countErrorNotFound,
        ];
    }

    /**
     * connect credit with invoice
     *
     * @param Invoice $invoice the invoice to book
     * @param Credit $credit credit to book with invoice
     */
    protected function bookCreditForInvoice(Invoice $invoice, Credit $credit): bool
    {
        $invoiceIsdone = false;
        $invoicePayment = new InvoicePayment();
        $invoicePayment->setAmountApplied(-$credit->getAmount());
        $invoicePayment->setBookingDate(new DateTime());
        $invoicePayment->setCredit($credit);
        $invoicePayment->setInvoice($invoice);
        $invoicePayment->setTenantId($credit->getTenantId());
        $invoicePayment->setDebitorId($credit->getDebitorId());
        $invoicePayment->setPostPayment(true);
        $invoice->addPostPaymentCreditAmount(-$credit->getAmount());

        if (abs($invoice->getAmountOpen()) < $this->floatPrecision) {
            $invoice->setStatus($this->invoicestatusPaid);
            $dateValuta = new DateTime();
            $invoice->setPaidDate($dateValuta);
            $invoiceIsdone = true;
        }
        
        if (!$this->testOnly) {
            // update invoice + Zahlung
            $this->invoiceRepository->update($invoice);
            $this->invoicePaymentRepository->add($invoicePayment);
        }
        return $invoiceIsdone;
    }


    /**
     * connect credit with payment
     *
     * @param Payment $payment the payment to book
     * @param Credit $credit credit to book with payment
     */
    protected function bookCreditForPayment(Payment $payment, Credit $credit): bool
    {
        $paymentIsdone = false;
        $invoicePayment = new InvoicePayment();
        $invoicePayment->setAmountApplied($credit->getAmount());
        $invoicePayment->setBookingDate(new DateTime());
        $invoicePayment->setCredit($credit);
        $invoicePayment->setPayment($payment);
        $invoicePayment->setTenantId($credit->getTenantId());
        $invoicePayment->setDebitorId($credit->getDebitorId());
        $payment->addAmountApplied($credit->getAmount());

        if (abs($payment->getAmountOpen()) < $this->floatPrecision) {
            $payment->setIsOk(true);
            $payment->setIsDone(true);
            $paymentIsdone = true;
        }
        
        if (!$this->testOnly) {
            // update invoice + Zahlung
            $this->paymentRepository->update($payment);
            $this->invoicePaymentRepository->add($invoicePayment);
        }
        return $paymentIsdone;
    }
}
