<?php

namespace Ig\IgFibu\Service\Period;

use DateInterval;
use DateTime;
use IntlDateFormatter;
use Locale;

abstract class AbstractPeriodService
{
    protected DateInterval $interval;
    protected IntlDateFormatter $formatter;
    protected int $durationInMonth = 12;
    protected string $valuePattern = 'yyyy';
    protected string $labelPattern = 'yyyy';
    protected string $filenamePartPattern = 'yyyy';

    public function __construct()
    {
        //\Locale::setDefault('de_DE.utf-8');
        $lcTime = setlocale(LC_TIME, 0);
        if ($lcTime === 'C' || $lcTime === 'POSIX') {
            $lcTime = 'de_CH';
        }
        Locale::setDefault($lcTime);
        $this->formatter = IntlDateFormatter::create(null, IntlDateFormatter::FULL, IntlDateFormatter::NONE);
        $this->interval = DateInterval::createFromDateString($this->durationInMonth . ' month');
    }

    /**
     * gets datetime of first day of period
     */
    public function createByValue(string $value): ?DateTime
    {
        if (strlen($value) === 7) {
            $datetime = new DateTime($value . '-01');
        } elseif (strlen($value) === 6) {
            $year = substr($value, 0, 4);
            $month = substr($value, 4, 2);
            $datetime = new DateTime($year . '-' . $month . '-01');
        } elseif (strlen($value) === 4) {
            $datetime = new DateTime($value . '-01-01');
        } else {
            if ($value === '') {
                return null;
            }
            die('Error: unknown period value "' . $value . '"');
        }
        return $this->toStartDate($datetime);
        //$this->value = $value;
    }
    /*
    public static function create(DateTime $datetime)
    {
        $this->datetime = $datetime;
        //$this->formatter->setPattern($this->valuePattern);
        //$this->value = $formatter->format($datetime);
    }
    */
    public function setDurationInMonth(int $durationInMonth): void
    {
        $this->durationInMonth = $durationInMonth;
    }

    /**
     * sets given datetime to fist day of period
     */
    public function toStartDate(DateTime $datetime): DateTime
    {
        $year = $datetime->format('Y');
        $month = $datetime->format('n');
        $durationCount = (int)(($month - 1) / $this->durationInMonth);
        $month = $durationCount * $this->durationInMonth + 1;
        $datetime->setDate($year, $month, 1)
->setTime(12, 0, 0);
        return $datetime;
    }

    /**
     * get value to store in DB of given period datetime (first day of period)
     */
    public function getValue(DateTime $datetime): string
    {
        $this->formatter->setPattern($this->valuePattern);
        return $this->formatter->format($datetime);
    }

    /**
     * get label of given period datetime (first day of period)
     */
    public function getLabel(?DateTime $datetime): string
    {
        if ($datetime === null) {
            return '';
        }
        $this->formatter->setPattern($this->labelPattern);
        return $this->formatter->format($datetime);
    }

    /**
     * get filename part of given period datetime (first day of period)
     */
    public function getFilenamePart(DateTime $datetime): string
    {
        $this->formatter->setPattern($this->filenamePartPattern);
        return $this->formatter->format($datetime);
    }

    /**
     * get all periodDatetimes in the given range (starting with fromDate)
     */
    public function getRangeByDate(DateTime $fromDate, ?DateTime $toDate = null): array
    {
        $max = 1000;
        $periods = [];
        if ($fromDate < $toDate) {
            $datetime = clone $fromDate;
            while ($datetime < $toDate && $max > 0) {
                $periods[] = clone $datetime;
                $datetime->add($this->interval);
                $max--;
            }
        } else {
            $datetime = clone $fromDate;
            while ($datetime > $toDate && $max > 0) {
                $periods[] = clone $datetime;
                $datetime->sub($this->interval);
                $max--;
            }
        }
        return $periods;
    }

    /**
     * get all periodDatetimes containing $fromDate and $countLower older periods and $countHigher news periods
     * sorting is from old to new or if $reverseSortingin is true from new to old
     */
    public function getRangeByCount(
        DateTime $fromDate,
        int $countLower,
        int $countHigher = 0,
        bool $reverseSorting = false
    ): array {
        $periods = [];
        $fromDate = $this->toStartDate($fromDate);
        if ($countLower > 0) {
            $datetime = clone $fromDate;
            $reversePeriods = [];
            for ($i = 0; $i < $countLower; $i++) {
                $datetime->sub($this->interval);
                $reversePeriods[] = clone $datetime;
            }
            $periods = array_reverse($reversePeriods);
        }
        $periods[] = clone $fromDate;

        if ($countHigher > 0) {
            $datetime = $fromDate;
            for ($i = 0; $i < $countHigher; $i++) {
                $datetime->add($this->interval);
                $periods[] = clone $datetime;
            }
        }
        if ($reverseSorting) {
            arsort($periods);
        }
        return $periods;
    }

    /**
     * convert array of periodDatetimes to array [ value => label] used for pulldowns
     */
    public function getValueLabel(array $periodDates): array
    {
        $valueLabelArray = [];
        foreach ($periodDates as $periodDate) {
            $key = $this->getValue($periodDate);
            $label = $this->getLabel($periodDate);
            $valueLabelArray[$key] = $label;
        }
        return $valueLabelArray;
    }
}
