<?php

namespace Ig\IgFibu\Utility;

use DateTime;
use Ig\IgFibu\Domain\Model\Payment;
use Ig\IgFibu\Domain\Repository\InvoiceRepository;
use SimpleXMLElement;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;

class XmlImportUtility implements SingletonInterface
{
    public const XML_FORMAT_CAMT_054 = 'camt.054'; // XML Format für Zahlungeingaenge - ISO 20022 CAMT.054
    public const XML_FORMAT_CAMT_053 = 'camt.053'; // XML Format für Konto Auszug - ISO 20022 CAMT.053 (not used / Duetschler)

    public const ALLOW_CAMT_053 = true; // allow import of CAMT.053 in payments
    protected const ALLOW_UNKNOWN_REFERENCES = true; // can unknown references imported

    public function getInfoFromXml($xmlStr)
    {
        $ntfctn = [];
        $xmlPayments = new SimpleXMLElement($xmlStr);
        if (isset($xmlPayments->BkToCstmrDbtCdtNtfctn->Ntfctn)) {
            $xmlRoot = $xmlPayments->BkToCstmrDbtCdtNtfctn->Ntfctn;
            $ntfctn['id'] = strval($xmlRoot->Id ?? '');
            $ntfctn['accountIban'] = strval($xmlRoot->Acct->Id->IBAN ?? '');
        }
        if (isset($xmlPayments->BkToCstmrDbtCdtNtfctn->GrpHdr)) {
            // XML_FORMAT_CAMT_054
            $msgId = strval($xmlRoot->GrpHdr->MsgId ?? '');
        } elseif (isset($xmlPayments->BkToCstmrStmt->GrpHdr)) {
            // XML_FORMAT_CAMT_053
            $xmlRoot = $xmlPayments->BkToCstmrStmt;
            $msgId = strval($xmlRoot->GrpHdr->MsgId ?? '');
            $ntfctn = [];

        }
        return [
            'msgId' => $msgId,
            'ntfctn' => $ntfctn,
        ];
    }

    // addFromXml
    public function getPaymentsFromXml($xmlStr, ?int $tenantId = null)
    {
        $this->invoiceUtility = GeneralUtility::makeInstance(InvoiceUtility::class);
        $this->invoiceRepository = GeneralUtility::makeInstance(InvoiceRepository::class);
        $payments = [];
        $xmlPayments = new SimpleXMLElement($xmlStr);
        $totalBetrag = 0;
        $totalAnzahl = 0;
        $totalChargesTotal = 0;
        $importDate = strftime('%Y-%m-%d', time());
        if (isset($xmlPayments->BkToCstmrDbtCdtNtfctn->Ntfctn)) {
            $xmlRoot = $xmlPayments->BkToCstmrDbtCdtNtfctn->Ntfctn;
            $format = self::XML_FORMAT_CAMT_054;
        } elseif (isset($xmlPayments->BkToCstmrStmt->Stmt)) {
            $xmlRoot = $xmlPayments->BkToCstmrStmt->Stmt;
            $format = self::XML_FORMAT_CAMT_053;
            if (!self::ALLOW_CAMT_053) {
                die('Format CAMT.053 ist ein Konto Auszug und nicht unterstützt');
            }
        } else {
            die('unknown XML format');
        }
        $DbitTotalAnzahl = 0;
        $DbitTotalBetrag = 0;
        $CrdtTotalAnzahl = 0;
        $CrdtTotalBetrag = 0;
        // $xmlPayments->BkToCstmrDbtCdtNtfctn->Ntfctn->Ntry
        
        foreach ($xmlRoot->Ntry as $xmlPaymentGruppe) {
            $subTotalBetrag = 0;
            $subTotalAnzahl = 0;
            // https://www.lukb.ch/documents/38421/354548/LUKB-CAMT.053-Handbuch-camt.pdf/efc38f8f-f31a-ce48-0c32-952478a8eebd?t=1593521338533
            if ($xmlPaymentGruppe->CdtDbtInd == 'DBIT') {
                $DbitTotalAnzahl++;
                foreach ($xmlPaymentGruppe->NtryDtls->TxDtls as $xmlPayment) {
                    $DbitTotalBetrag += floatval($xmlPayment->Amt);
                }
                continue;
            }
            if ($xmlPaymentGruppe->CdtDbtInd = 'CRDT') {
                $CrdtTotalAnzahl++;
            } else {
                die('error unknown CdtDbtInd=' . $xmlPaymentGruppe->CdtDbtInd);
            }
            $bookDate = $xmlPaymentGruppe->BookgDt->Dt;
            $valutaDate = $xmlPaymentGruppe->ValDt->Dt;
            if (isset($xmlPaymentGruppe->NtryDtls->TxDtls)) {
                // Process each transaction detail
                foreach ($xmlPaymentGruppe->NtryDtls->TxDtls as $xmlPayment) {
                    $amount = floatval($xmlPayment->Amt);
                    $totalBetrag += $amount;
                    $totalAnzahl++;
                    $subTotalBetrag += $amount;
                    $subTotalAnzahl++;
                    $chargesTotal = floatval($xmlPayment->Chrgs->TtlChrgsAndTaxAmt);
                    $totalChargesTotal += $chargesTotal;
                    $textInfo = strval($xmlPayment->AddtlTxInf);

                    $debitorOrganisation = strval($xmlPayment->RltdPties->Dbtr->Nm);
                    $PstlAdr = $xmlPayment->RltdPties->Dbtr->PstlAdr;
                    $debitorAddress = '';
                    // Existiert dies ? (Post?)
                    if ($PstlAdr !== null) {
                        if (isset($PstlAdr->AdrLine)) {
                            foreach ($PstlAdr->AdrLine as $adrLine) {
                                $debitorAddress .= strval($adrLine) . "\n";
                            }
                        }
                        if ($PstlAdr->StrtNm || $PstlAdr->BldgNb) {
                            $debitorAddress .= strval($PstlAdr->StrtNm) . ' ' . strval($PstlAdr->BldgNb) . "\n";
                        }
                        if ($PstlAdr->Ctry) {
                            $debitorAddress .= strval($PstlAdr->Ctry);
                        }
                        if ($PstlAdr->PstCd || $PstlAdr->TwnNm) {
                            if ($PstlAdr->Ctry) {
                                $debitorAddress .= '-';
                            }
                            $debitorAddress .= strval($PstlAdr->PstCd) . ' ' . strval($PstlAdr->TwnNm);
                        }
                    }

                    if (isset($xmlPayment->RltdPties->DbtrAcct->Id->IBAN)) {
                        $iban = strval($xmlPayment->RltdPties->DbtrAcct->Id->IBAN);
                    } else {
                        $iban = '';
                    }
                    $reference = trim(strval($xmlPayment->RmtInf->Strd->CdtrRefInf->Ref));
                    if ($reference === '' && isset($xmlPayment->RmtInf->Ustrd)) {
                        // Unstructured Remittance Information
                        $ustrd = trim(strval($xmlPayment->RmtInf->Ustrd));
                        $reference = $this->getReferenceFromText($ustrd);
                        if ($reference === '' && $ustrd !== '') {
                            $textInfo .= ' (' . $ustrd . ')';
                        }
                    }
                    //$acceptDate = substr($xmlPayment->RltdDts->AccptncDtTm, 0, 10);
                    $payments[] = $this->createPayment(
                        $tenantId,
                        $amount,
                        $reference,
                        $textInfo,
                        $debitorOrganisation,
                        $debitorAddress,
                        $bookDate,
                        $valutaDate,
                        $importDate,
                        $chargesTotal,
                        $iban
                    );
                }
            } else {
                // we do not have transaction details, so we use batch details (infos are limited)
                if ($format == self::XML_FORMAT_CAMT_053) {
                    $amount = 0;
                    // can this have several entries
                    foreach ($xmlPaymentGruppe->NtryDtls->Btch as $xmlPayment) {
                        $amount += floatval($xmlPayment->TtlAmt);
                    }
                    $textInfo = trim(strval($xmlPaymentGruppe->AddtlNtryInf));
                    $reference = trim(strval($xmlPayment->RmtInf->Strd->CdtrRefInf->Ref ?? ''));

                    if (!$reference) {
                        $reference = $this->getReferenceFromText($textInfo);
                    }
                    // missing values
                    $debitorOrganisation = null;
                    $debitorAddress = null;
                    $chargesTotal = 0;
                    $iban = null;
                    $payments[] = $this->createPayment(
                        $tenantId,
                        $amount,
                        $reference,
                        $textInfo,
                        $debitorOrganisation,
                        $debitorAddress,
                        $bookDate,
                        $valutaDate,
                        $importDate,
                        $chargesTotal,
                        $iban
                    );
                    //var_dump($payment, $reference, $referenceFormatterResult);exit(0);
                }
            }
        }
        return $payments;
    }

    public function createPayment(
        $tenantId,
        $amount,
        $reference,
        $textInfo,
        $debitorOrganisation,
        $debitorAddress,
        $bookDate,
        $valutaDate,
        $importDate,
        $chargesTotal,
        $iban
    ) {
        $referenceFormatterResult = $this->invoiceUtility->resolveReference($reference, (int)$tenantId);
        if ($referenceFormatterResult->isError()) {
            if (self::ALLOW_UNKNOWN_REFERENCES) {
                $referenceEntry = null;
                $debitorId = null;
                $referenceInvoice = null;
                $referenceFormat = null;
                $referenceTablenames = null;
            } else {
                die('error unknown reference:' . $referenceFormatterResult->getErrorMessage());
            }
        } else {
            $referenceEntry = $referenceFormatterResult->getReferenceDebitor();
            $debitorId = $referenceFormatterResult->getDebitorId();
            $referenceInvoice = $referenceFormatterResult->getInvoiceId();
            $referenceFormat = $referenceFormatterResult->getFormat();
            $referenceTablenames = $referenceFormatterResult->getTablenames();
        }
                
        $payment = GeneralUtility::makeInstance(Payment::class);
        $payment->setAmount($amount);
        $payment->setTextInfo($textInfo);
        $payment->setDebitorOrganisation($debitorOrganisation);
        $payment->setDebitorAddress($debitorAddress);
        if ($bookDate) {
            $payment->setDatePay(new DateTime($bookDate));//$acceptDate
        }
        if ($valutaDate) {
            $payment->setDateValuta(new DateTime($valutaDate));
        }
        if ($importDate) {
            $payment->setDateImport(new DateTime($importDate));
        }
        $payment->setReference($reference);
        //$payment->setReferenceBank($referenceBank);
        $payment->setReferenceFormat($referenceFormat);
        //$payment->setTablenamesByFormat($referenceFormat);
        $payment->setTablenames($referenceTablenames); // deprecated ?;

        $payment->setReferenceEntry($referenceEntry); // deprecated ?
        $payment->setEntryId((int)$referenceEntry); // deprecated

                
        $payment->setReferenceInvoice($referenceInvoice);
        if ($debitorId > 0) {
            $payment->setDebitorId($debitorId);
        }
                
        if ($invoice = $this->invoiceRepository->findByUid((int)$referenceInvoice)) {
            $payment->setInvoice($invoice);
            $invoiceDebitorId = $invoice->getRawEntryId();
            if ($invoiceDebitorId > 0) {
                $payment->setDebitorId($invoiceDebitorId);
            }
        }
        //$payment->setImport(1);
        if ($tenantId !== null) {
            $payment->setVerbandId($tenantId);
        }
        $payment->setCharges($chargesTotal);
        $payment->setIban($iban);
        return $payment;
    }
    public function getReferenceFromText(string $text): string
    {
        // Regular expression to match the reference number following "QRR"
        $pattern = '/QRR\s(\d{27})/';
        // Check if the pattern matches
        if (preg_match($pattern, $text, $matches)) {
            // The reference number is in the first capturing group
            return $matches[1];
        }

        $textWithoutSpaces = trim(str_replace(' ', '', $text));
        // Regular expression to match the reference number following "QRR" without spaces
        $pattern = '/QRR(\d{27})/';
        // Check if the pattern matches
        if (preg_match($pattern, $textWithoutSpaces, $matches)) {
            // The reference number is in the first capturing group
            return $matches[1];
        }
        // Regular expression to match the reference number following "RF"
        //$pattern = '/RF\d+/';
        $pattern = '/RF\d{22}/';
        // Check if the pattern matches
        if (preg_match($pattern, $textWithoutSpaces, $matches)) {
            // The reference number is in the first capturing group
            return $matches[0];
        }
        return '';
    }
}
