<?php

declare(strict_types=1);

namespace Ig\IgFibu\Controller;

use Ig\IgFibu\Domain\Model\Credit;
use Ig\IgFibu\Domain\Model\Invoice;
use Ig\IgFibu\Domain\Model\Payment;
use Ig\IgFibu\Domain\Repository\CreditRepository;
use Ig\IgFibu\Domain\Repository\MasterCreditRepository;
use Ig\IgFibu\Service\BookingService;
use Ig\IgFibu\Service\DebitorService;
use Internetgalerie\IgDoctrinePaginator\Pagination\DoctrinePaginator;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Messaging\FlashMessage;
use TYPO3\CMS\Core\Messaging\FlashMessageService;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Annotation\IgnoreValidation;
use TYPO3\CMS\Extbase\Http\ForwardResponse;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;

class CreditController extends ActionController
{
    protected $searchDefault = [];
    protected $optionsStatus = [
        0 => 'offen',
        1 => 'verbucht',
    ];

    /**
     * debitorService
     *
     * @var DebitorService
     */
    protected $debitorService = null;
        

    /**
     * flashMessageIdentifier
     *
     * @var string
     */
    protected $flashMessageIdentifier = 'extbase.flashmessages.tx_igfibu';

    /**
     * creditRepository
     *
     * @var CreditRepository
     */
    protected $creditRepository = null;


    
    public function injectDebitorService(DebitorService $debitorService): void
    {
        $this->debitorService = $debitorService;
    }

    
    public function injectCreditRepository(CreditRepository $creditRepository): void
    {
        $this->creditRepository = $creditRepository;
    }

 
    public function injectFlashMessageService(FlashMessageService $flashMessageService): void
    {
        $this->flashMessageService = $flashMessageService;
    }
    
    public function initializeAction(): void
    {
        parent::initializeAction();
        $this->search = $this->getSearch();
        $this->control = $this->getControl();
        //$this->redirectUrl = $this->request->hasArgument('redirectUrl') ? $this->request->getArgument('redirectUrl') : '';
        //var_dump($this->search, $this->request->getArguments());exit(0);
    }

    /**
     * action list
     */
    public function listAction(): ResponseInterface
    {
        $creditQueryBuilder = $this->creditRepository->findBySearch($this->search);
        $currentPage = $this->request->hasArgument('currentPage') ? (int) $this->request->getArgument(
            'currentPage'
        ) : 1;

        // $paginator = new QueryResultPaginator($credits, $currentPage, 50);
        $paginator = new DoctrinePaginator($creditQueryBuilder, $currentPage, 50, Credit::class);
        $pagination = $paginator->createSlidingWindowPagination(10);


        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);

        
        //$credits = $this->creditRepository->findAll();
        //$this->view->assign('credits', $credits);
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/tableSelectToggle.js');

        $default = $this->request->hasArgument('default') ? $this->request->getArgument('default') : [];

        if ($this->search['customerId'] ?? false) {
            $default = [
                'debitor' => (int)$this->search['customerId'],
            ];
        } elseif ($this->search['customerNumber'] ?? false) {
            $customerNumber = $this->search['customerNumber'];

            //var_dump($this->search['customerNumber']);exit(0);
            // $default = ['debitor' => 1535];
        }
        $this->view->assign('default', [
            'default' => $default,
        ]);

        
        $this->assignSearchForm();
        return $this->htmlResponse();
    }

    /**
     * action show
     */
    public function showAction(Credit $credit): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($credit)) {
            return $this->htmlResponse('Access Denied');
        }
        $activeTab = $this->request->hasArgument('activeTab') ? $this->request->getArgument('activeTab') : '';
        $this->view->assign('activeTab', $activeTab);
        $this->view->assign('credit', $credit);
        return $this->htmlResponse();
    }

    /**
     * action new
     */
    public function newAction(): ResponseInterface
    {
        $default = $this->request->hasArgument('default') ? $this->request->getArgument('default') : null;
        if (!isset($default['tenantId'])) {
            $default['tenantId'] = 0;
        }

        $this->assignFormfields((int)$default['tenantId']);
        $this->view->assign('default', $default);
        return $this->htmlResponse();
    }

    /**
     * action createWithPayment
     */
    public function createWithPaymentAction(Payment $payment): ResponseInterface
    {
        $bookingService = GeneralUtility::makeInstance(BookingService::class);
        $bookingService->setSettings($this->settings);
        if ($this->request->hasArgument('title')) {
            $title = $this->request->getArgument('title');
        } else {
            $title = 'Zahlung vom ' . $payment->getDatePay()->format('d.m.Y');
        }
        $credit = $bookingService->createCreditWithOpenAmountOfPayment($payment, $title);
        return $this->htmlResponse($this->redirectToPaymentShow($payment));
    }

    /**
     * action createWithInvoice
     */
    public function createWithInvoiceAction(Invoice $invoice): ResponseInterface
    {
        $bookingService = GeneralUtility::makeInstance(BookingService::class);
        $bookingService->setSettings($this->settings);

        $title = $this->request->hasArgument('title') ? $this->request->getArgument('title') : '';
        if ($title == '') {
            if ($invoice->getAmountOpen() > 0) {
                $title = 'Offener Betrag von Rechnung Nr. ' . $invoice->getUid();
            } else {
                $title = 'Zuviel bezahlter Betrag von Rechnung Nr. ' . $invoice->getUid();
            }
        }
        $credit = $bookingService->createCreditWithOpenAmountOfInvoice($invoice, $title);
        return $this->htmlResponse($this->redirectToInvoiceShow($invoice, 'payments'));
    }

    /**
     * action create
     */
    public function createAction(Credit $newCredit): ResponseInterface
    {
        $this->creditRepository->add($newCredit);
        $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
        $persistenceManager->persistAll();
        return (new ForwardResponse('show'))->withArguments([
            'credit' => $newCredit->getUid(),
        ]);
    }

    /**
     * action edit
     */
    #[IgnoreValidation([
        'argumentName' => 'credit',
    ])]
    public function editAction(Credit $credit): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($credit)) {
            return $this->htmlResponse('Access Denied');
        }
        //var_dump($this->getDebitorService()->getConfig('Contact'));exit(0);
        $this->assignFormfields($credit->getTenantId());
        $this->view->assign('credit', $credit);
        return $this->htmlResponse();
    }

    /**
     * action update
     */
    public function updateAction(Credit $credit): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($credit)) {
            return $this->htmlResponse('Access Denied');
        }
        $this->creditRepository->update($credit);
        return (new ForwardResponse('show'))->withArguments([
            'credit' => $credit->getUid(),
        ]);
    }


    /**
     * action delete
     */
    public function deleteAction(Credit $credit)
    {
        if (!$this->debitorService->hasPermission($credit)) {
            return $this->htmlResponse('Access Denied');
        }
        $debitor = $credit->getDebitor();
        $tenantId = $credit->getTenantId();
        $this->creditRepository->remove($credit);
        
        $uri = $this->debitorService->getUriByDebitor($this->request, 'show', $debitor, $tenantId);
        if ($uri) {
            return $this->redirectToUri($uri, 0, 302);
        }
        return $this->redirect('list');
    }

    /**
     * action book credit with invoice
     */
    public function bookAction(Invoice $invoice, Credit $credit): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($credit) || !$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }

        $bookingService = GeneralUtility::makeInstance(BookingService::class);
        $bookingService->setSettings($this->settings);
        $booked = $bookingService->bookCreditToInvoice($credit, $invoice, false);
        if ($booked) {
            $message = GeneralUtility::makeInstance(
                FlashMessage::class,
                'Gutschrift wurde verrechnet.',
                '',
                ContextualFeedbackSeverity::OK,
                true
            );
        } else {
            $message = GeneralUtility::makeInstance(
                FlashMessage::class,
                'Gutschrift kann nicht verrechnet werden.',
                '',
                ContextualFeedbackSeverity::ERROR,
                true
            );
        }
        $this->flashMessageService
            ->getMessageQueueByIdentifier($this->flashMessageIdentifier)
            ->enqueue($message);

        return $this->htmlResponse($this->redirectToInvoiceShow($invoice, 'payments'));
    }
    /**
     * action: try to book all open credits with invoices in draft state
     */
    public function bookFormAction(): ResponseInterface
    {
        $this->bookAndAssignCredits(false);
        $this->assignSearchForm();
        return $this->htmlResponse();
    }

    /**
     * action: try to book all open credits with invoices in draft state
     */
    public function bookOpenAction(): ResponseInterface
    {
        $totals = $this->bookAndAssignCredits(true);
        $this->addFlashMessage(
            $totals['booked'] . ' Gutschriften wurden verbucht.',
            '',
            ContextualFeedbackSeverity::OK
        );
        $this->assignSearchForm();
        return $this->htmlResponse();
    }
    

    /**
     * assign for all actions
     */
    public function assignFormfields(?int $tenantId = null): void
    {
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/ig_fibu.js');
        
        $this->view->assign('debitorSettings', $this->debitorService->getConfig('Contact'));
        $tenants = [];
        if ($this->settings['mandant']['active']) {
            $this->verbandRepository = GeneralUtility::makeInstance($this->settings['mandant']['repository']);
            $tenants = $this->verbandRepository->findAllWithAcl(true);
        }
        $this->masterCreditRepository = GeneralUtility::makeInstance(MasterCreditRepository::class);
        //$masterCredits = $this->masterCreditRepository->findAll();
        $masterCredits = $this->masterCreditRepository->findByTenantId($tenantId);
        $this->view->assign('masterCredits', $masterCredits);
        $this->view->assign('tenants', $tenants);
    }

    protected function getSearch()
    {
        $search = [];
        if ($this->request->hasArgument('search')) {
            $search = $this->request->getArgument('search');
        } else {
            $search = $this->searchDefault;
        }

        return $search;
    }

    protected function getControl()
    {
        return $this->request->hasArgument('control') ? $this->request->getArgument('control') : [];
    }
    protected function assignSearchForm()
    {
        $verbands = [];
        if ($this->settings['mandant']['active']) {
            $this->verbandRepository = GeneralUtility::makeInstance($this->settings['mandant']['repository']);
            $verbands = $this->verbandRepository->findAllWithAcl(true);
        }
        //foreach ($verbands as $verband) { var_dump($verband);}exit(0);
        $this->view->assign('verbands', $verbands);
        $this->view->assign('optionsStatus', $this->optionsStatus);
        $this->view->assign('search', $this->search);
        $this->view->assign('control', $this->control);
    }
    
    protected function bookAndAssignCredits(bool $doBooking = true): array
    {
        $creditQueryBuilder = $this->creditRepository->findOpenBySearch($this->search);
        $credits = $this->creditRepository->findBySearchExecute($creditQueryBuilder);
        $bookingService = GeneralUtility::makeInstance(BookingService::class);
        $bookingService->setSettings($this->settings);

        $totals = [
            'total' => 0,
            'booked' => 0,
        ];
        $bookedCredits = [];
        foreach ($credits as $credit) {
            $totals['total']++;
            $invoice = $bookingService->bookCreditToDraftInvoices($credit, $doBooking);
            $booked = $invoice instanceof Invoice;
            if ($booked) {
                $totals['booked']++;
            }
            $bookedCredits[] = [
                'booked' => $booked,
                'invoice' => $invoice,
                'credit' => $credit,
            ];
        }

        $this->view->assign('bookedCredits', $bookedCredits);
        $this->view->assign('totals', $totals);
        return $totals;
    }


    protected function redirectToInvoiceShow($invoice, $activeTab)
    {
        $this->uriBuilder
            ->reset()
            ->setRequest($this->request)
            ->setTargetPageUid((int)$this->settings['invoicePid']);
        $uri = $this->uriBuilder->uriFor(
            'show',
            [
                'invoice' => $invoice->getUid(),
                'activeTab' => $activeTab,
            ],
            'Invoice',
            null,
            'Re'
        );
        return $this->redirectToUri($uri);
    }
    protected function redirectToPaymentShow($payment)
    {
        $this->uriBuilder
            ->reset()
            ->setRequest($this->request)
            ->setTargetPageUid((int)$this->settings['paymentPid']);
        $uri = $this->uriBuilder->uriFor('show', [
            'payment' => $payment->getUid(),
        ], 'Payment', null, 'Za');
        return $this->redirectToUri($uri);
    }
}
