<?php

declare(strict_types=1);

namespace Ig\IgFibu\Database\Query;

use Ig\IgFibu\Service\DebitorService;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Utility\GeneralUtility;

class CreditQueryBuilder extends BaseQueryBuilder
{
    public $deliveryMethodStatusIds = [1, 2, 16];
    protected $tablename = 'tx_igfibu_domain_model_credit';
    protected $tenantIdAttribute = 'tenant_id';

    protected $entryTablename = null;
    protected $entryPrimaryKey = 'uid';
    protected $invoicepaymentTablename = 'tx_igfibu_domain_model_invoicepayment';

    
    /**
     * debitorService
     *
     * @var DebitorService
     */
    protected $debitorService = null;


    /**
     * Select
     *
     * @var Select
     */
    protected $sqlSelect = '*';


    
    public function injectDebitorService(DebitorService $debitorService): void
    {
        $this->debitorService = $debitorService;
    }

    public function resetSelect()
    {
        $this->sqlSelect = '*';
        return $this;
    }

    public function setEntryTablename($entryTablename)
    {
        $this->entryTablename = $entryTablename;
        return $this;
    }

    public function getEntryTablename()
    {
        return $this->entryTablename;
    }

    public function setEntryPrimaryKey($entryPrimaryKey)
    {
        $this->entryPrimaryKey = $entryPrimaryKey;
        return $this;
    }

    public function getEntryPrimaryKey()
    {
        return $this->entryPrimaryKey;
    }

    
    public function addSearch($search)
    {
        $constraints = [];
        if (isset($search['uid']) && $search['uid'] !== '') {
            $constraints[] = $this->expr()->eq(
                $this->tablename . '.uid',
                $this->createNamedParameter(trim((string) $search['uid']), Connection::PARAM_INT)
            );
        }
        if (isset($search['isDone']) && $search['isDone'] !== '') {
            $constraints[] = $this->expr()->eq(
                'is_done',
                $this->createNamedParameter($search['isDone'], Connection::PARAM_INT)
            );
        }

        if (isset($search['verband']) && $search['verband'] !== '') {
            $constraints[] = $this->expr()->eq('tenant_id', (int) $search['verband']);
        }
        if (isset($search['debitorUid']) && $search['debitorUid'] !== '') {
            $constraints[] = $this->expr()->eq('debitor_id', (int) $search['debitorUid']);
        }

        if (isset($search['ids']) && $search['ids'] !== '') {
            $ids = GeneralUtility::intExplode(',', $search['ids'], true);
            if (!empty($ids)) {
                $constraints[] = $this->expr()->in(
                    $this->tablename . '.uid',
                    $this->createNamedParameter($ids, Connection::PARAM_INT_ARRAY)
                );
            }
        }

        $constraints = $this->addEqualRangeConstrain($constraints, 'amount', $search['amount'] ?? null);
        $constraints = $this->addEqualRangeConstrain(
            $constraints,
            $this->tablename . '.uid',
            $search['number'] ?? null
        );

        $hasBookFrom = (isset($search['bookFrom']['date']) && $search['bookFrom']['date'] !== '');
        $hasBookTo = (isset($search['bookTo']['date']) && $search['bookTo']['date'] !== '');
        if ($hasBookFrom || $hasBookTo) {
            $subselectWhere = [];
            if ($hasBookFrom) {
                $subselectWhere[] = 'DATE(booking_date) >= ' . $this->createNamedParameter($search['bookFrom']['date']);
            }
            if ($hasBookTo) {
                $subselectWhere[] = 'DATE(booking_date) <= ' . $this->createNamedParameter($search['bookTo']['date']);
            }
            $constraints[] = $this->tablename . '.uid IN (SELECT credit FROM ' . $this->invoicepaymentTablename . ' WHERE  ' . implode(
                ' AND ',
                $subselectWhere
            ) . ')';
        }


        // @todo re_betrieb in pedos ("concat" form config)
        if (isset($search['customerId']) && $search['customerId'] !== '') {
            $attribute = $this->debitorService->getEntryPrimaryKey();
            $constraints[] = $this->expr()->eq($attribute, (int)$search['customerId']);
        }
        if (isset($search['customerNumber']) && $search['customerNumber'] !== '') {
            $entryNumberName = $this->debitorService->getEntryNumberName();
            $constraints[] = $entryNumberName . ' LIKE ' . $this->createNamedParameter($search['customerNumber'] . '%');
        }
        if (isset($search['keyword']) && $search['keyword'] !== '') {
            // Personen sub search
            $keywordsArray = GeneralUtility::trimExplode(' ', $search['keyword'], true);
            $entryKeywordSearchAttributes = array_merge(['title'], $this->debitorService->getKeywordSearchAttributes());
                
            if (!empty($keywordsArray) && !empty($entryKeywordSearchAttributes)) {
                foreach ($keywordsArray as $keyword) {
                    $subConstraints = [];
                    foreach ($entryKeywordSearchAttributes as $type => $attribute) {
                        if ($type == 'concat') {
                            $concatAttributes = GeneralUtility::trimExplode(',', $attribute, true);
                            $concat = [];
                            foreach ($concatAttributes as $concatAttribute) {
                                if (str_starts_with($concatAttribute, '\'')) {
                                    $concatAttribute = trim($concatAttribute, '\'');
                                    $concat[] = $this->createNamedParameter($concatAttribute);
                                } else {
                                    $concat[] = $this->quoteIdentifier($concatAttribute);
                                }
                            }
                            $subConstraints[] = 'concat(' . implode(
                                ',',
                                $concat
                            ) . ') LIKE ' . $this->createNamedParameter('%' . $keyword . '%');
                            /*
                            // risky and probably we don't want that
                            } elseif ($type == 'function') {
                            $subConstraints[] = $attribute . ' LIKE ' . $this->createNamedParameter('%' . $keyword . '%');
                            */
                        } else {
                            $subConstraints[] = $this->expr()->like(
                                $attribute,
                                $this->createNamedParameter('%' . $keyword . '%')
                            );
                        }
                    }
                    $subConstraints[] = $this->expr()->eq(
                        $this->tablename . '.uid',
                        $this->createNamedParameter($keyword, Connection::PARAM_INT)
                    );
                    $constraints[] = $this->expr()->or(...$subConstraints);
                }

                //$constraints[] = $this->expr()->eq($this->entryTablename . '.uid', $this->quoteIdentifier('entry_id'));
            }
        }
        if (!empty($constraints)) {
            $this->andWhere(...$constraints);
        }
        return $this;
    }
    /**
     * equal range constraints
     */
    protected function addEqualRangeConstrain(array $constraints, string $sttribute, array $searchEntry = null)
    {
        if ($searchEntry === null) {
            return $constraints;
        }
        $mode = $searchEntry['mode'] ?? null;
        if ($mode === 'equal' || $mode === null) {
            if (isset($searchEntry['equal']) && $searchEntry['equal'] !== '') {
                $constraints[] = $this->expr()->eq($sttribute, $this->createNamedParameter($searchEntry['equal']));
            }
        }
        if ($mode === 'range' || $mode === null) {
            if (isset($searchEntry['from']) && $searchEntry['from'] !== '') {
                $constraints[] = $this->expr()->gte($sttribute, $this->createNamedParameter($searchEntry['from']));
            }
            if (isset($searchEntry['to']) && $searchEntry['to'] !== '') {
                $constraints[] = $this->expr()->lte($sttribute, $this->createNamedParameter($searchEntry['to']));
            }
        }
        return $constraints;
    }
}
