<?php

namespace Ig\IgFibu\Domain\Repository;

use DBALException;
use Ig\IgFibu\Database\Query\InvoiceQueryBuilder;
use Ig\IgFibu\Domain\Model\Credit;
use Ig\IgFibu\Domain\Model\Invoice;
use Ig\IgFibu\Domain\Model\InvoiceDate;
use Ig\IgFibu\Domain\Model\Invoicestatus;
use Ig\IgFibu\Utility\PeriodUtility;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\QueryBuilder;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\Exception\IllegalObjectTypeException;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;

/**
 * The repository for Invoices
 */
class InvoiceRepository extends BaseRepository
{
    //protected $type = 'Invoice';
    protected static $tablename = 'tx_igfibu_domain_model_invoice';
    protected static $queryBuilderClass = InvoiceQueryBuilder::class;
    protected static $localDebitorAttribute = 'entry_id';
    protected static $localClientAttribute = 'verband_id';
    protected static $modelClass = Invoice::class;
    
    /**
     * calculate all subtotal/total
     *
     * @param object $invoice and invoice
     */
    public function calculate($invoice): void
    {
        if (!$invoice instanceof $this->objectType) {
            throw new IllegalObjectTypeException(
                'The object given to calculate() was not of the type (' . $this->objectType . ') this repository manages.',
                1668789581
            );
        }
        $this->invoiceUtility->calculateInvoice($invoice);
    }

    /**
     * Adds an object to this repository
     */
    public function add($invoice): void
    {
        $this->invoiceUtility->calculateInvoice($invoice);
        parent::add($invoice);
        $this->persistenceManager->persistAll();
        InvoiceDateRepository::log(
            $invoice->getUid(),
            $invoice->getStatus()
->getUid(),
            null,
            InvoiceDate::ACTION_CREATE
        );
    }

    /**
     * Replaces an existing object with the same identifier by the given object
     *
     * @param object $invoice The modified object
     */
    public function update($invoice): void
    {
        parent::update($invoice);
    }

    /**
     * Replaces an existing object with the same identifier by the given object
     *
     * @param object $invoice The modified object
     */
    public function updateAndCalculate($invoice): void
    {
        if ($invoice->_isDirty('status')) {
            InvoiceDateRepository::log(
                $invoice->getUid(),
                $invoice->getStatus()
->getUid(),
                null,
                InvoiceDate::ACTION_SET
            );
        }

        $this->invoiceUtility->calculateInvoice($invoice);
        $this->update($invoice);
        $this->persistenceManager->persistAll();
    }
    
    /**
     * add persists add invoice to DB. So the uid is set
     *
     * @param Invoice $invoice the invoice to add
     */
    public function persistAdd(Invoice $invoice): void
    {
        $this->add($invoice);
        // persistAll is now in add method
    }
    
    public static function andVerband(array $verbandsUids, array $search = [], $sqlPrefix = ' AND ')
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(static::$tablename);
        $constraints = [];
        if (!empty($verbandsUids)) {
            $constraints[] = static::$tablename . '.verband_id IN (' . implode(',', $verbandsUids) . ')';
        }
        if (isset($search['tablenames']) && $search['tablenames'] !== '') {
            $constraints[] = 'tablenames=' . $conn->quote($search['tablenames']) . '';
        }
        if (isset($search['type']) && $search['type'] !== '') {
            $constraints[] = 'type=' . $conn->quote($search['type']) . '';
        }
        if (isset($search['year']) && $search['year'] !== '') {
            $constraints[] = 'YEAR(create_date)=' . (int)$search['year'];
        }
        if (isset($search['period']) && $search['period'] !== '') {
            $constraints[] = 'tx_igfibu_domain_model_invoiceitem.period=' . $conn->quote($search['period']);
            //$constraints[] = 'SUBSTR(tx_igfibu_domain_model_invoiceitem.period,1,4)=' . $conn->quote($search['period']);
        }
        return empty($constraints) ? '' : ' ' . $sqlPrefix . ' (' . implode(' AND ', $constraints) . ')';
        //return empty($verbandsUids) ? '' : ' ' . $sqlPrefix . ' verband_id IN (' . implode(',', $verbandsUids) . ') ';
    }

    /*
      public function queryBuilderAddSearch($queryBuilder, $search)
      {
      }
    */
    public function getDebitorConfig()
    {
        return $this->debitorService->getAllConfig();
    }
    public function findStatistikType(array $verbandsUids)
    {
        $sql = 'SELECT type FROM tx_igfibu_domain_model_invoice ' . static::andVerband(
            $verbandsUids,
            [],
            'WHERE'
        ) . ' GROUP BY type ORDER BY type';
        $types = [];
        foreach ($this->sqlFetchAll(static::$tablename, $sql) as $type) {
            if ($type['type'] == 'InvoiceAnnual') {
                $label = 'Jahresrechnung';
            } elseif ($type['type'] == 'Invoice') {
                $label = 'Rechnung';
            } else {
                $label = $type['type'];
            }

            $types[] = [
                'type' => $type['type'],
                'label' => $label,
            ];
        }
        
        return $types;
    }

    
    public function findStatistikStatus(array $verbandsUids, array $search = [])
    {
        $sql = 'SELECT status, SUM(total_exkl) AS sum_exkl,SUM(total) AS sum_ez,count(1) AS entry_count,tx_igfibu_domain_model_invoicestatus.title, tx_igfibu_domain_model_invoicestatus.background_color AS color FROM tx_igfibu_domain_model_invoice LEFT OUTER JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid AND tx_igfibu_domain_model_invoicestatus.deleted=0 WHERE tx_igfibu_domain_model_invoice.deleted=0 ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY status,tx_igfibu_domain_model_invoicestatus.title ORDER BY tx_igfibu_domain_model_invoicestatus.sorting';
        return $this->sqlFetchAll(static::$tablename, $sql);
    }

    public function findStatistikMainStatus(array $verbandsUids, array $search = [])
    {
        // 1 = Entwurf / Neu
        // 2 = Offen
        // 16,17,32,33,256,257 = Mahnung / Überfällig
        // Teilzahlungen
        // 128, 513 = Warten / In Bearbeitung
        $sql = "SELECT CASE WHEN status IN (16,17,32,33,256,257) THEN '16,17,32,33,256,257' WHEN status IN (128, 513) THEN '128,513' ELSE status END AS main_status, SUM(total) AS sum_ez,count(1) AS entry_count,CASE WHEN status IN (16,17,32,33,256,257) THEN 'Überfällig (Mahnungen)' WHEN status IN (128, 513) THEN 'Warten/In Bearbeitung' ELSE tx_igfibu_domain_model_invoicestatus.title END AS title, CASE WHEN status IN (16,17,32,33,256,257) THEN '#a00000' WHEN status IN (128, 513) THEN '#666666' ELSE tx_igfibu_domain_model_invoicestatus.background_color END AS color FROM tx_igfibu_domain_model_invoice LEFT OUTER JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid AND tx_igfibu_domain_model_invoicestatus.deleted=0 WHERE tx_igfibu_domain_model_invoice.deleted=0 AND status IN (1,2,16,17,32,33,256,257,128,513) " . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY main_status ORDER BY tx_igfibu_domain_model_invoicestatus.sorting';
        return $this->sqlFetchAll(static::$tablename, $sql);
    }

    public function findStatistikCreatedate(array $verbandsUids, array $search = [])
    {
        $sql = 'SELECT SUM(total_exkl) AS sum_exkl,SUM(total) AS sum_ez,SUM(CASE WHEN is_finished=0 THEN total ELSE 0 END) AS sum_open,SUM(CASE WHEN is_paid THEN total ELSE 0 END) AS sum_paid,SUM(CASE WHEN is_finished AND is_paid=0 THEN total ELSE 0 END) AS sum_lost,count(1) AS entry_count,YEAR(create_date) AS jahr,MONTH(create_date) AS monat FROM tx_igfibu_domain_model_invoice LEFT OUTER JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid WHERE tx_igfibu_domain_model_invoice.deleted=0  AND is_draft=0 AND is_in_statistic=1 AND is_suspended=0 ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY YEAR(create_date),MONTH(create_date) ORDER BY jahr DESC,monat DESC';
        return $this->sqlFetchAll(static::$tablename, $sql);
    }

    public function findStatistikPaiddateJahr(array $verbandsUids, array $search = [])
    {
        $sql = 'SELECT SUM(total_exkl) AS sum_exkl,SUM(total) AS sum_ez,count(1) AS entry_count,YEAR(paid_date) AS jahr FROM tx_igfibu_domain_model_invoice WHERE tx_igfibu_domain_model_invoice.deleted=0 AND status=4 ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY YEAR(paid_date) ORDER BY jahr DESC';
        return $this->sqlFetchAll(static::$tablename, $sql);
    }

    public function findStatistikPaiddate(array $verbandsUids, array $search = [])
    {
        $sql = 'SELECT SUM(total_exkl) AS sum_exkl,SUM(total) AS sum_ez,count(1) AS entry_count,YEAR(paid_date) AS jahr,MONTH(paid_date) AS monat, DATE_FORMAT(paid_date, "%Y-%M-01") AS title FROM tx_igfibu_domain_model_invoice WHERE tx_igfibu_domain_model_invoice.deleted=0 AND status=4 ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY YEAR(paid_date),MONTH(paid_date) ORDER BY jahr DESC,monat DESC';
        return $this->sqlFetchAll(static::$tablename, $sql);
    }

    public function findStatistikPeriodeJahr(array $verbandsUids, array $search = [])
    {
        //$queryBuilder = $this->connectionPool->getQueryBuilderForTable($this->tablename);
        $queryBuilder = $this->createQueryBuilder();
        $select = 'SUM(1) AS anzahl,SUM(total_exkl) AS sum_exkl,SUM(total) AS sum_ez'; // Invoices
        $select .= ',SUM(CASE WHEN (is_paid OR !is_finished)  AND !is_draft AND is_in_statistic THEN 1 ELSE 0 END) AS total_open_paid_count'; // invoices without storniert/abgeschrieben
        $select .= ',SUM(CASE WHEN (is_paid OR !is_finished)  AND !is_draft AND is_in_statistic THEN total_exkl ELSE 0 END) AS total_open_paid_exkl'; // invoices without storniert/abgeschrieben
        $select .= ',SUM(CASE WHEN (is_paid OR !is_finished)  AND !is_draft AND is_in_statistic THEN total ELSE 0 END) AS total_open_paid_ez'; // invoices without storniert/abgeschrieben
        $select .= ',SUM(CASE WHEN !is_finished  AND !is_draft AND is_in_statistic THEN 1 ELSE 0 END) AS total_open_count'; // invoices open
        $select .= ',SUM(CASE WHEN !is_finished  AND !is_draft AND is_in_statistic THEN total ELSE 0 END) AS total_open_ez'; // invoices open
        $select .= ',SUM(CASE WHEN substring(create_date,1,7)<>substring(invoice_date,1,7) THEN total ELSE 0 END) AS mahnung_ez'; // Mahnung
        $select .= ',SUM(CASE WHEN status=17 THEN total_exkl ELSE 0 END) AS mahnung1_exkl,SUM(CASE WHEN status=17 THEN total ELSE 0 END) AS mahnung1_ez'; // Mahnung
        $select .= ',SUM(CASE WHEN status=33 THEN total_exkl ELSE 0 END) AS mahnung2_exkl,SUM(CASE WHEN status=33 THEN total ELSE 0 END) AS mahnung2_ez'; // Mahnung
        $select .= ',SUM(CASE WHEN is_finished AND is_paid=0 THEN 1 ELSE 0 END) AS lost_count,SUM(CASE WHEN is_finished AND is_paid=0 THEN total ELSE 0 END) AS lost_ez'; // Storniert/Abgeschrieben
        $select .= ',periode,substring(periode,1,4) AS jahr,substring(periode,6,2) AS monat'; //Datum
        $queryBuilder
            ->selectLiteral($select)
            ->from(static::$tablename)
            ->join(
                static::$tablename,
                'tx_igfibu_domain_model_invoicestatus',
                'tx_igfibu_domain_model_invoicestatus',
                $queryBuilder->expr()
->eq('status', $queryBuilder->quoteIdentifier('tx_igfibu_domain_model_invoicestatus.uid'))
            )
            // 64: Stornier, 128: Warten, 512: Gurschriften
            //  length(periode)>0
            ->where(
                'is_draft=0 AND is_in_statistic AND is_suspended=0 AND tx_igfibu_domain_model_invoicestatus.deleted=0 AND ' . static::$tablename . '.deleted=0 AND periode IS NOT NULL ' . static::andVerband(
                    $verbandsUids,
                    $search
                )
            )
            //->where("(status>0 AND status NOT IN (64,128,512)) AND periode IS NOT NULL " . static::andVerband($verbandsUids, $search)) // remove strange entries like '4. Quartal 2021'
            ->groupBy('periode')
            ->orderBy('periode', 'DESC');
        $res = $queryBuilder->executeQuery();
        $ret = $res->fetchAllAssociative();
        //var_dump($ret);exit(0);
        return $ret;
    }
    public function findStatistikInvoiceItemCostCenter(array $verbandsUids, array $search = [], $statusUids = [])
    {
        if (!empty($statusUids)) {
            $sqlWhereStatus = 'status IN (' . implode(',', $statusUids) . ')';
        } else {
            //$sqlWhereStatus = 'status NOT IN (1,64,128,130,512,513)';
            $sqlWhereStatus = 'is_draft=0 AND is_in_statistic AND is_suspended=0 AND (is_finished=0 OR is_paid=1)';
        }
        // ,MONTH(create_date) AS monat ,YEAR(create_date) AS jahr,YEAR(create_date)
        $sql = 'SELECT SUM(amount) AS total_amount, SUM(1) AS total_count, tx_igfibu_domain_model_costcenter.title,tx_igfibu_domain_model_costcenter.uid AS costcenter FROM tx_igfibu_domain_model_invoice JOIN  tx_igfibu_domain_model_invoiceitem ON tx_igfibu_domain_model_invoiceitem.deleted=0 AND invoice=tx_igfibu_domain_model_invoice.uid JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid LEFT OUTER JOIN tx_igfibu_domain_model_costcenter ON cost_center=tx_igfibu_domain_model_costcenter.uid AND tx_igfibu_domain_model_costcenter.hidden=0 AND tx_igfibu_domain_model_costcenter.deleted=0 WHERE tx_igfibu_domain_model_invoice.deleted=0 ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' AND ' . $sqlWhereStatus . ' GROUP BY tx_igfibu_domain_model_costcenter.uid,tx_igfibu_domain_model_costcenter.title ORDER BY title';
        //if ($search['period']=='2021') {var_dump($search);die($sql);}
        return $this->sqlFetchAll(static::$tablename, $sql);
    }


    public function getStatistikTotal($rows)
    {
        $totalCount = 0;
        $totalAmount = 0;
        foreach ($rows as $row) {
            $totalCount += $row['total_count'];
            $totalAmount += $row['total_amount'];
        }
        return [
            'total_amount' => $totalAmount,
            'total_count' => $totalCount,
            //'jahr' =>
            'title' => 'Total',
        ];
    }
    public function findStatistikInvoiceTopCustomers(array $verbandsUids, array $search = [], $statusUids = [])
    {
        if (!empty($statusUids)) {
            $sqlWhereStatus = 'status IN (' . implode(',', $statusUids) . ')';
        } else {
            //$sqlWhereStatus = 'status NOT IN (1,64,128,130,512,513)';
            $sqlWhereStatus = 'is_draft=0 AND is_in_statistic AND is_suspended=0 AND (is_finished=0 OR is_paid=1)';
        }
        // ,MONTH(create_date) AS monat ,YEAR(create_date) AS jahr,YEAR(create_date)

        //$sqlSelectName = "CASE WHEN me_companyname<>'' THEN me_companyname WHEN me_lastname<>'' OR me_firstname<>'' THEN CONCAT(me_lastname,' ',me_firstname) ELSE me_shortname END";
        //$sqlSelectName = "concat('B', bet_k,' ', bet_b, ' ', bet_a)";
        $sqlSelectName = $this->debitorService->getEntryName();

        $sql = 'SELECT SUM(CASE WHEN amount_paid >0 THEN amount_paid ELSE total END) AS total_amount, COUNT(1) AS total_count, entry_id, ' . $sqlSelectName . ' AS title FROM tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid JOIN ' . $this->entryTablename . ' ON ' . $this->entryTablename . '.' . $this->entryPrimaryKey . '=entry_id  WHERE tx_igfibu_domain_model_invoice.deleted=0 ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' AND ' . $sqlWhereStatus . ' GROUP BY entry_id,title ORDER BY total_amount DESC LIMIT 10';

        //var_dump($this->entryTablename, $this->entryPrimaryKey);        die($sql);
        
        //if ($search['period']=='2021') {var_dump($search);die($sql);}
        $data = $this->sqlFetchAll(static::$tablename, $sql);
        $uids = [];
        $total = 0;
        foreach ($data as $row) {
            $total += $row['total_amount'];
            $uids[] = (int)$row['entry_id'];
        }
        if (!empty($uids)) {
            $sql = 'SELECT SUM(total) AS total_amount, COUNT(1) AS total_count FROM tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid WHERE tx_igfibu_domain_model_invoice.deleted=0 AND entry_id NOT IN (' . implode(
                ',',
                $uids
            ) . ') ' . static::andVerband($verbandsUids, $search) . ' AND ' . $sqlWhereStatus . '';
            $other = $this->sqlFetchAll(static::$tablename, $sql);
            $otherTotalAmount = $other[0]['total_amount'] ?? 0;
            if ($otherTotalAmount > 0) {
                $data[] = [
                    'total_amount' => $otherTotalAmount,
                    'total_count' => $other[0]['total_count'],
                    'title' => 'Rest',
                    'entry_id' => 0,
                    
                ];
            }
        }
        return $data;
    }

    public function findStatistikItems(array $verbandsUids, array $search = [], $statusUids = [4])
    {
        $ret = [];
        $sql = 'SELECT * FROM tx_igfibu_domain_model_costcenter WHERE deleted=0 AND hidden=0 AND (verband_id IN (' . implode(
            ',',
            $verbandsUids
        ) . ') OR verband_id IS NULL)';
        $costCenters = $this->sqlFetchAll(static::$tablename, $sql);

        $invoiceCostCenters = [];
        $invoiceSelectArray = [];

        $invoiceItemCostCenters = [];
        $invoiceItemSelectArray = [];
        $invoiceItemAsNameArray = [];
        $invoiceItemUidArray = [];
        $count = 0;
        foreach ($costCenters as $costCenter) {
            $count++;
            if ($costCenter['invoice_attribute']) {
                $invoiceCostCenters[] = $costCenter;
                $invoiceSelectArray[] = 'SUM(' . $costCenter['invoice_attribute'] . ') AS item' . $count;
            } else {
                $invoiceItemCostCenters[] = $costCenter;
                $invoiceItemSelectArray[] = 'SUM(CASE WHEN cost_center=' . $costCenter['uid'] . ' THEN amount ELSE 0 END) AS item' . $count;
                $invoiceItemAsNameArray[] = 'SUM(item' . $count . ') AS item' . $count;
                $invoiceItemUidArray[] = $costCenter['uid'];
            }
        }
        if (empty($invoiceSelectArray) && empty($invoiceItemSelectArray)) {
            return [];
        }

        // Add other
        /*
        $count++;
        $otherCostCenter = new \Ig\IgFibu\Domain\Model\CostCenter();
        $otherCostCenter->setTitle('Sonstiges');
        $otherCostCenter->setCode('OTHER');
        $invoiceItemCostCenters[] = $otherCostCenter;
        $invoiceItemSelectArray[] = 'SUM(CASE WHEN cost_center IS NULL OR cost_center NOT IN (' . implode(',', $invoiceItemUidArray) . ') THEN amount ELSE 0 END) AS item' . $count;
        $invoiceItemAsNameArray[] = 'SUM(item' . $count . ') AS item' . $count;
        */
        
        $invoiceSelectSql = implode(', ', $invoiceSelectArray);

        $invoiceItemSelectSql = empty($invoiceItemSelectArray) ? '' : implode(', ', $invoiceItemSelectArray) . ',';
        $invoiceItemAsNameSql = empty($invoiceItemAsNameArray) ? '' : implode(', ', $invoiceItemAsNameArray) . ',';
        if (!empty($statusUids)) {
            $sqlWhereStatus = 'status IN (' . implode(',', $statusUids) . ')';
        } else {
            //$sqlWhereStatus = 'status NOT IN (1,64,128,130,512,513)';
            $sqlWhereStatus = 'is_draft=0 AND is_in_statistic AND is_suspended=0 AND (is_finished=0 OR is_paid=1)';
        }

        // @todo subquery
        $sql = 'SELECT ' . $invoiceItemAsNameSql . 'SUM(total) AS total,YEAR(create_date) AS jahr,MONTH(create_date) AS monat FROM tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid LEFT OUTER JOIN 
(SELECT ' . $invoiceItemSelectSql . 'invoice FROM tx_igfibu_domain_model_invoiceitem WHERE tx_igfibu_domain_model_invoiceitem.deleted=0 GROUP BY invoice) t ON (invoice=tx_igfibu_domain_model_invoice.uid) WHERE tx_igfibu_domain_model_invoice.deleted=0  ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' AND ' . $sqlWhereStatus . ' GROUP BY YEAR(create_date),MONTH(create_date) ORDER BY jahr DESC,monat DESC';
        //$sql = 'SELECT ' . $invoiceItemSelectSql . ',SUM(total) AS total,YEAR(create_date) AS jahr,MONTH(create_date) AS monat FROM tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoiceitem ON (invoice=tx_igfibu_domain_model_invoice.uid AND tx_igfibu_domain_model_invoiceitem.deleted=0) WHERE tx_igfibu_domain_model_invoice.deleted=0  ' . static::andVerband($verbandsUids, $search) . ' AND ' . $sqlWhereStatus . ' GROUP BY YEAR(create_date),MONTH(create_date) ORDER BY jahr DESC,monat DESC';
        $entriesYearMonth = $this->sqlFetchAll(static::$tablename, $sql);

        $sql = 'SELECT ' . $invoiceItemAsNameSql . 'SUM(total) AS total,YEAR(create_date) AS jahr FROM tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid LEFT OUTER JOIN 
(SELECT ' . $invoiceItemSelectSql . 'invoice FROM tx_igfibu_domain_model_invoiceitem WHERE tx_igfibu_domain_model_invoiceitem.deleted=0 GROUP BY invoice) t ON (invoice=tx_igfibu_domain_model_invoice.uid) WHERE tx_igfibu_domain_model_invoice.deleted=0  ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' AND ' . $sqlWhereStatus . ' GROUP BY YEAR(create_date) ORDER BY jahr DESC';
        //$sql = 'SELECT ' . $invoiceItemSelectSql . ',SUM(total) AS total,YEAR(create_date) AS jahr FROM tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoiceitem ON (invoice=tx_igfibu_domain_model_invoice.uid) WHERE tx_igfibu_domain_model_invoice.deleted=0  ' . static::andVerband($verbandsUids, $search) . ' AND ' . $sqlWhereStatus . ' GROUP BY YEAR(create_date) ORDER BY jahr DESC';
        $entriesYear = $this->sqlFetchAll(static::$tablename, $sql);

        /*
        $sql = 'SELECT ' . $invoiceSelectSql . ',SUM(total) AS total,YEAR(create_date) AS jahr,MONTH(create_date) AS monat FROM tx_igfibu_domain_model_invoice WHERE tx_igfibu_domain_model_invoice.deleted=0  ' . static::andVerband($verbandsUids, $search) . ' AND ' . $sqlWhereStatus . ' GROUP BY YEAR(create_date),MONTH(create_date) ORDER BY jahr DESC,monat DESC';
        $entriesYearMonth = $this->sqlFetchAll(static::$tablename, $sql);
        //echo($sql . ';<br />');
        $sql = 'SELECT ' . $invoiceSelectSql . ',SUM(total) AS total,YEAR(create_date) AS jahr FROM tx_igfibu_domain_model_invoice WHERE tx_igfibu_domain_model_invoice.deleted=0  ' . static::andVerband($verbandsUids, $search) . ' AND ' . $sqlWhereStatus . ' GROUP BY YEAR(create_date) ORDER BY jahr DESC';
        $entriesYear = $this->sqlFetchAll(static::$tablename, $sql);
        //echo($sql . ';<br />');exit(0);
         */
        $ret = [
            'yearMonth' => [
                'headers' => array_merge($invoiceCostCenters, $invoiceItemCostCenters),
                'entries' => $entriesYearMonth,
            ],
            'year' => [
                'headers' => array_merge($invoiceCostCenters, $invoiceItemCostCenters),
                'entries' => $entriesYear,
            ],
        ];
        return $ret;
    }

    /**
     * Invoices und Payments gruppiert nach Jahr, Monat, Type, Referenzart (tablenames) und mitgliedschaft
     */

    public function findStatistikGroupWithPayment(array $verbandsUids, bool $useMembership, array $search = [])
    {
        // ,MONTH(create_date) AS monat,MONTH(create_date)tx_igfibu_domain_model_invoice.verband_id IN (

        $sqlSelect = 'YEAR(create_date) AS jahr,type,tx_igfibu_domain_model_invoice.tablenames,SUM(total) AS total_invoice, SUM(payment_betrag) AS total_payment';
        $sqlFrom = ' tx_igfibu_domain_model_invoice LEFT JOIN (
SELECT invoice,SUM(betrag) AS payment_betrag FROM  tx_igfibu_domain_model_payment WHERE tx_igfibu_domain_model_payment.deleted=0 GROUP BY invoice
) tz ON invoice=tx_igfibu_domain_model_invoice.uid';
        $sqlWhere = 'tx_igfibu_domain_model_invoice.deleted=0 AND status=4 ' . static::andVerband(
            $verbandsUids,
            $search
        );
        $sqlGroupBy = 'YEAR(create_date),type,tx_igfibu_domain_model_invoice.tablenames';
        $sqlOrderBy = 'jahr DESC, type,tx_igfibu_domain_model_invoice.tablenames';
        if ($useMembership) {
            $sqlSelect .= ',mitgliedschaft,tx_igscrm_domain_model_mitgliedschaft.name';
            //$sqlFrom .= ' LEFT OUTER JOIN tx_igscrm_domain_model_mitgliedschaft ON tx_igscrm_domain_model_mitgliedschaft.uid=mitgliedschaft';
            $sqlFrom .= ' LEFT OUTER JOIN tx_igscrm_domain_model_mitgliedschaft ON tx_igscrm_domain_model_mitgliedschaft.uid=mitgliedschaft OR tx_igscrm_domain_model_mitgliedschaft.uid=(SELECT max(membership) FROM tx_igfibu_domain_model_invoiceitem WHERE deleted=0 AND tx_igfibu_domain_model_invoiceitem.invoice=tx_igfibu_domain_model_invoice.uid AND membership IS NOT NULL)';
            $sqlGroupBy .= ',mitgliedschaft,tx_igscrm_domain_model_mitgliedschaft.name';
            $sqlOrderBy .= ',mitgliedschaft';
        }
        
        $sql = 'SELECT ' . $sqlSelect . ' FROM ' . $sqlFrom . ' WHERE ' . $sqlWhere . ' 
GROUP BY ' . $sqlGroupBy . ' ORDER BY ' . $sqlOrderBy;
        return $this->sqlFetchAll(static::$tablename, $sql);
    }


    
    public function findOpenSettled(int $verbandUid)
    {
        $sql = 'SELECT tx_igfibu_domain_model_invoice.uid,total FROM tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid where verband_id=' . $verbandUid . ' AND can_pay=true AND tx_igfibu_domain_model_invoice.deleted=0 AND total=(SELECT sum(betrag) from tx_igfibu_domain_model_payment where invoice=tx_igfibu_domain_model_invoice.uid AND is_done=true AND is_ok=true)';
        return $this->sqlFetchAll(static::$tablename, $sql);
    }
    public function closeOpenSettled(int $verbandUid)
    {
        $sql = 'UPDATE tx_igfibu_domain_model_invoice SET status=4 WHERE status IN (SELECT uid FROM tx_igfibu_domain_model_invoicestatus WHERE can_pay=true) AND verband_id=' . $verbandUid . ' AND tx_igfibu_domain_model_invoice.deleted=0 AND total=(SELECT sum(betrag) from tx_igfibu_domain_model_payment where invoice=tx_igfibu_domain_model_invoice.uid AND is_done=true AND is_ok=true)';
        //die($sql .'<br />ig_fibu/Classes/Domain/Repository/InvoiceRepository.php');
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(static::$tablename);
        $stmt = $conn->prepare($sql);
        return $stmt->executeQuery();
    }
    
    public function findByVerbandPerson(int $verbandId, string $entryId, string $tablenames = '')
    {
        return $this->findByEntryId($verbandId, $entryId, $tablenames);
    }
    public function findByEntryId(int $verbandId, string $entryId, string $tablenames = '')
    {
        $query = $this->createQuery();
        $constraints = [$query->equals('verbandId', $verbandId), $query->equals('entryId', $entryId)];
        if ($tablenames) {
            $constraints[] = $query->equals('tablenames', $tablenames);
        }
        $query->matching($query->logicalAnd(...$constraints));
        $query->setOrderings(
            [
                'uid' => QueryInterface::ORDER_DESCENDING,
                'createDate' => QueryInterface::ORDER_DESCENDING,
            ]
        );
        return $query->execute();
    }


    public function findOpenByDebitorAndAmount(string $entryId, $amount, int $verbandId = null)
    {
        $query = $this->createQuery();
        $constraints = [
            $query->equals('entryId', $entryId),
            $query->equals('status.canPay', true),
            $query->equals('total', $amount),
        ];

        if ($verbandId > 0) {
            $constraints[] = $query->equals('verbandId', $verbandId);
        }

        //var_dumP($verbandId, $entryId, $amount);exit(0);
        $query->matching($query->logicalAnd(...$constraints));
        // oldest first
        $query->setOrderings(
            [
                'uid' => QueryInterface::ORDER_ASCENDING,
                'createDate' => QueryInterface::ORDER_ASCENDING, //ORDER_DESCENDING,
            ]
        );
        $res = $query->execute();
        return $res->getFirst();
    }


    public function findOpenByDebitor(int $debitorId, int $verbandId = null): array
    {
        $query = $this->createQuery();
        $constraints = [$query->equals('entryId', $debitorId), $query->equals('status.canPay', true)];

        if ($verbandId > 0) {
            $constraints[] = $query->equals('verbandId', $verbandId);
        }

        //var_dumP($verbandId, $entryId, $amount);exit(0);
        $query->matching($query->logicalAnd(...$constraints));
        // oldest first
        $query->setOrderings(
            [
                'uid' => QueryInterface::ORDER_ASCENDING,
                'createDate' => QueryInterface::ORDER_ASCENDING, //ORDER_DESCENDING,
            ]
        );
        $query->setLimit(10);
        return $query->execute()
->toArray();
    }

    
    
    public function findOneByVerbandPeriodeEntry(
        int $verbandId,
        string $periode,
        string $entryId,
        string $tablenames = ''
    ) {
        $query = $this->createQuery();
        $constraints = [
            $query->equals('verbandId', $verbandId),
            $query->equals('periode', $periode),
            $query->equals('entryId', $entryId),
        ];
        if ($tablenames) {
            $constraints[] = $query->equals('tablenames', $tablenames);
        }
        //var_dump($verbandId, $periode, $entryId, $tablenames);exit(0);
        $query->matching($query->logicalAnd(...$constraints));
        return $query->execute()
->getFirst();
    }



    public function findDraftByCredit(Credit $credit, bool $invoiceMustBeGreater = true)
    {
        $verbandId = $credit->getTenantId();

        //$queryBuilder = $this->createQueryBuilder();
            

        $query = $this->createQuery();
        $constraints = [$query->equals('entryId', $credit->getDebitorId()), $query->equals('status.isDraft', 1)];

        if ($invoiceMustBeGreater) {
            $constraints[] = $query->greaterThanOrEqual('total', $credit->getAmount());
            $constraints[] = 'total - amount_paid > ' . $credit->getAmount();
        }

        
        if ($verbandId > 0) {
            $constraints[] = $query->equals('verbandId', $verbandId);
        }

        //var_dumP($verbandId, $entryId, $amount);exit(0);
        $query->matching($query->logicalAnd(...$constraints));
        // oldest first
        $query->setOrderings(
            [
                'uid' => QueryInterface::ORDER_ASCENDING,
                'createDate' => QueryInterface::ORDER_ASCENDING, //ORDER_DESCENDING,
            ]
        );
        $res = $query->execute();
        return $res->getFirst();
    }



    
    public function getQueryBuilderBySearch(array $search = [], array $orderings = [])
    {
        $queryBuilder = $this->createQueryBuilderLeftJoinEntries();
        $queryBuilder->select(static::$tablename . '.*');
        $queryBuilder->addSearch($search);
        return $queryBuilder;
    }


    public function findSendMailList($search, $orderings = [])
    {
        return $this->findBySearch($search);
    }

    public function findByUidAndDelete($uids): void
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            static::$tablename
        );
        $queryBuilder
            ->delete(static::$tablename)
            ->where(
                $queryBuilder->expr()
->in('uid', $queryBuilder->createNamedParameter($uids, Connection::PARAM_INT_ARRAY))
            )
            ->executeStatement();
    }
    public function findRawStatusByInvoiceUid(int $invoiceUid)
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(static::$tablename);
        $sql = 'SELECT tx_igfibu_domain_model_invoicestatus.* from tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoicestatus ON status=tx_igfibu_domain_model_invoicestatus.uid AND tx_igfibu_domain_model_invoicestatus.sys_language_uid=0 AND tx_igfibu_domain_model_invoicestatus.deleted=0 WHERE tx_igfibu_domain_model_invoice.uid = ' . (int) $invoiceUid;
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        return $res->fetchAssociative();
    }
    public function findRawDeliveryStatusByInvoiceUid(int $invoiceUid)
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(static::$tablename);
        $sql = 'SELECT delivery_status.* from tx_igfibu_domain_model_invoice JOIN tx_igfibu_domain_model_invoicestatus current_status ON status=current_status.uid AND current_status.sys_language_uid=0 AND current_status.deleted=0 JOIN tx_igfibu_domain_model_invoicestatus delivery_status ON delivery_status.delivery_step=current_status.uid AND delivery_status.sys_language_uid=0 AND delivery_status.deleted=0 WHERE tx_igfibu_domain_model_invoice.uid = ' . (int) $invoiceUid;
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        return $res->fetchAssociative();
    }
    public function findRawLastByVerbandEntryId(int $verbandId, int $entryId, string $tablenames)
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(static::$tablename);
        $sql = 'SELECT * FROM tx_igfibu_domain_model_invoice WHERE verband_id=' . intval(
            $verbandId
        ) . ' AND entry_id=' . intval(
            $entryId
        ) . ' AND tablenames=' . $conn->quote($tablenames) . ' AND deleted=0 ORDER BY create_date DESC';
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        return $res->fetchAssociative();
    }

    public function updateStatus(int $uid, Invoicestatus $invoicestatus, $letterDate = null): void
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            static::$tablename
        );
        $queryBuilder
            ->update(static::$tablename)
            ->where(
                $queryBuilder->expr()
->eq('uid', $queryBuilder->createNamedParameter($uid, Connection::PARAM_INT))
            )->set('status', $invoicestatus->getUid());
        // if the new status is marked as send status and we got a date -> we set it
        if ($invoicestatus->getIsSent()) {
            if (!$letterDate) {
                $letterDate = date('Y-m-d');
            }
            $queryBuilder->set('invoice_date', $letterDate);
            //$queryBuilder->set('letter_date', $letterDate);
            InvoiceDateRepository::log(
                $uid,
                $invoicestatus->getUid(),
                $letterDate,
                InvoiceDate::ACTION_SET,
                InvoiceDate::DELIVERY_METHOD_UNKNOWN
            );
        } else {
            InvoiceDateRepository::log($uid, $invoicestatus->getUid(), $letterDate, InvoiceDate::ACTION_SET);
        }
        $queryBuilder->executeStatement();
    }


    /**
     * Invoices auf verschickt setzten
     *
     * @param array $invoiceUids Komma getrennte Liste der InvoicesIDs
     * @param string $letterDateString Datum des Briefes im Format dd.mm.YY[YY]
     */
    public function updateSetSend(
        array $invoiceUids,
        string $letterDateString,
        int $deliveryMode = 0,
        int $logMail = null
    ): array {
        $invoiceDate = strftime('%Y-%m-%d', time());
        $notUpdated = [];
        if (!empty($invoiceUids)) {
            foreach ($invoiceUids as $invoiceUid) {
                $status = $this->findRawStatusByInvoiceUid($invoiceUid);
                $deliveryStatusUid = (int)($status['delivery_step'] ?? 0);
                if ($deliveryStatusUid) {
                    $queryBuilder = $this->connectionPool->getQueryBuilderForTable(static::$tablename);
                    $queryBuilder->update(static::$tablename)
                                 ->where('uid=' . (int)$invoiceUid) // status IN (0,16,32,256) AND
                                 ->set('status', $deliveryStatusUid, false)
                                 ->set('invoice_date', $invoiceDate);
                    // convert format 30.04.72 ro 1972-04-30
                    $dateArray = explode('.', $letterDateString);
                    if (count($dateArray) == 3) {
                        $letterDate = ($dateArray[2] < 1000 ? $dateArray[2] + 2000 : $dateArray[2]) . '-' . $dateArray[1] . '-' . $dateArray[0];
                        $queryBuilder->set('letter_date', $letterDate);
                        $invoiceDate = $letterDate;
                    } elseif (count($dateArray) == 3) {
                        // is the format iso
                        $dateArray = explode('-', $letterDateString);
                        $letterDate = $letterDateString;
                        $queryBuilder->set('letter_date', $letterDate);
                        $invoiceDate = $letterDate;
                    }
                    try {
                        $affectedRows = $queryBuilder->executeStatement();
                        InvoiceDateRepository::log(
                            $invoiceUid,
                            $deliveryStatusUid,
                            $invoiceDate,
                            InvoiceDate::ACTION_DELIVERY,
                            $deliveryMode,
                            $logMail
                        );
                    } catch (DBALException $e) {
                        die('ERROR: updateSetSend ' . $e->getMessage());
                    }
                } else {
                    $notUpdated[] = $invoiceUid;
                }
            }
        }
        return $notUpdated;
    }

    /**
     * Invoices undo:  verschickt setzten (invoice date etc. fehlt ist nur Status)
     *
     * @param array $invoiceUids Komma getrennte Liste der InvoicesIDs
     */
    public function updateSetUnsend(array $invoiceUids): array
    {
        $invoiceDate = strftime('%Y-%m-%d', time());

        $notUpdated = [];
        if (!empty($invoiceUids)) {
            foreach ($invoiceUids as $invoiceUid) {
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable(static::$tablename);
                $status = $this->findRawDeliveryStatusByInvoiceUid($invoiceUid);
                $preDeliveryStatusUid = (int)($status['uid'] ?? 0);
                if ($preDeliveryStatusUid) {
                    $queryBuilder->update(static::$tablename)
                                 ->where('uid=' . (int)$invoiceUid)
                                 ->set('status', $preDeliveryStatusUid, false);
                    try {
                        $affectedRows = $queryBuilder->executeStatement();
                        if ($affectedRows) {
                            InvoiceDateRepository::log(
                                $invoiceUid,
                                $preDeliveryStatusUid,
                                $invoiceDate,
                                InvoiceDate::ACTION_UNDO
                            );
                        }
                    } catch (DBALException $e) {
                        die('ERROR: updateSetUnsend ' . $e->getMessage());
                    }
                }
            }
        }
        return $notUpdated;
    }

    public function findGroupByPeriode(array $search, int $maxResults = 40)
    {
        $queryBuilder = $this->createQueryBuilder();
        if (isset($search['verband'])) {
            $queryBuilder->addSearch([
                'verband' => $search['verband'],
            ]);
        }
        //$queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(static::$tablename);
        $queryBuilder
            ->from(static::$tablename)
            ->selectLiteral('periode,sum(1) AS total')
            ->andWhere("periode IS NOT NULL AND periode<>''")// .  static::andVerband($verbandsUids, []))
            ->groupBy('periode')
            ->orderBy('periode', 'DESC')
            ->setMaxResults($maxResults);
        $res = $queryBuilder->executeQuery();
        $rows = [];
        while ($row = $res->fetchAssociative()) {
            $rows[$row['periode']] = [
                'label' => $row['periode'] . ' (' . $row['total'] . ')',
                'periode' => $row['periode'],
                'count' => $row['total'],
            ];
        }
        return $rows;
    }

    public function findGroupByPeriodeForLabel($invoicePeriodeLengthInMonth, array $search, int $maxResults = 40)
    {
        $rows = $this->findGroupByPeriode($search, $maxResults);
        $labels = [];
        //$config = $this->debitorService->getMandantConfig();
        //$invoicePeriodeLengthInMonth = (int) $config['default']['invoicePeriodeLengthInMonth'];
        $periodUtility = GeneralUtility::makeInstance(PeriodUtility::class);
        $periodService = $periodUtility->getPeriodServiceByDuration($invoicePeriodeLengthInMonth);

        foreach ($rows as $key => $periode) {
            $periodDate = $periodService->createByValue($periode['periode']);
            $periodeLabel = $periodService->getLabel($periodDate);
            //$periodeLabel = $this->debitorService->getPeriodeLabel($periode['periode']);
            /*
              $year = (int) substr($periode['periode'], 0, 4);
              $month = (int) substr($periode['periode'], 5, 2);
              $periodeNumber = (int)(($month - 1) / $invoicePeriodeLengthInMonth) + 1;
              $arguments = [$year, $periodeNumber, ''];
              // leave stange periode untouched
              $periodeLabel = $year < 1000 ? $periode['periode'] : LocalizationUtility::translate('periode-' . $invoicePeriodeLengthInMonth, 'ig_fibu', $arguments);
            */
            $labels[] = [
                'value' => $key,
                'label' => $periodeLabel . ' (' . $periode['count'] . ')',
            ];
            //var_dump( $arguments, $invoicePeriodeLengthInMonth, $labels[$periode] );exit(0);
        }
        return $labels;
    }

    public function findGroupByStatus(array $search)
    {
        if (static::$localDebitorAttribute === null) {
            $queryBuilder = $this->createQueryBuilder();
            $queryBuilder->from(static::$tablename);
        } else {
            $queryBuilder = $this->createQueryBuilderLeftJoinEntries();
        }
        $queryBuilder->addSearch($search);
        $queryBuilder->addUserRestriction();
        $queryBuilder->joinInvoicestatus();
        $queryBuilder
            ->selectLiteral(
                'tx_igfibu_domain_model_invoicestatus.uid, tx_igfibu_domain_model_invoicestatus.title,tx_igfibu_domain_model_invoicestatus.color, tx_igfibu_domain_model_invoicestatus.background_color, status,sum(1) AS total_count,sum(total) as total_inkl,sum(total_exkl) as total_exkl'
            )
            //->andWhere("periode IS NOT NULL AND periode<>''")// .  static::andVerband($verbandsUids, []))
            ->groupBy('status')
            //->addGroupBy('tx_igfibu_domain_model_invoicestatus.uid')
            ->orderBy('tx_igfibu_domain_model_invoicestatus.sorting', 'ASC');
        $res = $queryBuilder->executeQuery();
        return $res->fetchAllAssociative();
    }


    public function findTotal(array $search)
    {
        $queryBuilder = $this->createQueryBuilderLeftJoinEntries();
        //$queryBuilder->select(static::$tablename . '.*');
        //$queryBuilder = $this->createQueryBuilder();

        $queryBuilder->addSearch($search);
        $queryBuilder->addUserRestriction();
        //$queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(static::$tablename);
        $queryBuilder
            //->from(static::$tablename)
            ->selectLiteral(
                'sum(total) as inkl,sum(total_exkl) as exkl, sum(amount_paid) AS payment, sum(pre_payment_credit_amount) AS pre_payment_credit, sum(post_payment_credit_amount) AS post_payment_credit, sum(CASE WHEN total > amount_paid +pre_payment_credit_amount THEN total-amount_paid-pre_payment_credit_amount ELSE 0 END) AS open'
            )
            //->groupBy('*')
        ;
        //->andWhere("periode IS NOT NULL AND periode<>''")// .  static::andVerband($verbandsUids, []))
        $res = $queryBuilder->executeQuery();
        $row = $res->fetchAssociative();
        return [
            [
                'label' => 'Total exkl. MWST',
                'amount' => $row['exkl'],
                'payment' => null,
                'amountOpen' => null,
                'prePaymentCredit' => null,
                'postPaymentCredit' => null,
            ],
            [
                'label' => 'Total inkl. MWST',
                'amount' => $row['inkl'],
                'payment' => $row['payment'],
                'amountOpen' => $row['open'],
                'prePaymentCredit' => $row['pre_payment_credit'],
                'postPaymentCredit' => $row['post_payment_credit'],
            ],
        ];
        //return $res->fetchOne();;
    }
    protected function queryBuilderAddUserSorting(QueryBuilder $queryBuilder, $sortingPropertyWithTable, $sortingOrder)
    {
        if ($sortingPropertyWithTable == 'amountOpen') {
            $queryBuilder->add('orderBy', 'total-amount_paid - pre_payment_credit_amount ' . $sortingOrder, true);
            return true;
        }
        return false;
    }
}
