<?php

namespace Ig\IgFibu\Domain\Repository;

use Ig\IgFibu\Database\Query\PaymentQueryBuilder;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;

/**
 * The repository for Payments
 */
class PaymentRepository extends BaseRepository
{
    protected static $tablename = 'tx_igfibu_domain_model_payment';
    protected static $queryBuilderClass = PaymentQueryBuilder::class;
    protected static $localDebitorAttribute = null;
    protected static $localClientAttribute = null;
    protected $entryTablename = null;
    protected $entryPrimaryKey = null;



    protected $defaultOrderings = [
        'uid' => QueryInterface::ORDER_DESCENDING,
    ];


    /**
     * return a queryBuilder for invoices
     *
     * @return PaymentQueryBuilder
     */
    public function createQueryBuilder()
    {
        $queryBuilder = parent::createQueryBuilder();
        $queryBuilder->setEntryTablename($this->entryTablename);
        $queryBuilder->setEntryPrimaryKey($this->entryPrimaryKey);
        return $queryBuilder;
    }

    public static function andVerband(array $verbandsUids, array $search = [], $sqlPrefix = ' AND ')
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable(static::$tablename);
        $constraints = [static::$tablename . '.deleted=0'];
        if (!empty($verbandsUids)) {
            $constraints[] = static::$tablename . '.verband_id IN (' . implode(',', $verbandsUids) . ')';
        }
        if (isset($search['tablenames']) && $search['tablenames'] !== '') {
            $constraints[] = 'r.tablenames=' . $conn->quote($search['tablenames']);
        }
        if (isset($search['type']) && $search['type'] !== '') {
            $constraints[] = 'r.type=' . $conn->quote($search['type']);
        }
        if (isset($search['year']) && $search['year'] !== '') {
            $constraints[] = 'YEAR(date_pay)=' . (int)$search['year'];
        }
        return empty($constraints) ? '' : ' ' . $sqlPrefix . ' (' . implode(' AND ', $constraints) . ')';
        //return empty($verbandsUids) ? '' : ' ' . $sqlPrefix . ' verband_id IN (' . implode(',', $verbandsUids) . ') ';
    }

    public function findForBooking(?int $tenantId = null)
    {
        $query = $this->createQuery();
        $constraints = [];
        $constraints[] = $query->equals('isDone', 0);
        if ($tenantId !== null) {
            $constraints[] = $query->equals('verbandId', $tenantId);
        }
        $query->matching(...$constraints);
        return $query->execute();
    }

    public function findOpenByInvoice(int $invoiceUid, ?int $debitorId = null)
    {
        $query = $this->createQuery();
        $orConstraints = [];
        $orConstraints[] = $query->logicalAnd($query->equals('invoice', $invoiceUid), $query->equals('isDone', 0));
        if ($debitorId > 0) {
            $orConstraints[] = $query->logicalAnd(
                $query->equals('debitorId', $debitorId),
                $query->equals('isDone', 0)
            );
        }
        $query->matching($query->logicalOr(...$orConstraints));
        return $query->execute()
->toArray();
    }
    public function findOpenByDebitor(int $debitorId)
    {
        $query = $this->createQuery();
        $query->matching($query->logicalAnd($query->equals('debitorId', $debitorId), $query->equals('isDone', 0)));
        return $query->execute()
->toArray();
    }

    /**
     * find by search
     */
    /*
    public function findBySearch(array $search = [])
    {
        $queryBuilder = $this->createQueryBuilder();
        $queryBuilder->select(static::$tablename . '.*');
    }
     */
    public function findByUidAndDelete($uids): void
    {
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(
            'tx_igfibu_domain_model_payment'
        );
        $queryBuilder
            ->delete('tx_igfibu_domain_model_payment')
            ->where(
                $queryBuilder->expr()
->in('uid', $queryBuilder->createNamedParameter($uids, Connection::PARAM_INT_ARRAY))
            )
            ->executeStatement();
    }

    public function sqlFetchAll(string $tablename, string $sql)
    {
        $conn = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable($tablename);
        $stmt = $conn->prepare($sql);
        $res = $stmt->executeQuery();
        return $res->fetchAllAssociative();
    }


    public function findStatistikStatus(array $verbandsUids, array $search = [])
    {
        $sql = 'SELECT is_ok,is_done,CASE WHEN is_done THEN 1 ELSE 0 END + CASE WHEN is_ok THEN 2 ELSE 0 END AS status, SUM(betrag) AS sum_inkl,count(1) entry_count FROM ' . $this->joinInvoice() . ' ' . static::andVerband(
            $verbandsUids,
            $search,
            'WHERE'
        ) . ' GROUP BY is_ok,is_done ORDER BY is_ok,is_done';
        //die($sql);
        return $this->sqlFetchAll('tx_igfibu_domain_model_payment', $sql);
    }

    public function findStatistikValutadateJahr(array $verbandsUids, array $search = [])
    {
        $sql = 'SELECT SUM(betrag) AS sum_inkl,count(1) AS entry_count,YEAR(date_valuta) AS jahr FROM ' . $this->joinInvoice() . ' WHERE is_ok=1 AND date_valuta IS NOT NULL ' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY YEAR(date_valuta) ORDER BY jahr DESC';
        return $this->sqlFetchAll('tx_igfibu_domain_model_payment', $sql);
    }

    public function findStatistikValutadate(array $verbandsUids, array $search = [])
    {
        $sql = "SELECT SUM(betrag) AS sum_inkl,count(1) AS entry_count,YEAR(date_valuta) AS jahr,MONTH(date_valuta) AS monat, DATE_FORMAT(date_valuta, '%Y-%m') AS datum FROM " . $this->joinInvoice() . ' WHERE is_ok=1 AND date_valuta IS NOT NULL' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY YEAR(date_valuta),MONTH(date_valuta),datum ORDER BY jahr DESC,monat DESC';
        return $this->sqlFetchAll('tx_igfibu_domain_model_payment', $sql);
    }
    public function findStatistikPaiddateJahr(array $verbandsUids, array $search = [])
    {
        // date_pay is null
        $sql = 'SELECT SUM(betrag) AS sum_inkl,count(1) AS entry_count,YEAR(date_pay) AS jahr FROM ' . $this->joinInvoice() . ' WHERE is_ok=1 AND date_pay IS NOT NULL' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY YEAR(date_pay) ORDER BY jahr DESC';
        return $this->sqlFetchAll('tx_igfibu_domain_model_payment', $sql);
    }

    public function findStatistikPaiddate(array $verbandsUids, array $search = [])
    {
        // date_pay is null
        $sql = "SELECT SUM(betrag) AS sum_inkl,count(1) AS entry_count,YEAR(date_pay) AS jahr,MONTH(date_pay) AS monat, DATE_FORMAT(date_pay, '%Y-%m') AS datum FROM " . $this->joinInvoice() . ' WHERE is_ok=1 AND date_pay IS NOT NULL' . static::andVerband(
            $verbandsUids,
            $search
        ) . ' GROUP BY YEAR(date_pay),MONTH(date_pay),datum ORDER BY jahr DESC,monat DESC';
        return $this->sqlFetchAll('tx_igfibu_domain_model_payment', $sql);
    }

    public function getYears(array $verbandsUids)
    {
        //SELECT YEAR(min(date_pay)) FROM tx_igfibu_domain_model_payment UNION SELECT YEAR(min(create_date)) FROM tx_igfibu_domain_model_invoice;
        //$andWhere = ' AND deleted=0  AND verband_id IN (' . implode(',', $verbandsUids) . ')';
        $andWhere = static::andVerband($verbandsUids);
        $sql = 'SELECT DISTINCT YEAR(date_pay) AS jahr FROM tx_igfibu_domain_model_payment WHERE date_pay IS NOT NULL ' . $andWhere . ' UNION SELECT DISTINCT YEAR((create_date)) AS jahr FROM tx_igfibu_domain_model_invoice WHERE create_date IS NOT NULL ' . InvoiceRepository::andVerband(
            $verbandsUids
        ) . ' ORDER BY jahr DESC';
        $years = [];
        foreach ($this->sqlFetchAll('tx_igfibu_domain_model_payment', $sql) as $entry) {
            $years[$entry['jahr']] = $entry['jahr'];
        }
        return $years;
    }

    public function findTotal(array $search): array
    {
        $queryBuilder = $this->createQueryBuilder();
        $queryBuilder->addSearch($search);
        //$queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(static::$tablename);

        $select = 'SUM(betrag) AS total_amount, SUM(1) AS count_amount';
        $select .= ', sum(CASE WHEN betrag>0 THEN betrag ELSE 0 END) AS total_positve, SUM(CASE WHEN betrag>0 THEN 1 ELSE 0 END) AS count_positve';
        $select .= ', sum(CASE WHEN betrag<0 THEN betrag ELSE 0 END) AS total_negative, SUM(CASE WHEN betrag<0 THEN 1 ELSE 0 END) AS count_negative';
        $select .= ', SUM(CASE WHEN is_done THEN 0 ELSE betrag END) AS total_not_done, SUM(CASE WHEN is_done THEN 0 ELSE 1 END) AS count_not_done';
        $select .= ', SUM(CASE WHEN is_ok THEN betrag ELSE 0 END) AS total_ok, SUM(CASE WHEN is_ok THEN 1 ELSE 0 END) AS count_ok';
        $select .= ', SUM(CASE WHEN is_done AND NOT is_ok THEN betrag ELSE 0 END) AS total_error, SUM(CASE WHEN is_done AND NOT is_ok THEN 1 ELSE 0 END) AS count_error';

        $queryBuilder
            ->from(static::$tablename)
            //->selectLiteral('sum(total) as total_inkl')
            ->selectLiteral($select)
            //->groupBy('*')
        ;
        //->andWhere("periode IS NOT NULL AND periode<>''")// .  static::andVerband($verbandsUids, []))
        $res = $queryBuilder->executeQuery();
        $row = $res->fetchAssociative();
        $totals = [
            [
                'label' => 'Total [CHF]',
                'amount' => $row['total_amount'],
                'count' => $row['count_amount'],
                'digits' => 2,
            ],
        ];
        if ($row['total_negative'] != 0.0) {
            $totals[] = [
                'label' => 'Total Negativbuchungen [CHF]',
                'amount' => $row['total_negative'],
                'count' => $row['count_negative'],
                'digits' => 2,
            ];
            $totals[] = [
                'label' => 'Total Positivbuchungen [CHF]',
                'amount' => $row['total_positve'],
                'count' => $row['count_positve'],
                'digits' => 2,
            ];
        }
        $totals[] = [
            'label' => 'Total korrekt Verbucht [CHF]',
            'amount' => $row['total_ok'],
            'count' => $row['count_ok'],
            'digits' => 2,
        ];
        $totals[] = [
            'label' => 'Total offen [CHF]',
            'amount' => $row['total_not_done'],
            'count' => $row['count_not_done'],
            'digits' => 2,
        ];
        $totals[] = [
            'label' => 'Total Fehler [CHF]',
            'amount' => $row['total_error'],
            'count' => $row['count_error'],
            'digits' => 2,
        ];
        
        return $totals;
    }


    public function findTotalOpen(array $search)
    {
        $queryBuilder = $this->createQueryBuilder();
        $queryBuilder->addSearch($search);
        //$queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable(static::$tablename);

        $select = 'SUM(betrag) AS total_amount, SUM(1) AS total_count';

        $queryBuilder
            ->from(static::$tablename)
            //->selectLiteral('sum(total) as total_inkl')
            ->selectLiteral($select)
            ->andWhere('is_done = 0')
            //->groupBy('*')
        ;
        //->andWhere("periode IS NOT NULL AND periode<>''")// .  static::andVerband($verbandsUids, []))
        $res = $queryBuilder->executeQuery();
        $row = $res->fetchAssociative();
        return [
            'amount' => $row['total_amount'],
            'count' => $row['total_count'],
            'digits' => 2,
        ];
    }
    

    private function joinInvoice()
    {
        return static::$tablename . ' LEFT OUTER JOIN tx_igfibu_domain_model_invoice r ON invoice=r.uid AND r.deleted=0';
    }
}
