<?php

namespace Ig\IgFibu\Domain\Model;

use Ig\IgFibu\Domain\Repository\CostCenterRepository;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\DomainObject\AbstractEntity;

class InvoiceItem extends AbstractEntity
{
    /**
     * type
     *
     * @var string
     */
    protected $type = '';

    /**
     * sorting
     *
     * @var integer
     */
    protected $sorting = 0;
    

    /**
     * is text defined
     *
     * @var bool
     */
    protected $isLocalText = true;


    /**
     * title
     *
     * @var string
     */
    protected $title = '';

    /**
     * description
     *
     * @var string
     */
    protected $description = '';

    /**
     * taxBehavior (0 = exkl., 1 = inkl.)
     *
     * @var int
     */
    protected $taxBehavior = 0;

    /**
     * taxRate
     *
     * @var float
     */
    protected $taxRate = null;

    /**
     * amount
     *
     * @var float
     */
    protected $amount = 0.0;

    /**
     * totalExkl
     *
     * @var float
     */
    protected $totalExkl = 0.0;

    /**
     * totalInkl
     *
     * @var float
     */
    protected $totalInkl = 0.0;

    /**
     * quantity
     *
     * @var float
     */
    protected $quantity = 0.0;

    /**
     * unitPrice
     *
     * @var float
     */
    protected $unitPrice = 0.0;

    /**
     * costCenter
     *
     * @var CostCenter
     */
    protected $costCenter = null;

    /**
     * period
     *
     * @var string
     */
    protected $period;

    /**
     * creates a new invoiceItem
     *
     * @param  ClientInterface $client Verband/Cient
     * @param  string $description description text for invoice item
     * @param  float $amount amount for invoice item
     * @param  float $taxRate tax rate for invoice item, if null tax rate from client is taken
     * @return InvoiceItem
     */
    public static function createWithDescription(
        ClientInterface $client = null,
        string $description = null,
        float $amount = null,
        float $taxRate = null
    ) {
        $invoiceitem = GeneralUtility::makeInstance(self::class);
        $costCenterRepository = GeneralUtility::makeInstance(CostCenterRepository::class);

        $costCenter = $costCenterRepository->findForVerbandType($client->getUid(), self::class);

        if ($costCenter !== null) {
            $invoiceitem->setCostCenter($costCenter);
        }
        if ($description !== null) {
            $invoiceitem->setDescription($description);
        }
        if ($client->getTaxBehaviorIsExkl()) {
            $invoiceitem->setTotalExkl($amount, $taxRate ?? $client->getTaxRate());
        } else {
            $invoiceitem->setTotalInkl($amount, $taxRate ?? $client->getTaxRate());
        }
        return $invoiceitem;
    }
    
    /**
     * Returns the type
     *
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * Sets the type
     *
     * @param  string $type
     */
    public function setType($type): void
    {
        $this->type = $type;
    }
    public function getTypePartial()
    {
        return end(explode('\\', $this->type));
    }

    /**
     * Returns the sorting
     *
     * @return integer
     */
    public function getSorting()
    {
        return $this->sorting;
    }

    /**
     * Sets the sorting
     *
     * @param integer $sorting
     */
    public function setSorting($sorting): void
    {
        $this->sorting = $sorting;
    }

    public function getIsLocalText(): bool
    {
        return $this->isLocalText;
    }

    public function setIsLocalText(bool $isLocalText): void
    {
        $this->isLocalText = $isLocalText;
    }

    public function isIsLocalText(): bool
    {
        return $this->isLocalText;
    }



    
    /**
     * Returns the title
     *
     * @return string
     */
    public function getTitle()
    {
        return $this->title;
    }

    /**
     * Sets the title
     *
     * @param  string $title
     */
    public function setTitle($title): void
    {
        $this->title = $title;
    }

    /**
     * Returns the description
     *
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Sets the description
     *
     * @param  string $description
     */
    public function setDescription($description): void
    {
        $this->description = $description;
    }

    /**
     * Returns the taxBehavior
     *
     * @return ?int $taxBehavior
     */
    public function getTaxBehavior()
    {
        return $this->taxBehavior;
    }
    public function getTaxBehaviorIsInkl()
    {
        return $this->taxBehavior === Invoice::TAXBEHAVIOR_INKL;
    }
    public function getTaxBehaviorIsExkl()
    {
        return $this->taxBehavior === Invoice::TAXBEHAVIOR_EXKL;
    }

    /**
     * Sets the taxBehavior
     *
     * @param  ?int $taxBehavior
     */
    public function setTaxBehavior($taxBehavior): void
    {
        $this->taxBehavior = $taxBehavior;
    }

    /**
     * Returns the amount
     *
     * @return float
     */
    public function getAmount()
    {
        return $this->amount;
    }

    /**
     * Sets the amount
     *
     * @param  float $amount
     */
    public function setAmount($amount): void
    {
        $this->amount = $amount;
    }

    /**
     * Returns the totalExkl
     *
     * @return float
     */
    public function getTotalExkl()
    {
        return $this->amount;
    }

    /**
     * Sets the amount with tax rates for given client
     *
     * @param  ClientInterface $client Verband/Cient
     * @param  float $amount amount for invoice item
     * @param  float $taxRate tax rate for invoice item, if null tax rate from client is taken
     */
    public function setTotalByClient(ClientInterface $client, $amount, $taxRate = null): void
    {
        if ($client->getTaxBehaviorIsExkl()) {
            $this->setTotalExkl($amount, $taxRate ?? $client->getTaxRate());
        } else {
            $this->setTotalInkl($amount, $taxRate ?? $client->getTaxRate());
        }
    }

    /**
     * Sets the totalExkl
     *
     * @param  float $amount
     * @param  float $taxRate
     */
    public function setTotalExkl($amount, $taxRate): void
    {
        //$this->totalExkl = $totalExkl;
        $this->amount = $amount;
        $this->taxRate = $taxRate;
        $this->taxBehavior = Invoice::TAXBEHAVIOR_EXKL;
    }

    /**
     * Returns the totalInkl
     *
     * @return float
     */
    public function getTotalInkl()
    {
        return $this->amount;
    }

    /**
     * Sets the totalInkl
     *
     * @param  float $amount
     * @param  float $taxRate
     */
    public function setTotalInkl($amount, $taxRate): void
    {
        $this->totalInkl = $amount; // remove
        $this->amount = $amount;
        $this->taxRate = $taxRate;
        $this->taxBehavior = Invoice::TAXBEHAVIOR_INKL;
    }
    public function getTotal()
    {
        return $this->amount;
        //return $this->getTaxBehaviorIsExkl() ? $this->totalExkl : $this->totalInkl;
    }

    /**
     * Returns the quantity
     *
     * @return float
     */
    public function getQuantity()
    {
        return $this->quantity;
    }

    /**
     * Sets the quantity
     *
     * @param  float $quantity
     */
    public function setQuantity($quantity): void
    {
        $this->quantity = $quantity;
    }

    /**
     * Returns the unitPrice
     *
     * @return float
     */
    public function getUnitPrice()
    {
        return $this->unitPrice;
    }

    /**
     * Sets the unitPrice
     *
     * @param  float $unitPrice
     */
    public function setUnitPrice($unitPrice): void
    {
        $this->unitPrice = $unitPrice;
    }


    /**
     * Returns the taxRate
     *
     * @return float
     */
    public function getTaxRate()
    {
        return $this->taxRate;
    }
    public function getMwst()
    {
        return $this->taxRate;
    }

    /**
     * Sets the taxRate
     *
     * @param  float $taxRate
     */
    public function setTaxRate($taxRate): void
    {
        $this->taxRate = $taxRate;
    }
    /**
     * Returns the costCenter
     *
     * @return CostCenter
     */
    public function getCostCenter()
    {
        return $this->costCenter;
    }

    /**
     * Sets the costCenter
     */
    public function setCostCenter(CostCenter $costCenter): void
    {
        $this->costCenter = $costCenter;
    }

    /**
     * Returns the period
     *
     * @return string
     */
    public function getPeriod()
    {
        return $this->period;
    }

    /**
     * Sets the period
     *
     * @param  string $period
     */
    public function setPeriod($period): void
    {
        $this->period = $period;
    }


    
    /**
     * set the default cost center for this invoice item for this customer
     */
    public function setDefaultCostCenter(int $customerUid): void
    {
        $costCenterRepository = GeneralUtility::makeInstance(CostCenterRepository::class);
        $costCenter = $costCenterRepository->findForVerbandType($customerUid, static::class);
        if ($costCenter !== null) {
            $this->setCostCenter($costCenter);
        }
    }
    
    /**
     * replace $description with period and set description
     */
    public function replaceAndSetDescription(string $description): void
    {
        $descriptionReplaced = str_replace('{period}', $this->getPeriod(), $description);
        $this->setDescription($descriptionReplaced);
    }

    public function calculate()
    {
    }
    
    public function setLocalByInvoice(InvoiceInterface $invoice)
    {
    }
}
