<?php

namespace Ig\IgFibu\Service;

use Ig\IgFibu\Domain\Model\Creditor;
use Ig\IgFibu\Domain\Model\FibuAclInterface;
use Ig\IgFibu\Event\UserRestrictionEvent;
use Internetgalerie\IgCrmTemplate\Domain\Model\TenantInterface;
use Psr\EventDispatcher\EventDispatcherInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\Query\QueryBuilder;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\TypoScript\TypoScriptService;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Mvc\RequestInterface;
use TYPO3\CMS\Extbase\Mvc\Web\Routing\UriBuilder;

class DebitorService implements SingletonInterface
{
    /**
     * @var array
     */
    protected $config = [];
    
    /**
     * @var array
     */
    protected $keywordSearchAttributes = [];

    /**
     * @var string
     */
    protected $languageAttribute = null;

    /**
     * @var TenantInterface
     */
    protected $tenant = null;

    protected ?array $tenantsRaw = null;
    protected ?array $tenantUids = null;
      
    public function __construct(
        protected TypoScriptService $typoScriptService,
        protected ConfigurationManagerInterface $configurationManager,
        private readonly EventDispatcherInterface $eventDispatcher,
        protected Context $context
    ) {
        $all = $this->configurationManager->getConfiguration(
            ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        );
        $this->settings = $this->typoScriptService->convertTypoScriptArrayToPlainArray(
            $all['plugin.']['tx_igfibu_re.']['settings.'] ?? []
        );
        if (!empty($this->settings['objects'])) {
            foreach ($this->settings['objects'] as $tablename => $object) {
                $this->addConfig($tablename, $object);
            }
        }
    }


    public function getMandantConfig()
    {
        return $this->settings['mandant'] ?? [];
    }
    public function getDebitorConfig()
    {
        return $this->settings['debitor'] ?? [];
    }

    public function getAllConfig()
    {
        return $this->config;
    }

    public function addConfig($tablename, $object): void
    {
        if (isset($object['pageUid'])) {
            $object['pageUid'] = (int) $object['pageUid'];
        }
        $this->config[$tablename] = $object;
    }

    public function getConfig($tablename)
    {
        if (!$tablename) {
            $tablename = 'default';
        }
        if (isset($this->config[$tablename])) {
            return $this->config[$tablename];
        } elseif (isset($this->config['default'])) {
            return $this->config['default'];
        }
        return false;
    }

    public function getDebitorById(int $debitorUid, string $tablename = null)
    {
        if ($debitorUid <= 0) {
            return null;
        }
        if ($tablename !== null && $tablename !== 'auto') {
            $config = $this->getConfig($tablename);
        } else {
            $config = $this->getConfig('default');
        }
        if (isset($config['repository'])) {
            $repository = GeneralUtility::makeInstance($config['repository']);
            return $repository->findByUid($debitorUid);
        }
        return null;
    }

    
    public function getDebitorMethod($tablename)
    {
        $config = $this->getConfig($tablename);
        return $config['debitorMethod'] ?? '';
    }
    public function getModelClassName($tablename)
    {
        $config = $this->getConfig($tablename);
        return $config['model'];
    }
    public function getModelName($tablename)
    {
        $config = $this->getConfig($tablename);
        return substr(strrchr((string) $config['model'], '\\'), 1);
    }
    public function getRepositoryClassName($tablename)
    {
        $config = $this->getConfig($tablename);
        return $config['repository'];
    }
    public function getDebitorRepository(string $tablename = null)
    {
        $config = $this->getConfig($tablename);
        if (isset($config['repository'])) {
            $repository = GeneralUtility::makeInstance($config['repository']);
            return $repository;
        }
        return null;
    }
    public function getLink($tablename, $id, $verbandId = null, string $debitorType = null)
    {
        $config = $this->getConfig($tablename);
        if (!is_array($config)) {
            return [];
        }
        $modelClassName = $config['model'];
        $modelName = substr((string) $modelClassName, strrpos((string) $modelClassName, '\\') + 1);
        $argumentName = ($config['argumentName'] ?? '') ?: strtolower($modelName);
        $additionalArguments = [
            $argumentName => $id,
        ];

        if ($verbandId === null) {
            if (!empty($config['showArguments'])) {
                foreach ($config['showArguments'] as $name => &$value) {
                    $value = str_replace('{verband.uid}', '', $value);
                }
            }
        } else {
            $additionalArguments['activeTab'] = 'verband-' . $verbandId;
            if (!empty($config['showArguments'])) {
                foreach ($config['showArguments'] as $name => &$value) {
                    $value = str_replace('{verband.uid}', $verbandId, $value);
                }
            }
        }
        unset($value);
        return array_merge(
            $config,
            [
                'name' => strtolower($modelName),
                'arguments' => array_merge(
                    is_array($config['showArguments'] ?? false) ? $config['showArguments'] : [],
                    $additionalArguments
                ),
            ]
        );
    }
    public function getUriByDebitor(RequestInterface $request, string $action, $debitor, ?int $tenantId = null): ?string
    {
        return is_object($debitor) ? $this->getUriByDebitorId(
            $request,
            $action,
            $debitor->getType(),
            $debitor->getUid(),
            $tenantId
        ) : null;
    }
    
    public function getUriByDebitorId(
        RequestInterface $request,
        string $action,
        ?string $type,
        int $debitorId,
        ?int $tenantId = null
    ): ?string {
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);
        $uriBuilder
            ->reset()
            ->setRequest($request);
        
        $debitorLink = $this->getLink($type, $debitorId, $tenantId);
        if (empty($debitorLink)) {
            return null;
        }
        $arguments = $debitorLink['arguments'];
        if ($debitorLink['pageUid']) {
            $uriBuilder->setTargetPageUid($debitorLink['pageUid']);
        }
        return $uriBuilder->uriFor(
            'show',
            $arguments,
            $debitorLink['controller'],
            $debitorLink['extensionName'],
            $debitorLink['pluginName'],
        );
    }


    public function getTablenamesByFormat($referenceFormat)
    {
        foreach ($this->config as $tablename => $config) {
            if ($config['referenceFormat'] == $referenceFormat) {
                return $tablename;
            }
        }
        return '';
    }
    public function getSorting($tablename)
    {
        $config = $this->getConfig($tablename);
        // add uid to entries
        $sorting = [];
        if (!empty($config['sorting'])) {
            foreach ($config['sorting'] as $id => $sort) {
                $sorting[] = array_merge($sort, [
                    'uid' => $id,
                ]);
            }
        }
        return $sorting;
    }
    public function getSortingById($tablename, $id)
    {
        $config = $this->getConfig($tablename);
        return $config['sorting'][$id] ?? null;
    }
    public function getSortingAttributesById($tablename, $id)
    {
        $sorting = $this->getSortingById($tablename, $id);
        if ($sorting === null) {
            return null;
        }
        return GeneralUtility::trimExplode(',', $sorting['attributes']);
    }

    public function getKeywordSearchAttributes()
    {
        return $this->keywordSearchAttributes;
    }
    public function getDebitorKeywordConstraints($queryBuilder, $keyword)
    {
        $subConstraints = [];
        if (!empty($this->keywordSearchAttributes)) {
            foreach ($this->keywordSearchAttributes as $type => $attribute) {
                if ($type == 'concat') {
                    $concatAttributes = GeneralUtility::trimExplode(',', $attribute, true);
                    $concat = [];
                    foreach ($concatAttributes as $concatAttribute) {
                        if (str_starts_with($concatAttribute, '\'')) {
                            $concatAttribute = trim($concatAttribute, '\'');
                            $concat[] = $queryBuilder->createNamedParameter($concatAttribute);
                        } else {
                            $concat[] = $queryBuilder->quoteIdentifier($concatAttribute);
                        }
                    }
                    $subConstraints[] = 'concat(' . implode(
                        ',',
                        $concat
                    ) . ') LIKE ' . $queryBuilder->createNamedParameter('%' . $keyword . '%');
                    /*
                    // risky and probably we don't want that
                    } elseif ($type == 'function') {
                    $subConstraints[] = $attribute . ' LIKE ' . $queryBuilder->createNamedParameter('%' . $keyword . '%');
                    */
                } else {
                    $subConstraints[] = $queryBuilder->expr()->like(
                        $attribute,
                        $queryBuilder->createNamedParameter('%' . $keyword . '%')
                    );
                }
            }
            $subConstraints[] = $queryBuilder->expr()->eq(
                $this->getEntryTable() . '.' . $this->getEntryPrimaryKey(),
                $queryBuilder->createNamedParameter($keyword, Connection::PARAM_INT)
            );
        }
        return $subConstraints;
    }
    
    public function setKeywordSearchAttributes($keywordSearchAttributes): void
    {
        if (empty($keywordSearchAttributes)) {
            $config = $this->settings['entryTable'];
            if ($config['keywordSearchAttributes']) {
                $keywordSearchAttributes = $config['keywordSearchAttributes'];
                //$keywordSearchAttributes = GeneralUtility::trimExplode(',', $config['keywordSearchAttributes'], true);
            }
        }
        $this->keywordSearchAttributes = $keywordSearchAttributes;
        //var_dump($this->keywordSearchAttributes);exit(0);
    }
    public function getEntryTable()
    {
        return $this->settings['entryTable']['tablename'] ?? null;
    }
    public function getEntryPrimaryKey()
    {
        return $this->settings['entryTable']['primaryKey'] ?? null;
    }
    public function getEntryName()
    {
        return $this->settings['entryTable']['name'] ?? null;
    }
    public function getEntryNumberName()
    {
        $entryNumberName = $this->settings['entryTable']['number'] ?? '';
        if (!$entryNumberName) {
            $entryNumberName = $this->settings['entryTable']['tablename'] ?? '';
            if ($entryNumberName) {
                $entryNumberName .= '.';
            }
            $entryNumberName .= $this->settings['entryTable']['primaryKey'] ?? 'uid';
        }
        return $entryNumberName;
    }

    public function getEntryDeliveryMethods()
    {
        return $this->settings['entryTable']['deliveryMethods'] ?? [];
    }
    public function getLanguageAttribute()
    {
        return $this->languageAttribute;
    }
    public function setLanguageAttribute($languageAttribute)
    {
        return $this->languageAttribute = $languageAttribute;
    }

    public function getMandants(bool $returnRawQueryResult = false)
    {
        if (!($this->settings['mandant']['active'] ?? false)) {
            return [];
        }
        if ($returnRawQueryResult === true && $this->tenantsRaw !== null) {
            return $this->tenantsRaw;
        }
        $repositoryClassName = $this->settings['mandant']['repository'];
        if (!$repositoryClassName) {
            return [];
        }
        $repository = GeneralUtility::makeInstance($repositoryClassName);
        $tenants = $repository->findAllWithAcl($returnRawQueryResult);
        if ($returnRawQueryResult === true) {
            $this->tenantsRaw = $tenants;
        }
        return $tenants;
    }

    public function getMandantUids(): array
    {
        if ($this->tenantUids === null) {
            $this->tenantUids = [];
            $tenants = $this->getMandants(true);
            if (!empty($tenants)) {
                foreach ($tenants as $tenant) {
                    $this->tenantUids[] = $tenant['uid'];
                }
            }
        }
        return $this->tenantUids;
    }


    public function getMandantByUid(?int $uid)
    {
        if (!($this->settings['mandant']['active'] ?? false)) {
            if ($this->tenant === null && is_array($this->settings['mandant']['default'] ?? false)) {
                $this->tenant = GeneralUtility::makeInstance(Creditor::class);
                $this->tenant->setArray($this->settings['mandant']['default']);
            }

            return $this->tenant;
        }
        if ((int)$uid == 0) {
            return false;
        }
        $repositoryClassName = $this->settings['mandant']['repository'];
        if (!$repositoryClassName) {
            return [];
        }
        $repository = GeneralUtility::makeInstance($repositoryClassName);
        return $repository->findByUid($uid);
    }
    public function hasPermission(FibuAclInterface $fibuAclObject)
    {
        $tenantUid = $fibuAclObject->getTenantId();
        $allowedTenantUids = $this->getMandantUids();
        return $tenantUid === 0 || in_array($tenantUid, $allowedTenantUids);
    }

    public function userRestriction(QueryBuilder $queryBuilder, string $table)
    {
        return $this->eventDispatcher
            ->dispatch(new UserRestrictionEvent($queryBuilder))
            ->getConstraints();
    }
    public function hasGroup(string $name): bool
    {
        return in_array($name, $this->context->getPropertyFromAspect('frontend.user', 'groupNames'));
    }
    
    public function isAdmin(): bool
    {
        return $this->hasGroup('fe_admin');
    }
    public function getAcl(): array
    {
        $perm = $this->hasGroup('fe_admin') || $this->hasGroup('fe_fibu');
        return [
            'igFibu' => [
                'edit' => $perm,
                'showAll' => $perm,
            ],
        ];
    }
}
