<?php

namespace Ig\IgFibu\Controller;

use DateTime;
use Ig\IgFibu\Domain\Model\Invoice;
use Ig\IgFibu\Domain\Model\Payment;
use Ig\IgFibu\Domain\Model\PaymentImport;
use Ig\IgFibu\Domain\Repository\InvoiceRepository;
use Ig\IgFibu\Domain\Repository\InvoicestatusRepository;
use Ig\IgFibu\Domain\Repository\PaymentImportRepository;
use Ig\IgFibu\Domain\Repository\PaymentRepository;
use Ig\IgFibu\Service\BookingService;
use Ig\IgFibu\Service\DebitorService;
use Ig\IgFibu\Utility\ConfUtility;
use Ig\IgFibu\Utility\XmlImportUtility;
use Internetgalerie\IgDoctrinePaginator\Pagination\DoctrinePaginator;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Messaging\FlashMessage;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Type\ContextualFeedbackSeverity;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Annotation\IgnoreValidation;
use TYPO3\CMS\Extbase\Http\ForwardResponse;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use TYPO3\CMS\Frontend\Controller\ErrorController;

use ZipArchive;

class PaymentController extends AbstractFibuController
{
    protected $objectNames = ['payment', 'newPayment'];
    protected $dateObjects = ['datePay', 'dateValuta', 'dateImport'];
    protected $dateTimeObjects = [];
    protected $searchDefault = [
        'status' => '0',
    ];

    // folder to save the uploaded file or null to deactivate the featrue
    //protected $pathForFileUploads = 'fileadmin_secure/ig_fibu';

    /**
     * debitorService
     *
     * @var DebitorService
     */
    protected $debitorService = null;

    /**
     * paymentRepository
     *
     * @var PaymentRepository
     */
    protected $paymentRepository = null;

    /**
     * invoiceRepository
     *
     * @var InvoiceRepository
     */
    protected $invoiceRepository = null;

    /**
     * paymentImportRepository
     *
     * @var PaymentImportRepository
     */
    protected $paymentImportRepository = null;

    /**
     * flashMessageIdentifier
     *
     * @var string
     */
    protected $flashMessageIdentifier = 'extbase.flashmessages.tx_igcrmadmin_re';

    
    public function injectDebitorService(DebitorService $debitorService): void
    {
        $this->debitorService = $debitorService;
    }

    
    public function injectPaymentRepository(PaymentRepository $paymentRepository): void
    {
        $this->paymentRepository = $paymentRepository;
    }

    
    public function injectInvoiceRepository(InvoiceRepository $invoiceRepository): void
    {
        $this->invoiceRepository = $invoiceRepository;
    }

    
    public function injectPaymentImportRepository(PaymentImportRepository $paymentImportRepository): void
    {
        $this->paymentImportRepository = $paymentImportRepository;
    }
    
    public function initializeAction(): void
    {
        parent::initializeAction();
        $this->search = $this->getSearch();
        $this->control = $this->getControl();
    }

    /**
     * action list
     */
    public function listAction(): ResponseInterface
    {
        $this->invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);
        $invoicestatus = $this->invoicestatusRepository->findAll();

        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/tableSelectToggle.js');

        $this->assignTotals();

        $this->view->assign('lineBreak', "\n");

        $currentPage = $this->request->hasArgument('currentPage') ? (int)$this->request->getArgument('currentPage') : 1;

        $paymentQueryBuilder = $this->paymentRepository->findBySearch($this->search);
        $paginator = new DoctrinePaginator($paymentQueryBuilder, $currentPage, 50, Payment::class);
        $pagination = $paginator->createSlidingWindowPagination(10);

        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);

        $this->view->assign('invoicestatus', $invoicestatus);
        $this->view->assign('paymentsStatus', ConfUtility::$paymentsStatus);

        $paymentImportQueryBuilder = $this->paymentImportRepository->findBySearch([]);
        $paymentImports = $paymentImportQueryBuilder->executeQuery()
->fetchAllAssociative();
        $this->view->assign('paymentImports', $paymentImports);

        $verbands = [];
        if ($this->settings['mandant']['active']) {
            $this->verbandRepository = GeneralUtility::makeInstance($this->settings['mandant']['repository']);
            $verbands = $this->verbandRepository->findAllWithAcl(true);
        }
        //foreach ($verbands as $verband) { var_dump($verband);}exit(0);
        $this->view->assign('verbands', $verbands);

        //$this->view->assign('payments', $payments);
        $this->view->assign('search', $this->search);
        
        $advancedBoxIsOpen = $this->hasAdvancedSearch(
            $this->search,
            ['status', 'keyword', 'output', 'verband', 'tenant']
        );
        //$advancedBoxIsOpen = $this->search['import'] ?? false || $this->search['debitorUid'] ?? false  || $this->search['invoiceUid']['equal'] ?? false || $this->search['paymentUid']['equal'] ?? false;
        //var_dump($advancedBoxIsOpen, $this->search);exit(0);

        $this->view->assign('advancedBoxIsOpen', $advancedBoxIsOpen);
        
        
        $this->view->assign('control', $this->control);
        if ($this->control['invoice'] ?? false) {
            $this->view->assign('invoice', $this->invoiceRepository->findByUid((int)$this->control['invoice']));
        }
        //$this->view->assign('anzahlPayments', count($payments));
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action show
     */
    public function showAction(Payment $payment): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($payment)) {
            return $this->htmlResponse('Access Denied');
        }
        $activeTab = $this->request->hasArgument('activeTab') ? $this->request->getArgument('activeTab') : '';
        $this->view->assign('activeTab', $activeTab);
        $this->view->assign('lineBreak', "\n");
        $this->view->assign('payment', $payment);
        $openInvoices = [];
        $referencedInvoices = [];
        $relatedInvoices = [];
        if (!$payment->getIsDone()) {
            $invoice = $payment->getInvoice();
            $tenantId = $payment->getTenantId();
            $debitorUid = $payment->getDebitorId();
            if ($debitorUid > 0 && ($invoice === null || $invoice->getIsPaid())) {
                $openInvoices = $this->invoiceRepository->findOpenByDebitor($tenantId, $debitorUid);
            }
            if ($invoice !== null && !$invoice->getIsPaid()) {
                $referencedInvoices[] = $this->invoiceRepository->findByUid($invoice->getUid());
            }
            if (empty($openInvoices) || 1) {
                $debitor = $payment->getDebitor();
                if (is_object($debitor) && method_exists($debitor, 'getChildren')) {
                    $children = $debitor->getChildren();
                    if (!empty($children)) {
                        foreach ($children as $child) {
                            if ($debitor !== $child) {
                                $relatedInvoices = array_merge(
                                    $relatedInvoices,
                                    $this->invoiceRepository->findOpenByDebitor($tenantId, $child->getUid())
                                );
                            }
                        }
                    }
                }
                if (is_object($debitor) && method_exists($debitor, 'getParent')) {
                    $parent = $debitor->getParent();
                    if (is_object($parent) && $debitor !== $parent) {
                        $relatedInvoices = array_merge(
                            $relatedInvoices,
                            $this->invoiceRepository->findOpenByDebitor($tenantId, $parent->getUid())
                        );
                        if (method_exists($parent, 'getChildren')) {
                            $children = $parent->getChildren();
                            if (!empty($children)) {
                                foreach ($children as $child) {
                                    if ($child !== $debitor && $child !== $parent) {
                                        $openRelated = $this->invoiceRepository->findOpenByDebitor(
                                            $tenantId,
                                            $child->getUid()
                                        );
                                        $relatedInvoices = array_merge($relatedInvoices, $openRelated);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        $this->view->assign('openInvoices', $openInvoices);
        $this->view->assign('referencedInvoices', $referencedInvoices);
        $this->view->assign('relatedInvoices', $relatedInvoices);
        return $this->htmlResponse($this->view->render());
    }


    /**
     * action new
     */
    public function newAction(): ResponseInterface
    {
        $this->assignEditFormfields();
                
        if ($this->request->hasArgument('data')) {
            $data = $this->request->getArgument('data');
            $payment = GeneralUtility::makeInstance(Payment::class);
            $amount = (float)($data['amount'] ?? $data['betrag'] ?? 0);
            $payment->setAmount($amount);
            if ($data['verband'] ?? 0) {
                $payment->setVerbandId((int) $data['verband']);
            }
            if ($data['debitorId'] ?? 0) {
                $payment->setDebitorId((int)$data['debitorId']);
            }
            if ($data['invoice'] ?? 0) {
                $invoice = $this->invoiceRepository->findByUid((int) $data['invoice']);
                $payment->setInvoice($invoice);
            }
            $payment->setDatePay(new DateTime());
            $payment->setDateValuta(new DateTime());

            $this->view->assign('newPayment', $payment);
        }
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action create
     */
    public function createAction(Payment $newPayment): ResponseInterface
    {
        $message = GeneralUtility::makeInstance(
            FlashMessage::class,
            'Zahlung wurde erstellt.',
            '',
            ContextualFeedbackSeverity::OK,
            true
        );
        $this->flashMessageService->getMessageQueueByIdentifier($this->flashMessageIdentifier)
->enqueue($message);

        $this->paymentRepository->add($newPayment);
        $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
        $persistenceManager->persistAll();
        $invoice = $newPayment->getInvoice();
        if ($invoice) {
            return $this->redirectToInvoiceShow($invoice);
        }
        $arguments = [
            'payment' => $newPayment->getUid(),
        ];
        if ($this->request->hasArgument('activeTab')) {
            $arguments['activeTab'] = $this->request->getArgument('activeTab');
        }

        return (new ForwardResponse('show'))->withArguments($arguments);
        //$this->redirect('list');
    }

    /**
     * action edit
     */
    #[IgnoreValidation([])] // $payment
    public function editAction(Payment $payment): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($payment)) {
            return $this->htmlResponse('Access Denied');
        }
        $this->assignEditFormfields();
        $this->view->assign('payment', $payment);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action update
     */
    public function updateAction(Payment $payment): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($payment)) {
            return $this->htmlResponse('Access Denied');
        }
        $message = GeneralUtility::makeInstance(
            FlashMessage::class,
            'Zahlung wurde aktualisiert..',
            '',
            ContextualFeedbackSeverity::OK,
            true
        );
        $this->flashMessageService->getMessageQueueByIdentifier($this->flashMessageIdentifier)
->enqueue($message);
        $this->paymentRepository->update($payment);
        $invoice = $payment->getInvoice();
        $this->redirectToInvoiceShow($invoice);
        return $this->redirect('list');
    }

    /**
     * action delete
     */
    public function deleteAction(Payment $payment): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($payment)) {
            return $this->htmlResponse('Access Denied');
        }
        $message = GeneralUtility::makeInstance(
            FlashMessage::class,
            'Zahlung CHF ' . number_format($payment->getBetrag(), 2) . ' wurde gelöscht.',
            '',
            ContextualFeedbackSeverity::OK,
            true
        );
        $this->flashMessageService->getMessageQueueByIdentifier($this->flashMessageIdentifier)
->enqueue($message);


        $invoice = $payment->getInvoice();
        $this->paymentRepository->remove($payment);
        $this->redirectToInvoiceShow($invoice);
        //$this->flashMessageService->getMessageQueueByIdentifier()->enqueue($message);

        return $this->redirect('list');
    }

    public function multipleDeleteAction(array $multipleDelete): ResponseInterface
    {
        $message = GeneralUtility::makeInstance(
            FlashMessage::class,
            'Zahlungen wurden gelöscht.',
            '',
            ContextualFeedbackSeverity::OK,
            true
        );
        $this->flashMessageService->getMessageQueueByIdentifier($this->flashMessageIdentifier)
->enqueue($message);
        // update or delete? + how do i get search in here?
        $this->paymentRepository->findByUidAndDelete($multipleDelete);
        return (new ForwardResponse('list'))->withArguments([
            'search' => $this->search,
        ]);
    }


    public function importXmlFile(string $xmlFilePath, int $verbandId)
    {
        if (str_contains($xmlFilePath, '..') || ($xmlStr = file_get_contents($xmlFilePath)) === false) {
            return null;
        }
        unlink($xmlFilePath);
        return $this->XmlImportUtility->getPaymentsFromXml($xmlStr, $verbandId);
        //array_push($payments, ...$newPayments);
        //echo $fileCount .') '. $xmlFile . '='.count($newPayments).'<br />';
    }

    
    /**
     * Action: upload payment file
     */
    public function uploadAction(): ResponseInterface
    {
        $this->XmlImportUtility = GeneralUtility::makeInstance(XmlImportUtility::class);
        $fileCount = 0;
        $sourceExt = null;
        $payments = [];
        $totalAnzahl = 0;
        $totalBetrag = 0;
        $info = [];
        $paymentimport = null;
        $importFileArray = [];
        $uploadedFiles = $this->request->getUploadedFiles();
        // for v11
        if (isset($uploadedFiles['tx_igfibu_za']) || isset($uploadedFiles['tx_igcrmadmin_za'])) {
            $uploadedFiles = $uploadedFiles['tx_igfibu_za'] ?? $uploadedFiles['tx_igcrmadmin_za'];
            if ($this->request->hasArgument('importFile')) {
                $importFileArray = $this->request->getArgument('importFile');
            }
        }
        if (isset($uploadedFiles['importFile'])) {
            $importFile = $uploadedFiles['importFile'];
        } else {
            $this->view->assign('message', 'Keine Datei zum Import vorhanden');
            return $this->htmlResponse($this->view->render());
        }
        /*
        if ($this->request->hasArgument('importFile')) {
            $importFile = $this->request->getArgument('importFile');
        } else {
            $this->view->assign('message', 'Keine Datei zum Import vorhanden');
            return $this->htmlResponse($this->view->render());
        }
        if ($importFile['error']) {
            $this->view->assign('message', 'Keine gültige Datei zum Import');
            return $this->htmlResponse($this->view->render());
        }
        */
        if ($importFile->getError()) {
            $this->view->assign('message', 'Keine gültige Datei zum Import');
            return $this->htmlResponse($this->view->render());
        }
        $verbandId = $this->request->hasArgument('verband') ? (int)$this->request->getArgument('verband') : null;

        
        //$fileIn = $importFile['tmp_name'];
        //$fileOrigname = $importFile['name'];
        if (isset($importFileArray['tmp_name'])) {
            $fileIn = $importFileArray['tmp_name'];
        } else {
            $fileIn = $importFile->getTemporaryFileName();
        }
        $fileOrigname = $importFile->getClientFilename();
        $ext = end((explode('.', (string) $fileOrigname)));

        $mimeType = $importFile->getClientMediaType(); //$importFile['type']
        
        if ($mimeType == 'application/zip' || $mimeType == 'application/x-zip-compressed' || $ext == 'zip') {
            $sourceExt = 'zip';
            $extractToPath = Environment::getPublicPath() . '/typo3temp/ig_fibu/';
            if (!is_dir($extractToPath)) {
                GeneralUtility::mkdir_deep($extractToPath);
            }

            $zip = new ZipArchive();
            $zip->open($fileIn);
            if ($zip->extractTo($extractToPath)) {
                $zip->close();
                //unlink($target_path);
                $fdir = opendir($extractToPath);
                while (($file = readdir($fdir))) {
                    $dirPath = $extractToPath . '/' . $file;
                    if ($file != '.' && $file != '..') {
                        if (is_dir($dirPath)) {
                            $fsubdir = opendir($dirPath . '/');
                            while (($xmlFile = readdir($fsubdir))) {
                                if ($xmlFile != '.' && $xmlFile != '..') {
                                    $newPayments = $this->importXmlFile($dirPath . '/' . $xmlFile, $verbandId);
                                    $fileCount++;
                                    if ($newPayments === null) {
                                        return $this->htmlResponse('File error');
                                    }
                                    array_push($payments, ...$newPayments);
                                }
                            }
                        } else {
                            $newPayments = $this->importXmlFile($dirPath, $xmlFile);
                            $fileCount++;
                            if ($newPayments === null) {
                                return $this->htmlResponse('File error');
                            }
                            array_push($payments, ...$newPayments);
                        }
                    }
                }
            }
        } elseif ($mimeType == 'text/xml' || $ext == 'xml') {
            $sourceExt = 'xml';
            if (($xmlStr = file_get_contents($fileIn)) === false) {
                return $this->htmlResponse('File error');
            }
            $fileCount++;
            $info = $this->XmlImportUtility->getInfoFromXml($xmlStr);
            $paymentimport = $this->paymentImportRepository->findByInfo($info, $verbandId);
            $payments = $this->XmlImportUtility->getPaymentsFromXml($xmlStr, $verbandId);
        } else {
            $this->view->assign(
                'message',
                'File "' . $fileOrigname . ' ": Unsupported File Type "' . $mimeType . '" (Extension: ' . $ext . ')'
            );
            return $this->htmlResponse('');
        }
        foreach ($payments as $payment) {
            $totalAnzahl++;
            $totalBetrag += $payment->getBetrag();
        }
        
        if ($paymentimport !== null) {
            $messageText = 'Diese Datei mit (MsgId: ' . $paymentimport->getMsgId() . ', Ntfctn ID: ' . $paymentimport->getNtfctnId() . ') wurde bereits am ' . $paymentimport->getCreateDate()->format(
                'd.m.Y H:i:s'
            ) . ' importiert (';
            if ($paymentimport->getTenantId()) {
                $messageText .= 'Mandant ID: ' . $paymentimport->getTenantId() . ', ';
            }
            $messageText .= 'Total Betrag: CHF ' . $totalBetrag . ', Einträge: ' . count($payments) . ')';
            $severity = ContextualFeedbackSeverity::ERROR;
        } else {
            $paymentimport = GeneralUtility::makeInstance(PaymentImport::class);
            $paymentimport->setFilename($fileOrigname);
            //$paymentimport->setCreateDate();
            if ($verbandId !== null) {
                $paymentimport->setVerbandId($verbandId);
            }
            if ($info['msgId']) {
                $paymentimport->setMsgId($info['msgId']);
            }
            if (!empty($info['ntfctn'])) {
                $paymentimport->setNtfctnId($info['ntfctn']['id'] ?? '');
                $paymentimport->setAccountIban($info['ntfctn']['accountIban'] ?? '');
            }
            

            $paymentimport->setFilesCount($fileCount);
            $paymentimport->setEntriesCount($totalAnzahl);
            $paymentimport->setEntriesBetrag($totalBetrag);
            $this->paymentImportRepository->add($paymentimport);

            foreach ($payments as $payment) {
                $payment->setImport($paymentimport);
                $this->paymentRepository->add($payment);
            }

            $persistenceManager = GeneralUtility::makeInstance(PersistenceManager::class);
            $persistenceManager->persistAll();


            if (PaymentImport::$pathForFileUploads !== null) {
                $uid = $paymentimport->getUid();

                $absolutePathForFileUploads = Environment::getPublicPath() . '/' . PaymentImport::$pathForFileUploads;
                $uploadPath = $absolutePathForFileUploads . '/' . $uid . '/';
                if (!is_dir($uploadPath)) {
                    GeneralUtility::mkdir_deep($uploadPath);
                }
                copy($fileIn, $uploadPath . $fileOrigname);
            }

            $messageText = 'Total Betrag:  CHF ' . number_format(
                $totalBetrag,
                2,
                '.',
                "'"
            ) . ', Zahlungen importiert: ' . $totalAnzahl;
            if ($sourceExt == 'zip') {
                $messageText .= ', XML-Files in zip: ' . $fileCount;
            }
            $severity = ContextualFeedbackSeverity::OK;
        }
        $message = GeneralUtility::makeInstance(FlashMessage::class, $messageText, '', $severity, true);
        $this->flashMessageService->getMessageQueueByIdentifier($this->flashMessageIdentifier)
->enqueue($message);
        return new ForwardResponse('list');
    }


    /**
     * action: book all open payments with open invoices (Hochgeladene/eingelesene Zahlungen verbuchen)
     */
    public function bookAllAction(): ResponseInterface
    {
        $test = false; // falls true wird nichts verbucht

        //$this->invoicestatusRepository = GeneralUtility::makeInstance(InvoicestatusRepository::class);
        //$invoicestatusPaid = $this->invoicestatusRepository->findByUid(intval($this->settings['invoicesStatus']['paid']));

        $bookingService = GeneralUtility::makeInstance(BookingService::class);
        $bookingService->setSettings($this->settings);
        $useBookRule = $this->request->hasArgument('useBookRule') ? $this->request->getArgument('useBookRule') : '';
        $bookingService->setUseBookRule($useBookRule);


        // $useBookByDebitor = $this->request->hasArgument('useBookByDebitor') ? (bool)$this->request->getArgument(
        //     'useBookByDebitor'
        // ) : false;
        // $useBookOverpaid = $this->request->hasArgument('useBookOverpaid') ? (bool)$this->request->getArgument(
        //     'useBookOverpaid'
        // ) : false;
        // $bookingService->setUseBookOverpaid($useBookOverpaid);
        // $useCreateCredit = $this->request->hasArgument('useCreateCredit') ? (bool)$this->request->getArgument(
        //     'useCreateCredit'
        // ) : false;
        // $bookingService->setUseCreateCredit($useCreateCredit);
        $anz_ok = 0;
        $anz_betrag_falsch = 0;
        $anz_error = 0;

        $tenantId = (int) ($this->search['verband'] ?? 0);
        
        $payments = $this->paymentRepository->findForBooking($tenantId);
        foreach ($payments as $payment) {
            $bookingService->book($payment);
        }
        $this->view->assign('count', $bookingService->getStatistic());

        // invoices mit manuellen Zahlungen als bezahlt markieren
        $invoices = $this->invoiceRepository->findOpenSettled($tenantId);
        $invoiceSettled = 0;
        if (!empty($invoices)) {
            $this->invoiceRepository->closeOpenSettled($tenantId);
            $invoiceSettled = count($this->invoices);
        }
        $this->view->assign('invoiceSettled', $invoiceSettled);
        $this->view->assign('search', $this->search);

        $pageUid = $this->request->getAttribute('frontend.page.information')?->getId() ?? $GLOBALS['TSFE']->id;
        $this->view->assign('pageUid', $pageUid);
        return $this->htmlResponse($this->view->render());
    }




    /**
     * action book the payment with invoice
     */
    public function bookAction(Invoice $invoice, Payment $payment): ResponseInterface
    {
        if (!$this->debitorService->hasPermission($payment) || !$this->debitorService->hasPermission($invoice)) {
            return $this->htmlResponse('Access Denied');
        }

        $bookingService = GeneralUtility::makeInstance(BookingService::class);
        $bookingService->setSettings($this->settings);
        $booked = $bookingService->bookPaymentToInvoice($payment, $invoice);

        if ($booked) {
            $message = GeneralUtility::makeInstance(
                FlashMessage::class,
                'Zahlung wurde verbucht.',
                '',
                ContextualFeedbackSeverity::OK,
                true
            );
        } else {
            $message = GeneralUtility::makeInstance(
                FlashMessage::class,
                'Zahlung kann nicht verbucht werden.',
                '',
                ContextualFeedbackSeverity::ERROR,
                true
            );
        }
        $this->flashMessageService->getMessageQueueByIdentifier($this->flashMessageIdentifier)
->enqueue($message);

        $redirect = $this->request->hasArgument('redirect') ? $this->request->getArgument('redirect') : 'invoice';
        if ($redirect === 'payment') {
            return $this->redirectToPaymentShow($payment, 'payments');
        }
        return $this->redirectToInvoiceShow($invoice, 'payments');
    }


    /**
     * assign for edit/new form
     */
    public function assignEditFormfields(): void
    {
        $pageRenderer = GeneralUtility::makeInstance(PageRenderer::class);
        $pageRenderer->addJsFooterFile('EXT:ig_fibu/Resources/Public/JavaScript/edit_invoice.js');

        $debitorSettings = $this->debitorService->getConfig('default');
        //var_dump($debitorSettings);exit(0);
        $this->view->assign('debitorSettings', $debitorSettings);
        $this->view->assign('mandants', $this->debitorService->getMandants());
        
        if ($this->request->hasArgument('activeTab')) {
            $this->view->assign('activeTab', $this->request->getArgument('activeTab'));
        }
    }
    public function assignTotals(): void
    {
        $totals = $this->paymentRepository->findTotal($this->search);
        $this->view->assign('totals', $totals);
    }


    protected function getSearch()
    {
        $search = [];
        if ($this->request->hasArgument('@search')) {
            $search = $this->request->getArgument('@search');
        } elseif ($this->request->hasArgument('search')) {
            $search = $this->request->getArgument('search');
        } else {
            $search = $this->searchDefault;
        }

        return $search;
    }

    protected function getControl()
    {
        return $this->request->hasArgument('control') ? $this->request->getArgument('control') : [];
    }

    protected function redirectToInvoiceShow($invoice = null, string $activeTab = 'payments'): ResponseInterface
    {
        if (is_object($invoice)) {
            $this->uriBuilder
                ->reset()
                ->setRequest($this->request)
                ->setTargetPageUid((int)$this->settings['invoicePid']);
            $uri = $this->uriBuilder->uriFor(
                'show',
                [
                    'invoice' => $invoice->getUid(),
                    'activeTab' => $activeTab,
                ],
                'Invoice',
                null,
                'Re'
            );
            return $this->redirectToUri($uri);
        }
        return GeneralUtility::makeInstance(ErrorController::class)->pageNotFoundAction(
            $GLOBALS['TYPO3_REQUEST'],
            'The requested page needs an invoice'
        );
    }
    protected function redirectToPaymentShow($payment = null, string $activeTab = 'payments'): ResponseInterface
    {
        if (is_object($payment)) {
            $this->uriBuilder
                ->reset()
                ->setRequest($this->request)
                ->setTargetPageUid((int)$this->settings['paymentPid']);
            $uri = $this->uriBuilder->uriFor(
                'show',
                [
                    'payment' => $payment->getUid(),
                    'activeTab' => $activeTab,
                ],
                'Payment',
                null,
                'Za'
            );
            return $this->redirectToUri($uri);
        }
        return GeneralUtility::makeInstance(ErrorController::class)->pageNotFoundAction(
            $GLOBALS['TYPO3_REQUEST'],
            'The requested page needs a payment'
        );
    }
}
