class PdfToLightBoxConverter {
    constructor({ container, source }) {
        this.triggerElement = container;
        this.pdfSourceUrl = source;
        this.pdfDocument = null;
        this.totalPages = 0;
        this.uniqueId = "0";
        this.isReady = false;
        this.lightboxInstance = null;
        this.renderFormat = "image/jpeg";
        this.renderQuality = 1;
        this.renderScale = 3;
        this.useCanvas = true;
        this.initializationPromise = this.initializeProcess();
        this.lightboxContainer = null;
        this.slides = null;
        this.thumbnails = null;
        this.pagesCache = [];
        this.loadingOffset = Math.min(20, Math.floor(window.innerWidth / 2 / 80));
    }
    async initializeProcess() {
        try {
            await this.loadPdfDocument();
            // Setup the lightbox component
            this.prepareLightbox();
            // Render first page
            await this.renderPages(1, 1);
            // open lightbox
            this.open();
            this.renderPages();
        } catch (error) {
            this.handleInitializationError(error);
        }
    }

    /**
     * Fetches the PDF data and initializes the PDF document object.
     */
    async loadPdfDocument() {
        const fetchResponse = await fetch(this.pdfSourceUrl);
        if (!fetchResponse.ok) {
            throw new Error(`Failed to fetch PDF: ${fetchResponse.statusText}`);
        }

        const arrayBuffer = await fetchResponse.arrayBuffer();
        const fileArray = new Uint8Array(arrayBuffer);

        // Use a clearer check for pdfjsLib availability
        if (typeof pdfjsLib === "undefined" || !pdfjsLib.getDocument) {
            throw new Error("PDF viewer library (pdfjsLib) is not available after loading modules.");
        }

        this.pdfDocument = await pdfjsLib.getDocument({
            data: fileArray,
            useSystemFonts: true,
        }).promise;
        this.uniqueId = this.createTimeBasedId();
        this.totalPages = this.pdfDocument.numPages;

        console.log(`PDF Loaded: ${this.totalPages} pages.`);
    }

    /**
     * Creates placeholder elements and initializes the lightbox instance.
     */
    prepareLightbox() {
        if (!IgLightboxConfig) {
            throw new Error('Global configuration "IgLightboxConfig" is missing.');
        }

        this.pagePlaceholders = Array.from({ length: this.totalPages }, () => {
            const link = document.createElement("a");
            link.setAttribute("href", "data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7");
            return link;
        });
        this.pagePlaceholders.forEach((page) => {
            this.pagesCache.push({
                element: page,
                loaded: false,
            });
        });
        const bookConfig = {
            ...IgLightboxConfig,
            ...{
                id: this.uniqueId,
                sources: this.pagePlaceholders,
                downloadTarget: this.pdfSourceUrl,
                download: true,
                book: true,
                preload: this.loadingOffset,
                loop: false,
                visibleSlidesAmount: 8,
            },
        };
        this.lightboxInstance = new IgLightbox(bookConfig);
        this.lightboxInstance.open();
        this.lightboxContainer = document.querySelector(`.ig-lightbox-container[data-id="${this.uniqueId}"]`);
        this.slides = this.lightboxContainer.querySelector(".ig-lightbox-slides");
        this.thumbnails = this.lightboxContainer.querySelector(".ig-lightbox-thumbnails");
        this.thumbnails?.classList.add("is-loading");
        this.lightboxContainer.addEventListener("lightbox:goTo", (event) => {
            let currentSlideIndex = event.detail.current.index + 1;
            this.renderPages(currentSlideIndex, this.loadingOffset);
        });
        this.lightboxContainer.addEventListener("lightbox:open", (event) => {
            this.triggerElement.classList.remove("loading");
        });
    }

    /**
     * Renders all PDF pages concurrently.
     */
    async renderPages(centerPage = 1, offset = this.loadingOffset) {
        let page = centerPage;
        const renderPromises = [];
        for (let i = page; i <= page + offset; i++) {
            if (this.pagesCache[i - 1] && !this.pagesCache[i - 1].loaded) {
                renderPromises.push(this.renderPage(i));
            }
        }
        await Promise.allSettled(renderPromises.map((promise, index) => promise.catch((error) => console.error(`Failed to render page ${index + 1}:`, error))));
    }

    /**
     * Renders a single PDF page into a Canvas or Data URL.
     * @param {number} pageNumber - The 1-based index of the page.
     * @returns {Promise<HTMLElement | string>} The Canvas element or Data URL.
     */
    async renderPage(pageNumber) {
        const page = await this.pdfDocument.getPage(pageNumber);
        this.pagesCache[pageNumber - 1].loaded = true;

        // --- MAIN RENDER LOGIC ---
        const viewport = page.getViewport({ scale: this.renderScale });
        const canvas = document.createElement("canvas");
        const ctx = canvas.getContext("2d");

        canvas.width = viewport.width;
        canvas.height = viewport.height;

        // Render the high-res version
        const renderTask = page.render({
            canvasContext: ctx,
            viewport: viewport,
        });
        await renderTask.promise;

        // --- THUMBNAIL RENDER LOGIC (100px width) ---
        const thumbScale = 100 / (viewport.width / this.renderScale);
        const thumbViewport = page.getViewport({ scale: thumbScale.toFixed(2) });

        const thumbCanvas = document.createElement("canvas");
        const thumbCtx = thumbCanvas.getContext("2d", { alpha: false });

        thumbCanvas.width = 100;
        thumbCanvas.height = thumbViewport.height;

        // Render the low-res thumbnail
        const thumbRenderTask = page.render({
            canvasContext: thumbCtx,
            viewport: thumbViewport,
        });
        await thumbRenderTask.promise;
        this.updateLightboxDom(pageNumber, canvas, thumbCanvas);

        page.cleanup();

        return canvas;
    }

    /**
     * Finds the corresponding slide/thumbnail elements and replaces the placeholder.
     * @param {number} pageNumber - The 1-based index of the page.
     * @param {HTMLCanvasElement} canvas - The rendered canvas element.
     */
    updateLightboxDom(pageNumber, canvas, thumbCanvas) {
        const slideIndex = pageNumber - 1;

        if (!this.lightboxContainer) {
            console.warn(`Lightbox container with ID ${this.uniqueId} not found for page ${pageNumber}.`);
            return;
        }
        const slideContainer = this.slides?.querySelector(`[data-index="${slideIndex}"]`);
        const thumbnailContainer = this.thumbnails?.querySelector(`[data-index="${slideIndex}"]`);

        if (slideContainer) {
            const slideImg = slideContainer.firstElementChild;
            thumbnailContainer?.classList.add("is-loaded");
            slideContainer.replaceChild(canvas, slideImg);
            if (pageNumber > 1) {
                const shadow = document.createElement("span");
                shadow.style.aspectRatio = canvas.width / canvas.height;
                slideContainer.appendChild(shadow);
            }
        }

        if (thumbnailContainer) {
            const thumbnailImg = thumbnailContainer.firstElementChild;
            thumbnailContainer.replaceChild(thumbCanvas, thumbnailImg);
        }
    }

    /**
     * Opens the initialized lightbox.
     */
    open() {
        this.lightboxInstance?.open();
    }

    /**
     * Handles and displays errors during the initialization process.
     * @param {Error} error - The error object.
     */
    handleInitializationError(error) {
        const errorMessage = `Error loading PDF: ${error.message}`;
        this.triggerElement.innerHTML = `<p class="text-red-600">${errorMessage}</p>`;
        console.error("Initialization Failed:", error);
    }

    /**
     * Creates a unique ID based on current time.
     * @returns {string} The unique ID.
     */
    createTimeBasedId() {
        const timestamp = Date.now().toString(36);
        const randomPart = Math.random().toString(36).substring(2, 7);
        return `${timestamp}-${randomPart}`;
    }
}
window.PdfToLightBoxConverter = PdfToLightBoxConverter;
