<?php

declare(strict_types = 1);

namespace Internetgalerie\IgFormlog\Controller\Backend;

/*
 * This file is part of the Pagemachine TYPO3 Formlog project.
 */

use Internetgalerie\IgFormlog\Domain\FormLog\Filters;
use Internetgalerie\IgFormlog\Domain\Repository\FormLogEntryRepository;
use Internetgalerie\IgFormlog\Export\CsvExport;
use Internetgalerie\IgFormlog\Export\XlsxExport;
use Psr\Http\Message\ResponseInterface;
use TYPO3\CMS\Backend\Routing\UriBuilder;
use TYPO3\CMS\Backend\Template\ModuleTemplateFactory;
use TYPO3\CMS\Core\Information\Typo3Version;
use TYPO3\CMS\Core\Page\PageRenderer;
use TYPO3\CMS\Core\Pagination\SimplePagination;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Pagination\QueryResultPaginator;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

use TYPO3\CMS\Form\Domain\Model\FormDefinition;
use TYPO3\CMS\Form\Domain\Model\FormElements\AbstractFormElement;
use TYPO3\CMS\Form\Domain\Factory\ArrayFormFactory;

/**
 * Controller for form log management
 */
class FormLogController extends ActionController
{
    const EXCLUDE_FIELDS = ['Honeypot', 'StaticText', 'ContentElement', 'GridRow', 'SummaryPage', 'SrFreecap', 'Recaptcha'];
 
    /**
     * @var \TYPO3\CMS\Form\Mvc\Persistence\FormPersistenceManagerInterface
     */
    protected $formPersistenceManager;
 
    /**
     * @param \TYPO3\CMS\Form\Mvc\Persistence\FormPersistenceManagerInterface $formPersistenceManager
     * @internal
     */
    public function injectFormPersistenceManager(\TYPO3\CMS\Form\Mvc\Persistence\FormPersistenceManagerInterface $formPersistenceManager)
    {
        $this->formPersistenceManager = $formPersistenceManager;
    }
    
    protected array $viewFormatToExportMap = [
        'csv' => CsvExport::class,
        'xlsx' => XlsxExport::class,
    ];

    protected FormLogEntryRepository $formLogEntryRepository;

    public function __construct(protected ModuleTemplateFactory $moduleTemplateFactory)
    {
    }

    public function injectFormLogEntryRepository(FormLogEntryRepository $formLogEntryRepository)
    {
        $this->formLogEntryRepository = $formLogEntryRepository;
    }

    /**
     * Initialize all actions
     *
     * @return void
     */
    public function initializeAction(): void
    {
        if ($this->arguments->hasArgument('filters')) {
            $filters = $this->request->hasArgument('filters') ? $this->request->getArgument('filters') : [];

            if ((new Typo3Version())->getMajorVersion() < 11) {
                $this->request->setArgument('filters', $filters);
            } else {
                $this->request = $this->request->withArgument('filters', $filters);
            }

            $filtersArgument = $this->arguments->getArgument('filters');
            $filtersArgument->getPropertyMappingConfiguration()
                ->allowAllProperties()
                ->forProperty('*')
                    ->allowAllProperties();
        }
    }

    /**
     * Main overview action
     */
    public function indexAction(Filters $filters, int $currentPageNumber = 1): ResponseInterface
    {
        $entries = $this->formLogEntryRepository->findAllFiltered($filters);
        $paginator = new QueryResultPaginator($entries, $currentPageNumber);
        /** @var UriBuilder */
        $uriBuilder = GeneralUtility::makeInstance(UriBuilder::class);

        $this->view->assignMultiple([
            'entries' => $paginator->getPaginatedItems(),
            'entriesCount' => count($entries),
            'filters' => $filters,
            'pagination' => new SimplePagination($paginator),
            'currentPageNumber' => $currentPageNumber,
            'dateFormat' => $this->settings['dateTimeFormat'] ?: \DateTime::W3C,
            'isoDateFormat' => \DateTime::W3C,
            'daterangepickerTranslations' => $this->prepareDaterangepickerTranslations(),
            'inlineSettings' => [
                'formlog' => [
                    'suggestUri' => (string)$uriBuilder->buildUriFromRoute('ajax_ig_formlog_suggest'),
                    'language' => $GLOBALS['BE_USER']->user['lang'],
                ],
            ],
        ]);

        GeneralUtility::makeInstance(PageRenderer::class)->addRequireJsConfiguration([
            'paths' => [
                'TYPO3/CMS/IgFormlog/moment' => 'https://cdnjs.cloudflare.com/ajax/libs/moment.js/2.29.4/moment-with-locales.min',
            ],
        ]);

        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $moduleTemplate->setContent($this->view->render());

        return $this->htmlResponse($moduleTemplate->renderContent());
    }

    /**
     * Export CSV of form log entries
     */
    public function exportAction(Filters $filters): ResponseInterface
    {
        $export = GeneralUtility::makeInstance($this->viewFormatToExportMap[$this->request->getFormat()]);

        $now = new \DateTime();
        $fileBasename = sprintf('formlog-%s', $now->format('Y-m-d-H-i-s'));

        $logEntries = $this->formLogEntryRepository->findAllFiltered($filters);

        /* IG ab hier */
        $columns = $this->settings['export']['columns'];

        if($filters->getIdentifier()) {
            $formIdentifier = $filters->getIdentifier()->getValue();
            $formIdentifier = preg_replace('/-\d+$/', '', $formIdentifier);// remove a hyphen followed by digits at the end of formIdentifier
            if($formIdentifier) {
                $formPersistenceIdentifier ='';
                foreach ($this->formPersistenceManager->listForms() as $formDefinition) {
                    if($formDefinition['identifier']==$formIdentifier) {
                        $formPersistenceIdentifier = $formDefinition['persistenceIdentifier'];
                    }
                }
                if($formPersistenceIdentifier) {
                    //$formPersistenceIdentifier = 'Materialbestellung';
                    $formDefinition = $this->getFormDefinitionObject($formPersistenceIdentifier, true);
                    $columns = $this->getColumns($formDefinition);
                    //var_dump($columns);exit(0);
                }
            }
        }
        $export->setConfiguration([
            'columns' => $columns,
            'dateTimeFormat' => $this->settings['dateTimeFormat'],
            'fileBasename' => $fileBasename,
        ]);

        $export->dump($logEntries);

        return $this->responseFactory->createResponse();
    }

    /**
     * Prepare localized daterangepicker labels
     *
     * @return array
     */
    protected function prepareDaterangepickerTranslations(): array
    {
        $translationIdentifiers = [
            'labels' => [
                'applyLabel',
                'cancelLabel',
                'fromLabel',
                'toLabel',
                'customRangeLabel',
            ],
            'ranges' => [
                'last30days',
                'lastYear',
            ],
        ];
        $translations = [];

        foreach ($translationIdentifiers as $section => $identifiers) {
            foreach ($identifiers as $identifier) {
                $translations[$section][$identifier] = LocalizationUtility::translate('formlog.daterangepicker.' . $section . '.' . $identifier, 'Formlog');
            }
        }

        return $translations;
    }

    /**
     * Gets a form definition by a persistence form identifier
     */
    protected function getFormDefinitionObject(
        string $formPersistenceIdentifier,
        bool $useFieldStateDataAsRenderables = false
    ): FormDefinition {
        $configuration = $this->formPersistenceManager->load($formPersistenceIdentifier);
        //$configuration = $this->getFormDefinition($formPersistenceIdentifier, $useFieldStateDataAsRenderables);
        /** @var ArrayFormFactory $arrayFormFactory */
        $arrayFormFactory = GeneralUtility::makeInstance(ArrayFormFactory::class);
        return $arrayFormFactory->build($configuration);
    }
 
 
    /**
     * Gets an array of all form renderables (recursive) by a form definition
     */
    protected function getColumns(FormDefinition $formDefinition): array
    {

        $columns = [];
        /** @var AbstractFormElement $renderable */
        foreach ($formDefinition->getRenderablesRecursively() as $renderable) {
            //echo($renderable->getType() .'= ' . $renderable->getLabel(). '<br />');
            if ($renderable instanceof AbstractFormElement && in_array($renderable->getType(), self::EXCLUDE_FIELDS, true) === false) {
                $columns[] = [
                    'property' => 'data.' . $renderable->getIdentifier(),
                    'label' => $renderable->getLabel(),
                ];
            }
        }
        return $columns;
    }


}
