<?php

declare(strict_types=1);

namespace Internetgalerie\IgFrontendUser\Controller;


use GeorgRinger\NumberedPagination\NumberedPagination;
use TYPO3\CMS\Core\Pagination\PaginatorInterface;
use TYPO3\CMS\Core\Pagination\SimplePagination;
use TYPO3\CMS\Core\Pagination\SlidingWindowPagination;
use Internetgalerie\IgFrontendUser\Domain\Model\FrontendUser;
use Internetgalerie\IgFrontendUser\Domain\Repository\FrontendUserRepository;
use Internetgalerie\IgFrontendUser\Domain\Repository\FrontendUserGroupRepository;
use TYPO3\CMS\Extbase\Pagination\QueryResultPaginator;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Crypto\PasswordHashing\PasswordHashFactory;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Annotation\Validate as ValidateAnnotation;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use Psr\Http\Message\ResponseInterface;

/**
 * FrontendUserController
 */
class FrontendUserController extends ActionController
{

    /**
     * frontendUserRepository
     *
     * @var FrontendUserRepository
     */
    protected $frontendUserRepository = null;

    /**
     * frontendUserGroupRepository
     *
     * @var FrontendUserGroupRepository
     */
    protected $frontendUserGroupRepository = null;

    /**
     * @param FrontendUserRepository $frontendUserRepository
     */
    public function injectFrontendUserRepository(FrontendUserRepository $frontendUserRepository)
    {
        $this->frontendUserRepository = $frontendUserRepository;
    }

    /**
     * @param FrontendUserGroupRepository $frontendUserGroupRepository
     */
    public function injectFrontendUserGroupRepository(FrontendUserGroupRepository $frontendUserGroupRepository)
    {
        $this->frontendUserGroupRepository = $frontendUserGroupRepository;
    }

    /**
     * action show
     */
    public function showAction(FrontendUser $frontendUser): ResponseInterface
    {
        $this->view->assign('frontendUser', $frontendUser);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action list
     */
    public function listAction(array $search = []): ResponseInterface
    {
        $adminFrontendUserGroupUid = (int) $this->settings['adminFrontendUserGroupUid'];

        $frontendUserId = GeneralUtility::makeInstance(Context::class)->getPropertyFromAspect('frontend.user', 'id');
        $this->view->assign('loggedInFrontendUserUid', $frontendUserId);
        //$this->view->assign('loggedInFrontendUser', $GLOBALS['TSFE']->fe_user->user);
        
        //if ($adminFrontendUserGroupUid)
        $frontendUsers = $this->frontendUserRepository->findBySearch($search, $this->settings);
        $this->view->assign('frontendUsers', $frontendUsers);
        $currentPage = $this->request->hasArgument('currentPage') ? (int)$this->request->getArgument('currentPage') : 1;

        $paginator = new QueryResultPaginator($frontendUsers, $currentPage, 20);
        $pagination  = $this->createPagination($paginator,10); // zweiter Argument: maximal Anzahl Links
        $fegroups = [];
        $searchFrontendUserGroupUid = (int) $this->settings['searchFrontendUserGroupUid'] ?? 0;
        if ($searchFrontendUserGroupUid) {
            $fegroups = [
                $searchFrontendUserGroupUid => $this->settings['searchLabelHasGroup'],
                -$searchFrontendUserGroupUid => $this->settings['searchLabelHasNotGroup'],
            ];
        }
        // @todo add all fegroups and change to multiselect
        $this->view->assign('fegroups', $fegroups);
        $this->view->assign('search', $search);
        $this->view->assign('pagination', [
            'paginator' => $paginator,
            'pagination' => $pagination,
        ]);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action new
     */
    public function newAction(array $search = []): ResponseInterface
    {
        $frontendUserGroups = $this->frontendUserGroupRepository->findAll();
        if($this->settings['forceFrontendUserGroupUid']) {
            $this->view->assign('forceFrontendUserGroupUid', (int)$this->settings['forceFrontendUserGroupUid']);
        }

        $this->view->assign('search', $search);
        $this->view->assign('frontendUserGroups', $frontendUserGroups);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action create
     *
     * @param FrontendUser $newFrontendUser
     * @param array $search
     * @param array $password
     * @ValidateAnnotation(param="newFrontendUser", validator="Internetgalerie\IgFrontendUser\Domain\Validator\FrontendUserRegistrationValidator")
     * @ValidateAnnotation(param="password", validator="Internetgalerie\IgFrontendUser\Domain\Validator\PasswordValidator")
     * @ValidateAnnotation(param="password", validator="Internetgalerie\IgFrontendUser\Domain\Validator\PasswordRepeatValidator")
     * @ValidateAnnotation(param="password", validator="Internetgalerie\IgFrontendUser\Domain\Validator\PasswordNotEmptyValidator")
     */
    public function createAction(FrontendUser $newFrontendUser, array $password = [], array $search = []): ResponseInterface
    {
        // Hash the password and reset it before adding to the database
        $hashInstance = GeneralUtility::makeInstance(PasswordHashFactory::class)->getDefaultHashInstance('FE');
        $hashedPassword = $hashInstance->getHashedPassword($password['0']);
        $newFrontendUser->setPassword($hashedPassword);

        $this->frontendUserRepository->add($newFrontendUser);
        return $this->redirect('list', null, null, ['search' => $search]);
    }

    /**
     * action edit
     *
     * @param FrontendUser $frontendUser
     * @param array $password
     * @param array $search
     * @TYPO3\CMS\Extbase\Annotation\IgnoreValidation("frontendUser")
     */
    public function editAction(FrontendUser $frontendUser, array $password = [], array $search = []): ResponseInterface
    {
        $frontendUserGroups = $this->frontendUserGroupRepository->findAll();

        if($this->settings['forceFrontendUserGroupUid']) {
            $this->view->assign('forceFrontendUserGroupUid', (int)$this->settings['forceFrontendUserGroupUid']);
        }

        $this->view->assign('frontendUser', $frontendUser);
        $this->view->assign('frontendUserGroups', $frontendUserGroups);
        $this->view->assign('search', $search);
        $this->view->assign('password', $password);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action update
     *
     * @param FrontendUser $frontendUser
     * @param array $password
     * @param array $search
     * @ValidateAnnotation(param="frontendUser", validator="Internetgalerie\IgFrontendUser\Domain\Validator\FrontendUserRegistrationValidator")
     * @ValidateAnnotation(param="password", validator="Internetgalerie\IgFrontendUser\Domain\Validator\PasswordValidator")
     * @ValidateAnnotation(param="password", validator="Internetgalerie\IgFrontendUser\Domain\Validator\PasswordRepeatValidator")
     */
    public function updateAction(FrontendUser $frontendUser, array $password = [], array $search = []): ResponseInterface
    {
         if (isset($password[0]) && $password[0] !== null && $password[0] !== '') {
             // Hash the password and reset it before adding to the database
             $hashInstance = GeneralUtility::makeInstance(PasswordHashFactory::class)->getDefaultHashInstance('FE');
             $hashedPassword = $hashInstance->getHashedPassword($password[0]);
             $frontendUser->setPassword($hashedPassword);
         }
         $this->frontendUserRepository->update($frontendUser);
         return $this->redirect('list', null, null, ['search' => $search]);
    }

    /**
     * action confirmDelete
     *
     * @param FrontendUser $frontendUser
     * @param array $search
     */
    public function confirmDeleteAction(FrontendUser $frontendUser, array $search = []): ResponseInterface
    {
        $this->view->assign('frontendUser', $frontendUser);
        $this->view->assign('search', $search);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action delete
     *
     * @param FrontendUser $frontendUser
     * @param array $search
     */
    public function deleteAction(FrontendUser $frontendUser, array $search = []): ResponseInterface
    {
        $this->frontendUserRepository->remove($frontendUser);
        return $this->redirect('list', null, null, ['search' => $search]);
    }

    /**
     * action editMyPasswordAction
     */
    public function editMyPasswordAction(): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $frontendUserUid = $context->getPropertyFromAspect('frontend.user', 'id');

        // no user is logged in
        if(!$frontendUserUid) {
        return $this->htmlResponse('');
        }

        $frontendUser = $this->frontendUserRepository->findByUid($frontendUserUid);

        $this->view->assign('password', $password);
        $this->view->assign('frontendUser', $frontendUser);
        return $this->htmlResponse($this->view->render());
    }

    /**
     * action updateMyPasswordAction
     *
     * @param FrontendUser $frontendUser
     * @param array $password
     * @param array $search
     * @ValidateAnnotation(param="frontendUser", validator="Internetgalerie\IgFrontendUser\Domain\Validator\OldPasswordValidator")
     * @ValidateAnnotation(param="password", validator="Internetgalerie\IgFrontendUser\Domain\Validator\PasswordValidator")
     * @ValidateAnnotation(param="password", validator="Internetgalerie\IgFrontendUser\Domain\Validator\PasswordRepeatValidator")
     */
    public function updateMyPasswordAction(FrontendUser $frontendUser, array $password = [], array $search = []): ResponseInterface
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $loggedInFrontendUserUid = $context->getPropertyFromAspect('frontend.user', 'id');

        // Redirect away from if the user to change is not the user that is logged in
        if($loggedInFrontendUserUid != $frontendUser->getUid()) {
            return $this->redirect('editMyPassword');
        }

        // Hash the password and reset it before adding to the database
        $hashInstance = GeneralUtility::makeInstance(PasswordHashFactory::class)->getDefaultHashInstance('FE');
        $hashedPassword = $hashInstance->getHashedPassword($password['0']);
        $frontendUser->setPassword($hashedPassword);

        $this->frontendUserRepository->update($frontendUser);

        return $this->redirect('list');
    }

    protected function createPagination(PaginatorInterface $paginator, int $maximumNumberOfLinks = 0)
    {
        if (class_exists(SlidingWindowPagination::class)) {
           return new SlidingWindowPagination($paginator, $maximumNumberOfLinks);
       }
        if (class_exists(NumberedPagination::class)) {
           return new NumberedPagination($paginator, $maximumNumberOfLinks);       
       }
       return new SimplePagination($paginator);
    }
}
