<?php

declare(strict_types=1);

namespace Internetgalerie\IgFrontendUser\Domain\Repository;

use Internetgalerie\IgFrontendUser\Event\AddFrontendUserConstraintsEvent;
use Psr\EventDispatcher\EventDispatcherInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Query\Restriction\DeletedRestriction;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Persistence\Generic\Typo3QuerySettings;
use TYPO3\CMS\Extbase\Persistence\Repository;

/**
 * A Frontend User repository
 */
class FrontendUserRepository extends Repository
{
    protected ConfigurationManager $configurationManager;
    protected EventDispatcherInterface $eventDispatcher;
    protected static ?array $cachedCurrentFrontendUser = null;

    public function setStoragePageId(int $storagePageId): void
    {
        $this->initDefaultQuerySettings();
        $this->defaultQuerySettings->setStoragePageIds([$storagePageId]);
    }
    public function setRespectStoragePage(bool $respectStoragePage): void
    {
        $this->initDefaultQuerySettings();
        $this->defaultQuerySettings->setRespectStoragePage($respectStoragePage);
    }
    protected function initDefaultQuerySettings()
    {
        if ($this->defaultQuerySettings === null) {
            $this->defaultQuerySettings = GeneralUtility::makeInstance(Typo3QuerySettings::class);
        }
    }

    public function injectEventDispatcher(EventDispatcherInterface $eventDispatcher): void
    {
        $this->eventDispatcher = $eventDispatcher;
    }
    
    public function injectConfigurationManager(ConfigurationManager $configurationManager): void
    {
        $this->configurationManager = $configurationManager;
        $querySettings = GeneralUtility::makeInstance(Typo3QuerySettings::class);
        $storagePid = (int) ($this->configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT)['plugin.']['tx_igfrontenduser.']['persistence.']['storagePid'] ?? 0);
        if ($storagePid) {
            $querySettings->setStoragePageIds([$storagePid]);
        } else {
            $querySettings->setRespectStoragePage(false);
        }
        $this->setDefaultQuerySettings($querySettings);
 
    }
    public function findAllFromStoragePid()
    {
        $query = $this->createQuery();
        $this->configurationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
        $storagePid = (int) ($this->configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT)['plugin.']['tx_igfrontenduser.']['persistence.']['storagePid'] ?? 0);
        $query->getQuerySettings()->setStoragePageIds([$storagePid])->setRespectStoragePage(true);
        return $query->execute();
    }
          
    public function findWithUsergroups(array $usergroupUids, bool $returnRawQueryResult = false)
    {
        $query = $this->createQuery();

        $constraints = [];
        foreach ($usergroupUids as $usergroupUid) {
            // FIND_IN_SET
            $constraints[] = $query->logicalOr(
                $query->equals('usergroup', (int)$usergroupUid),
                $query->like('usergroup', (int)$usergroupUid . ',%'),
                $query->like('usergroup', '%,' . (int)$usergroupUid),
                $query->like('usergroup', '%,' . (int)$usergroupUid . ',%'),
            );
        }
        if (!empty($constraints)) {
            $query->matching(
                $query->logicalAnd(...$constraints)
            );
        }
        return $query->execute($returnRawQueryResult);

    }

    public function findBySearch(array $search, array $settings = [], bool $returnRawQueryResult = false) {
        $query = $this->createQuery();

        $constraints = [];
        $constraints  = $this->eventDispatcher->dispatch(
            new AddFrontendUserConstraintsEvent($query, $search, $settings)
        )->getConstraints();
        // Andere Sucher
        //$queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable('fe_users');
        //$queryBuilder->expr()->inSet('usergroup', '12');//$query->equals('usergroup', $fegroup);
        $constraints  = $this->eventDispatcher->dispatch(
            new AddFrontendUserConstraintsEvent($query, $search, $settings)
        )->getConstraints();

        if($settings['forceFrontendUserGroupUid']) {
            $fegroup = (int)$settings['forceFrontendUserGroupUid'];
            $feGroupConstrain = $query->logicalOr(
                $query->equals('usergroup', (int)$fegroup),
                $query->like('usergroup', (int)$fegroup . ',%'),
                $query->like('usergroup', '%,' . (int)$fegroup),
                $query->like('usergroup', '%,' . (int)$fegroup . ',%'),
            );
            $constraints[] = $feGroupConstrain;
        }

        if (isset($search['fegroup']) && $search['fegroup'] !== null && $search['fegroup'] !== '') {
            // @todo find_in_set + explode
            $fegroup = (int) $search['fegroup'];
            $feGroupHasNot = $fegroup < 0;
            $fegroup = abs($fegroup);
            // fegroup
            $feGroupConstrain = $query->logicalOr(
                $query->equals('usergroup', (int)$fegroup),
                $query->like('usergroup', (int)$fegroup . ',%'),
                $query->like('usergroup', '%,' . (int)$fegroup),
                $query->like('usergroup', '%,' . (int)$fegroup . ',%'),
            );
            if ($feGroupHasNot) {
                $constraints[] =  $query->logicalNot($feGroupConstrain);
            } else {
                $constraints[] = $feGroupConstrain;
            }
        }

        $keywordsArray = GeneralUtility::trimExplode(' ', $search['keywords'] ?? '', true);
        if (!empty($keywordsArray)) {
            foreach ($keywordsArray as $keyword) {
                $constraints[] = $query->logicalOr(
                    $query->like('username', '%' . $keyword . '%'),
                    $query->like('first_name', '%' . $keyword . '%'),
                    $query->like('last_name', '%' . $keyword . '%')
                );
            }
        }
        if (!empty($constraints)) {
            $query->matching(
                $query->logicalAnd(...$constraints)
            );
        }
        return $query->execute($returnRawQueryResult);
    }

    public static function findRawByUid(int $uid): array
    {
        if ($uid == 0) {
            return [];
        }
        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable('fe_users');
        $queryBuilder->getRestrictions()->removeAll()->add(GeneralUtility::makeInstance(DeletedRestriction::class));
        return $queryBuilder
            ->select('*')
            ->from('fe_users')
            ->where($queryBuilder->expr()->eq('uid', $queryBuilder->createNamedParameter($uid, Connection::PARAM_INT)))
            ->executeQuery()
            ->fetchAssociative() ?: [];
    }
    public static function getCurrentFrontendUserRaw(): array
    {
        if (self::$cachedCurrentFrontendUser === null) {
            $context = GeneralUtility::makeInstance(Context::class);
            $userId = (int) $context->getPropertyFromAspect('frontend.user', 'id');
            self::$cachedCurrentFrontendUser = self::findRawByUid($userId);
        }
        return self::$cachedCurrentFrontendUser;
    }

}
